/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GTooltip.h"
#include "glib/gui/GToolbarButton.h"
#include "glib/gui/GFrameWindow.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GAbstractToolbarWindow.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/sys/GSystem.h"
#include "glib/GProgram.h"

GTooltip GTooltip::TheInstance;
const GColor GTooltip::DefBckColor(255, 255, 180);
const GColor GTooltip::DefFrgColor(GColor::BLACK);
int GTooltip::DefAutoHideMillis(6000);
int GTooltip::DefDelayShowMillis(750);

const GString GTooltip::DefaultFont = "9.WarpSans";

class TooltipWindow : public GWindow
{
public:
   int delayShowMillis;
   int autoHideMillis;

   TooltipWindow ( GWindow* parentWin, GWindow* ownerWin )
         :GWindow("Tooltip", GString::Empty, parentWin, ownerWin, 0, WS2_OS2Y | WS2_AUTO_PAINT_TEXT | WS2_AUTO_PAINT_BACKGROUND),
          delayShowMillis(0),                    
          autoHideMillis(0)
   {
      setFontNameSize(defaultFontNameSize = GTooltip::DefaultFont);
      setBackgroundColor(defaultBackgroundColor = GTooltip::DefBckColor);
      setForegroundColor(defaultForegroundColor = GColor::BLACK);
      setLayoutManager(new GBorderLayout(), true);
      GLineBorder* border = new GLineBorder();
      border->setLineColor(GColor::BLACK);
      setBorder(border, true);
      setInsets(new GInsets(1, 1, 1, 1), true);
   }

   virtual ~TooltipWindow ()
   {
   }

   void showNow ()
   {
      if (GTooltip::TheInstance.currentOwner == null)
         return;
      if (!GTooltip::TheInstance.currentOwner->isMouseCapture())
         return;
      setVisible(true);
      invalidateAll(true);
      if (autoHideMillis > 0)
         startTimer("AutoHideTooltip", autoHideMillis);
   }

   virtual bool onTimer ( const GString& timerID, GObject* userData )
   {
      if (timerID == "DelayShowTooltip")
      {
         stopTimer("DelayShowTooltip");
         showNow();
         return true;
      }
      else
      if (timerID == "AutoHideTooltip")
      {
         stopTimer("AutoHideTooltip");
         GTooltip::Hide();
         return true;
      }
      else
      {
         return GWindow::onTimer(timerID, userData);
      }
   }

private:
   // Disable the copy constructor and assignment operator.
   TooltipWindow ( const TooltipWindow& );
   void operator= ( const TooltipWindow& );
};

GTooltip::GTooltip ()
         :currentOwner(null),
          win(null)
{
}

GTooltip::~GTooltip ()
{
   if (win != null)
      delete win;
}

void GTooltip::createTheWindow ( GWindow& ownerWin )
{
   if (win != null)
      return; // Window already created!
   win = new TooltipWindow(null, &ownerWin);
}

GWindow* GTooltip::GetCurrentOwner ()
{
   return TheInstance.currentOwner;
}

void GTooltip::Show ( GWindow& forWin, const GString& text, int delayShowMillis, int autoHideMillis )
{
   if (!forWin.isShowTooltip())
      return;

   // Use the specified window for tooltip positioning if its parent 
   // window is not some type of a toolbar window. 
   GWindow* posWin = forWin.getParentWindow();
   if (posWin == null || dynamic_cast<GAbstractToolbarWindow*>(posWin) == null)
      posWin = &forWin;

   if (TheInstance.win == null)
      TheInstance.createTheWindow(*posWin);

   GString text_ = text;
   if (text == "")
   {
      // If there is no tooltip text to show, and the tooltip window 
      // is not already visible, then deny showing the tooltip.
      if (!TheInstance.win->isVisible())
         return;
      // Ok, so just show a dummy text instead of a blank.
      text_ = "---";
   }

   TheInstance.win->setVisible(false);
   TheInstance.win->setText(text_); // Will also load the text if it is loadable.
   TheInstance.currentOwner = &forWin;

   // Get the loaded text (in case it was loadable).
   text_ = TheInstance.win->getText();

   int width = TheInstance.win->getWidthOfString(text_) + 15;
   int height = TheInstance.win->getHeightOfString(text_) + 4;
   int forWinX = forWin.getWindowPosOnScreen().x;
   int posY = posWin->getWindowPosOnScreen().y;
   int posH = posWin->getWindowRect().height;

   int ypos;
   switch (forWin.getTooltipPosition())
   {
      case GTooltip::PosOverlap:
         ypos = posY;
         break;

      case GTooltip::PosAbove:
         ypos = posY + posH;
         if (ypos >= GSystem::GetScreenSize().height)
            ypos = posY - height - 2;
         if (posWin == &forWin)
            ypos += 1;
         break;

      case GTooltip::PosBelow:
      case GTooltip::PosInheritFromParent:
      default:
         ypos = posY - height - 2;
         if (posWin == &forWin)
            ypos -= 1;
         if (ypos < 0)
            ypos = posY + posH;
         break;
   }

   TheInstance.win->setWindowSize(width, height);
   TheInstance.win->setWindowPosOnScreen(forWinX, ypos, true);

   if (TheInstance.win->isTimerStarted("DelayShowTooltip"))
      TheInstance.win->stopTimer("DelayShowTooltip");
   if (TheInstance.win->isTimerStarted("AutoHideTooltip"))
      TheInstance.win->stopTimer("AutoHideTooltip");

   TheInstance.win->autoHideMillis = (autoHideMillis >= 0 ? autoHideMillis : GTooltip::DefAutoHideMillis);
   TheInstance.win->delayShowMillis = (delayShowMillis >= 0 ? delayShowMillis : GTooltip::DefDelayShowMillis);
   if (TheInstance.win->delayShowMillis > 0)
      TheInstance.win->startTimer("DelayShowTooltip", TheInstance.win->delayShowMillis);
   else
      TheInstance.win->showNow();
}

void GTooltip::Hide ()
{
   if (TheInstance.win == null)
      return; // Already hidden.

   GWindow* co = TheInstance.currentOwner;
   if (co != null)
   {
      co->captureMouse(false); // Release the mouse capture
      co->updateHintString(GString::Empty);
      co->invalidateAll(false);
   }

   TheInstance.currentOwner = null;
   TheInstance.win->setVisible(false);
}

void GTooltip::ReDisplay ()
{
   GWindow* co = TheInstance.currentOwner;
   if (TheInstance.win != null && co != null)
   {
      int autoHideMillis = TheInstance.win->autoHideMillis;
      int delayShowMillis = TheInstance.win->delayShowMillis;
      if (TheInstance.win->isVisible())
         delayShowMillis = 0; // Re-display immediately!

      GString text = co->getTooltipText();
      GTooltip::Show(*co, text, delayShowMillis, autoHideMillis);

      // Update the hint-cell of the statusbar as well.
      GFrameWindow* pwin = dynamic_cast<GFrameWindow*>(&co->getTopLevelWindow());
      if (pwin != null)
      {
         GToolbarButton* tbb = dynamic_cast<GToolbarButton*>(co);
         GString cmdID = tbb == null ? co->getName() : tbb->getIDString();
         if (cmdID == "")
            pwin->setStatusbarHint("---");
         else
            pwin->updateHintString(cmdID);
      }
   }
}

void GTooltip::SetDefaultAutoHideMillis ( int value )
{
   DefAutoHideMillis = (value >= 0 ? value : 6000);
}

void GTooltip::SetDefaultDelayShowMillis ( int value )
{
   DefDelayShowMillis = (value >= 0 ? value : 750);
}
