/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TOOLBARBUTTON
#define __GLIB_TOOLBARBUTTON

#include "glib/gui/GToolbarElement.h"
#include "glib/gui/event/GAbstractCommand.h"

/**
 * Class to represent a single button on a toolbar window.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GToolbarButton : public GToolbarElement
{
   public:

      class Command : public GAbstractCommand
      {
         friend class GToolbarButton;
      private:
         Command ( const GString& id = GString::Empty, 
                   const GString& userData1 = GString::Empty );
         virtual ~Command ();
      };

      /** Flags used to identify the icon position with respect to the button text, if any. */
      enum ICONPOS
      {
         /** Center icon and don't draw the text even if the text is not empty. */
         IP_CENTER,

         /** Draw the icon below the text. */
         IP_BOTTOM,

         /** Draw the icon above the text. */
         IP_TOP,

         /** Draw the icon to the left of the text. */
         IP_LEFT,

         /** Draw the icon to the right of the text. */
         IP_RIGHT
      };

   private:

      /** Where the icon is to be positioned with respect to the button text, if any. */
      ICONPOS iconPos;

      /** True if button is currently "down". */
      bool isDown;

      /** True if mouse button is down. */
      bool mouseIsDown;

      /** The location of the most recent WM_MOUSEMOVE message. */
      GPoint recentMousePos;

      /** The object used to represent the "abstract command" of the tooobar button. */
      Command command;

   public:

      /** The ID of which icon resource to show on this button. */
      GString iconName;

   public:

      GToolbarButton ( const GString& name,
                       const GString& constraints,
                       GWindow& parentWin,
                       const GString& id,
                       const GString& text = GString::Empty,
                       const GString& iconName = GString::Empty,
                       const GString& userData1 = GString::Empty,
                       ICONPOS iconPos = IP_CENTER,
                       long winStyle = WS_VISIBLE );

      virtual ~GToolbarButton ();

   private:

      /** Disable the copy constructor. */
      GToolbarButton ( const GToolbarButton& src ) 
         : GToolbarElement(GString::Empty, GString::Empty, *this) {}

      /** Disable the assignment operator. */
      GToolbarButton& operator= ( const GToolbarButton& ) { return *this; }

   protected:

      bool onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      bool onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      bool onPaint ( GGraphics& g, const GRectangle& rect );

      /**
       * Clear the button face, using the same color as of out parent
       * toolbar window.
       */
      bool onPaintBackground ( GGraphics& g, const GRectangle& rect );

   public:

      void paintButton ( bool inclFace = true );
      void paintButton ( GGraphics& g, bool inclFace = true );

      /**
       * Draw a 3D-border at specified rectangle positions. If isDown is
       * true, then we will paint the border so it looks like it is
       * pressed down.
       *
       * @param rect  Entry: Rectangle area of where to draw border (inside).
       *              Return: The rectangle area within the painted border.
       * @param force True if we shall force the border to be painted even if
       *              the toolbar has been set up to use flat buttons.
       */
      void paintButtonBorder ( const GRectangle& rect, bool force = false );

      /**
       * Draw a 3D-border at specified rectangle positions. If isDown is
       * true, then we will paint the border so it looks like it is
       * pressed down.
       *
       * @param g     Graphics object of where to paint the border.
       * @param rect  Entry: Rectangle area of where to draw border (inside).
       *              Return: The rectangle area within the painted border.
       * @param force True if we shall force the border to be painted even if
       *              the toolbar has been set up to use flat buttons.
       */
      void paintButtonBorder ( GGraphics& g, 
                               const GRectangle& rect, 
                               bool force = false );

      /**
       * @author  Leif Erik Larsen
       * @since   2000.09.08
       * @param   tokenizer The tokenizer of where to get tokens to parse.
       * @param   table     The resource table of where to lookup referenced 
       *                    resources such as e.g. icons and texts.
       * @throws  GRcException in case of any error.
       */
      virtual void loadFromScript ( GRcTokenizer& tokenizer, 
                                    GResourceTable* table );

   protected:

      virtual void paintButtonImpl ( GGraphics& g, bool inclFace );
      virtual void paintButtonBorderImpl ( GGraphics& g, const GRectangle& rect, bool force );
      virtual void paintButtonFaceImpl ( GGraphics& g, const GRectangle& rect );

      /**
       * Called automatically by {@link #paintButtonFaceImpl} if the text
       * if the button is too wide to fit within the specified maximum 
       * width of the button face area in pixels. 
       *
       * It is up to the implementation of this method to shorten the 
       * specified text (which is to be painted) some way in order for it
       * to fit within the specified width.
       *
       * The default implementation of this method will use
       * {@link GGraphics#getShortenedText} with its default dots-argument
       * to shorten the text.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.04
       */
      virtual void shortenVisibleText ( GGraphics& g,
                                        GString& txt, 
                                        int maxPixelWidth ) const;

      /**
       * Use this method to dismiss the mouse capture mode and hide the
       * tooltip window associated with a toolbar button (if any).
       *
       * If the mouse button is currently down under the control of the
       * default toolbar button implementation then this method will
       * do nothing but return. This is also the case if the mouse is
       * currently not captured by the default toolbar button implementation.
       *
       * @return True on success, or else false if the mouse capture could
       *         not be dismissed/released (probably because the mouse button
       *         is still pressed down by the user).
       */
      bool dismissMouseCapture ();

      /**
       * Return true if and only if the mouse button used to click the
       * toolbar button is currently pressed down under the control of
       * the default toolbar button implementation.
       */
      bool isMouseDown () const;

      /**
       * This method is called by {@link #onMouseMove} in order to let
       * the sub class (if any) override the handling of the specified
       * mouse position.
       *
       * The sub class can override this method and return true if and only
       * if the specified mouse position is to be handled exclusively by the
       * sub class implementation. This is not possible while the mouse
       * cursor is currently captured and the mouse button is currently down,
       * however. That is, this method will not be called if the mouse button
       * is down under the control of the default toolbar implementation.
       *
       * The default implementation of this method does nothing
       * but return false.
       */
      virtual bool isMousePosOverridden ( int xpos, int ypos ) const;

      /**
       * Return true if and only if the button should currently be
       * painted with a border. That is if <i>force</i> is true, the button
       * is currently "pressed down" or if the owner abstract toolbar says
       * that the buttons should be painted with borders (buttons are
       * not flat).
       *
       * This method is typically to be used by {@link #paintButtonBorderImpl}
       * only, to determine if the border is actually to be painted or not.
       */
      bool needToPaintBorder ( bool force = false ) const;

   public:

      /** Return true if the button is currently pressed "down". */
      bool isPressedDown () const;

      /** Get the command-id of our contained command. */
      virtual const GString& getIDString () const;

      /** Set the command-id of our contained command. */
      void setIDString ( const GString& newID );

      /** Get a reference to our contained command. */
      GAbstractCommand& getCommand ();

      /**
       * Get a pointer to the icon resource of which to be painted
       * on the toolbar button. If the button has no icon, then return null.
       * We will not delete the returned icon when we are finished with it,
       * so the implementing method keeps the owner responsibility.
       *
       * The default implementation of this method will return the icon
       * as of which was specified by name to the constructor.
       */
      virtual const GIcon* getIcon () const;

      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;
      virtual GString getKeyboardShortcutKeyText () const;

      /**
       * Get the human readable name of the keyboard shortcut key for the
       * specified command ID with respect to the currently focused window.
       * If there are no keyboard shortcut then return an empty string.
       */
      GString getKeyboardShortcutKeyTextForID ( const GString cmdID ) const;

      /**
       * This method is called by {@link GTooltip::Show} to get the 
       * concrete text of which to display in the tooltip of the button.
       *
       * @return The text of which to display in the tooltip for the
       *         button. If the button has no tooltip then just return
       *         an empty string.
       */
      virtual GString getTooltipText () const;

      /**
       * This method is automatically called when the user has requested the
       * associated command to be performed. For instance, if he has clicked
       * on the button with the mouse.
       */
      virtual void performAction ();

   private:

      /**
       * This function must be called upon every mouse movement when the
       * mouse is captured by the toolbar button. We will toggle the button
       * with respect to the specified mouse position.
       */
      void handleMouseMove ( int xpos, int ypos );

      /**
       * @author  Leif Erik Larsen
       * @since   2005.12.13
       */
      void initMouseCapture ( bool mouseIsDown );
};

#endif
