/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GToolbarButton.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GTooltip.h"
#include "glib/util/GMath.h"
#include "glib/GProgram.h"

GToolbarButton::Command::Command ( const GString& id, const GString& userData1 )
                        :GAbstractCommand(id, userData1)
{
}

GToolbarButton::Command::~Command ()
{
}

GToolbarButton::GToolbarButton ( const GString& name,
                                 const GString& constraints,
                                 GWindow& parentWin,
                                 const GString& id,
                                 const GString& text,
                                 const GString& iconName,
                                 const GString& userData1,
                                 ICONPOS iconPos,
                                 long winStyle )
               :GToolbarElement(name, constraints, parentWin, winStyle, WS2_OS2Y | WS2_NOT_STATIC | WS2_AUTO_SHOW_TOOLTIP),
                command(id, userData1),
                iconPos(iconPos),
                isDown(false),
                mouseIsDown(false),
                iconName(iconName)
{
   GString ltxt = GProgram::LoadText(text, GResourceTable::LT_PREFER_TEXT, 0);
   setText(ltxt);
}

GToolbarButton::~GToolbarButton ()
{
}

bool GToolbarButton::isPressedDown () const
{
   return isDown;
}

const GIcon* GToolbarButton::getIcon () const 
{ 
   bool enabled = isEnabled();
   return GIcon::GetIcon(iconName, !enabled);
}

const GString& GToolbarButton::getIDString () const 
{ 
   return command.getIDString(); 
}

void GToolbarButton::setIDString ( const GString& newID ) 
{ 
   command.setIDString(newID); 
}

GAbstractCommand& GToolbarButton::getCommand ()
{
   return command;
}

int GToolbarButton::getPreferredHeight () const
{
   int ret;
   const GIcon* icon = getIcon();
   if (icon != null)
   {
      ret = icon->getHeight();
      if (iconPos == IP_TOP || iconPos == IP_BOTTOM)
         if (getTextLength() > 0)
            ret += getHeightOfString("X");
   }
   else
   {
      ret = getHeightOfString("X");
   }
   ret += 8;
   return ret;
}

int GToolbarButton::getPreferredWidth () const
{
   int ret;
   GString txt = getText();
   const GIcon* icon = getIcon();
   if (icon != null)
   {
      ret = icon->getWidth();
      if (iconPos == IP_LEFT || iconPos == IP_RIGHT)
         if (txt != "")
            ret += getWidthOfString(txt) + 4;
   }
   else
   {
      ret = getWidthOfString(txt);
   }
   ret += 8;
   return ret;
}

GString GToolbarButton::getKeyboardShortcutKeyTextForID ( const GString cmdID ) const
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mainWin = prg.getMainWindow();
   const GWindow* focusWin = mainWin.getFocusedWindow();
   if (focusWin == null)
      focusWin = this;

   GString txt;
   GAccelItem *ai = focusWin->getAccelItemByCommandID(cmdID);
   if (ai != null)
      txt = ai->getKeyNameForHumans() + ": ";
   return txt;
}

GString GToolbarButton::getKeyboardShortcutKeyText () const
{
   GString id = getIDString();
   return getKeyboardShortcutKeyTextForID(id);
}

GString GToolbarButton::getTooltipText () const
{
   GString text = GWindow::getTooltipText();
   if (text != "")
      return text;

   text = getText();
   if (text == "")
   {
      GString id = getIDString();
      if (id == "")
         return id; // No tooltip is available for this button.
      text = "%" + id;
   }

   GString txt = getKeyboardShortcutKeyText();
   txt += GProgram::LoadText(text, GResourceTable::LT_PREFER_TEXT, 0);
   return txt;
}

void GToolbarButton::performAction ()
{
   const GString& cmdID = getIDString();
   if (cmdID != "") // Don't execute command if it has no ID!
      postCommand(&command);
}

bool GToolbarButton::onButton1Up ( int xpos, 
                                   int ypos, 
                                   const GWindowMessage::InputFlags& /*flags*/ )
{
   if (!isMouseCapture())
      return true;

   // Command should not be executed if mouse button was not captured by us
   // before the button was released.
   bool doTheCommand = mouseIsDown;

   mouseIsDown = false;
   captureMouse(false); // Release the mouse capture
   GTooltip::Hide();

   isDown = false;
   paintButton();

   // Request hint-text to be removed
   updateHintString("");

   if (doTheCommand)
      if (isPointInWindowRect(xpos, ypos))
         performAction();

   return true;
}

bool GToolbarButton::onButton1DblClk ( int xpos, 
                                       int ypos, 
                                       const GWindowMessage::InputFlags& flags )
{
   return onButton1Down(xpos, ypos, flags);
}

bool GToolbarButton::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   // Activate our owner window.
   GWindow& top = getTopLevelWindow();
   top.setActive();

   // Handle the mouse click only if button is not disabled.
   if (isEnabled())
   {
      initMouseCapture(true);
      handleMouseMove(xpos, ypos);
   }

   return true;
}

bool GToolbarButton::onMouseMove ( int xpos, 
                                   int ypos, 
                                   const GWindowMessage::InputFlags& /*flags*/ )
{
   if (xpos == recentMousePos.x && ypos == recentMousePos.y)
      return false;

   recentMousePos.x = xpos;
   recentMousePos.y = ypos;

   if (isMouseCapture())
   {
      if (isDown)
      {
         handleMouseMove(xpos, ypos);
      }
      else
      {
         bool overriddenPos = false;
         if (!mouseIsDown)
            overriddenPos = isMousePosOverridden(xpos, ypos);

         GPoint pt = getWindowPos();
         GDimension tbdim = getParentWindow()->getWindowSize();
         if (!overriddenPos && 
             isPointInWindowRect(xpos, ypos) && 
             pt.x + xpos < tbdim.width && 
             pt.y + ypos < tbdim.height)
         {
            handleMouseMove(xpos, ypos);
         }
         else
         if (!mouseIsDown)
         {
            dismissMouseCapture();
         }
      }
   }
   else
   {
      initMouseCapture(false);
   }

   // Return true if and only if you have actually set the mouse cursor.
   return false;
}

bool GToolbarButton::dismissMouseCapture ()
{
   if (mouseIsDown)
      return false; // Can't dismiss the capture at this time!

   if (!isMouseCapture())
      return true; // Nothing to dismiss, but this is not an error!

   // Hide the temporary border around the button underneath the mouse
   // cursor in case toolbar buttons are flat.
   if (isFlatButtonBorders())
   {
      GRectangle r = getWindowFaceRect();
      paintButtonBorder(r, false);
   }

   captureMouse(false); // Release the mouse capture
   updateHintString(GString::Empty);
   return true;
}

void GToolbarButton::handleMouseMove ( int xpos, int ypos )
{
   bool wasDown = isDown;
   isDown = mouseIsDown && isPointInWindowRect(xpos, ypos);
   if (isDown != wasDown)
      paintButton();
}

void GToolbarButton::initMouseCapture ( bool mouseIsDown )
{
   const GString& cmdID = getIDString();
   updateHintString(cmdID);

   if (isActive()) // If we are part of the a active (frame-) window.
   {
      // Paint a temporary border around the button underneath the mouse
      // cursor in case toolbar buttons are flat.
      if (isFlatButtonBorders())
      {
         GRectangle r = getWindowFaceRect();
         paintButtonBorder(r, true);
      }
   }

   this->mouseIsDown = mouseIsDown;
   captureMouse(true); // Will also show the tooltip box.
}

bool GToolbarButton::onPaint ( GGraphics& g, const GRectangle& /*rect*/ )
{
   paintButton(g, false); // Face has already been painted, upon "onPaintBackground()"
   return true;
}

bool GToolbarButton::onPaintBackground ( GGraphics& g, 
                                         const GRectangle& rect )
{
   GColor bck = getBackgroundColor();
   g.drawFilledRectangle(rect, bck);
   return true;
}

void GToolbarButton::paintButton ( bool inclFace )
{
   GGraphics g(*this);
   paintButton(g, inclFace);
}

void GToolbarButton::paintButton ( GGraphics& g, bool inclFace )
{
   paintButtonImpl(g, inclFace);
}

void GToolbarButton::paintButtonBorder ( const GRectangle& rect, bool force )
{
   GGraphics g(*this);
   paintButtonBorder(g, rect, force);
}

void GToolbarButton::paintButtonBorder ( GGraphics& g, 
                                         const GRectangle& rect, 
                                         bool force )
{
   paintButtonBorderImpl(g, rect, force);
}

void GToolbarButton::paintButtonImpl ( GGraphics& g, bool inclFace )
{
   GRectangle r = getWindowFaceRect();

   if (inclFace)
      onPaintBackground(g, r);

   bool forceBorder = false;
   if (GTooltip::GetCurrentOwner() == this)
      forceBorder = true;

   paintButtonBorder(g, r, forceBorder);

   r.inflateRect(-1);

   if (isDown)
   {
      // Displace the text and/or icon one pixel down and right
      r.x += 1;
      r.y -= 1;
   }

   paintButtonFaceImpl(g, r);
}

void GToolbarButton::paintButtonFaceImpl ( GGraphics& g, 
                                           const GRectangle& rect )
{
   GString txt = getText();
   GRectangle r(rect);

   const GIcon* icon = getIcon();
   if (icon != null)
   {
      // Calculate horizontal and vertical centered icon position.
      int xpos = r.x + (r.width/2) - (icon->getWidth()/2) - 1;
      int ypos = r.y + (r.height/2) - (icon->getHeight()/2) - 1;
      xpos = GMath::Max(0, xpos);
      ypos = GMath::Max(0, ypos);

      // Paint the icon at the correct position
      if (txt == "" || iconPos == IP_CENTER)
      {
         g.drawIcon(xpos, ypos, *icon);
         return;
      }
      else
      switch (iconPos)
      {
         case IP_BOTTOM:
              r.nudgeY(3);
              g.drawIcon(xpos, r.y, *icon);
              r.nudgeY(icon->getHeight() + 1);
              break;

         case IP_TOP:
              r.height -= icon->getHeight() + 3;
              g.drawIcon(xpos, r.y + r.height - 1, *icon);
              r.height -= 1;
              break;

         case IP_LEFT:
              r.nudgeX(5);
              g.drawIcon(r.x, ypos + 1, *icon);
              r.nudgeX(icon->getWidth() + 1);
              break;

         case IP_RIGHT:
              r.width -= icon->getWidth() + 5;
              g.drawIcon(r.x + r.width, ypos + 1, *icon);
              r.width -= 1;
              break;

         case IP_CENTER:
              break;
      }
   }

   // Paint the text horizontally and vertically centered on the button face
   if (txt != "")
   {
      // Make the text fit into visible area (if needed).
      if (g.getWidthOfString(txt) > r.width)
         shortenVisibleText(g, txt, r.width);

      // ---
      if (isEnabled())
      {
         const GColor& frg = getForegroundColor();
         g.drawText(txt, r, frg, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
      }
      else
      {
         g.drawText(txt, r, GColor::WHITE, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
         r.x -= 1;
         r.y += 1;
         g.drawText(txt, r, GColor::DGRAY, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
      }
   }
}

void GToolbarButton::shortenVisibleText ( GGraphics& g,
                                          GString& txt, 
                                          int maxPixelWidth ) const
{
   txt = g.getShortenedText(txt, maxPixelWidth);
}

bool GToolbarButton::needToPaintBorder ( bool force ) const
{
   return !isFlatButtonBorders() || isDown || force;
}

void GToolbarButton::paintButtonBorderImpl ( GGraphics& g, 
                                             const GRectangle& rect, 
                                             bool force )
{
   bool drawBorder = needToPaintBorder(force);
   const GColor backgroundColor = getBackgroundColor();
   const GColor& colorBlack  = (drawBorder ? GColor::BLACK : backgroundColor);
   const GColor& colorShadow = (drawBorder ? GColor::DGRAY : backgroundColor);
   const GColor& colorWhite  = (drawBorder ? GColor::WHITE : backgroundColor);

   GRectangle r(rect);

   g.setColor(isDown ? colorBlack : colorWhite);
   int x = r.x;
   int y = r.y;
   g.setPosition(x, y);
   y += r.height - 1;
   g.drawLineTo(x, y);
   x += r.width - 2;
   g.drawLineTo(x, y);
   g.setColor(isDown ? colorWhite : colorBlack);
   x += 1;
   g.setPosition(x, y);
   y = r.y;
   g.drawLineTo(x, y);
   x = r.x;
   g.drawLineTo(x, y);

   r.inflateRect(-1);

   g.setColor(isDown ? colorShadow : backgroundColor);
   x = r.x;
   y = r.y;
   g.setPosition(x, y);
   y += r.height - 1;
   g.drawLineTo(x, y);
   x += r.width - 2;
   g.drawLineTo(x, y);
   g.setColor(isDown ? backgroundColor : colorShadow);
   x += 1;
   g.setPosition(x, y);
   y = r.y;
   g.drawLineTo(x, y);
   x = r.y;
   g.drawLineTo(x, y);
}

bool GToolbarButton::isMouseDown () const 
{ 
   return mouseIsDown; 
}

bool GToolbarButton::isMousePosOverridden ( int xpos, int ypos ) const
{ 
   return false; 
}

void GToolbarButton::loadFromScript ( GRcTokenizer& tokenizer, 
                                      GResourceTable* table ) 
{
}
