/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TABBEDPANEL
#define __GLIB_TABBEDPANEL

#include "glib/gui/GStaticText.h"

/**
 * This is the low level window class that implements a tabbed panel control.
 *
 * @author  Leif Erik Larsen
 * @since   2001.01.29
 */
class GTabbedPanel : public GWindow
{
   public:

      /**
       * Flags used to set the position of the tabs.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.29
       * @see     #setTabPlacement
       */
      enum TabsPos
      {
         TOP,
         BOTTOM
      };

      /**
       * Internal class used to represent each tab in memory.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      class Page : public GObject
      {
         public:

            int sysID;
            GString userID;
            GString textStr;
            GString hintStr;
            GWindow* win;
            bool autoDelWin;
            bool showHintStr;

         public:

            Page ( int sysID, 
                   const GString& userID, 
                   const GString& textStr,
                   const GString& hintStr,
                   bool showHintStr,
                   const GString& iconName,
                   GWindow* win, 
                   bool autoDelWin );

            virtual ~Page ();

         private:

            /** Disable the copy constructor. */
            Page ( const Page& src ) : win(src.win) {}

            /** Disable the assignment operator. */
            Page& operator= ( const Page& ) { return *this; }

         public:

            int getSysID () const;
            const GString& getUserID () const;
            GWindow& getWin () const;
      };

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.08.30
       */
      class Peer : public GWindow
      {
         private:

            GTabbedPanel& tp;
            TabsPos tabsPos;

         public:

            Peer ( GTabbedPanel& tp, TabsPos tabsPos = TOP );

            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) : tp(src.tp) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }

         public:

            /** 
             * Get the insets used by the tab page. This is typically the 
             * space occupied by the tabs bar, hint bar, system borders, etc.
             */
            GInsets getPageInsets ();

            virtual void layout ();
      };

      Peer peer;

      /**
       * Index of which is the current active tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      int activeTabIndex;

      /**
       * This bag will contain each and every tabbed window, using the
       * tab-ID as the key.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      GArray<Page> tabs;

      /**
       * The horizontal number of pixels in the icons used on the tabs.
       */
      int iconWidth;

      /**
       * The vertical number of pixels in the icons used on the tabs.
       */
      int iconHeight;

      /** False if the tabs are not visible on screen. */
      bool tabsAreVisible;

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       * @param   iconWidth The width (in pixels) of the icons used on the 
       *                    tabs (if any). All icons must have the same size.
       *                    If an icon with a different size is used then
       *                    it will be stretched or shrinked to this size.
       * @param   iconHeight Same as <i>iconWidth</i>, but for the height
       *                    of the icons.
       */
      GTabbedPanel ( const GString& name,
                     const GString& constraints,
                     GWindow& parentWinl,
                     long winStyle = WS_VISIBLE,
                     TabsPos tabsPos = TOP,
                     int iconWidth = 20,
                     int iconHeight = 20 );

      ~GTabbedPanel ();

   private:

      /** Disable the copy constructor. */
      GTabbedPanel ( const GTabbedPanel& src ) : peer(*this) {}

      /** Disable the assignment operator. */
      GTabbedPanel& operator= ( const GTabbedPanel& ) { return *this; }

   private:

      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      virtual bool isEmpty () const;
      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual GString queryValue () const;

      virtual void setEnabled ( bool flag = true, bool repaint = true );
      virtual void setOily ( bool flag );

      /**
       * Calculation of the peer-window position and size needs to respect 
       * the "invisible tabs" property.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.10
       */
      virtual void layout ();

      virtual GWindow& getComponentByID ( const GString& id );
      virtual GWindow& getComponentByIndex ( int index );
      virtual int getComponentCount () const;

      /**
       * Activate the indexed tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @throws  GArrayIndexOutOfBoundsException if the specified index
       *                                          is out of bounds.
       */
      void activateTab ( int index );

      /**
       * Activate the specified tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @throws  GIllegalArgumentException if the specified tab
       *                                    does not exist.
       */
      void activateTab ( const GString& tabID );

      /**
       * Create and add a new tab and associate it with the specified dialog.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.30
       * @param   dlgID     ID of which dialog resource to use as template.
       * @param   msgProc   Pointer to the message handler. Can be null.
       * @param   tabID     The string to use as the ID of the new tab.
       *                    If an empty string is specified then we will
       *                    use the specified dialog resource name as the ID.
       * @param   tabText   The text to use on the tab, or an empty string
       *                    to use the default. The default is the same as
       *                    the TEXT parameter of the specified dialog
       *                    resource. The string can be a text-resource ID.
       * @param   tabHint   The text to use as the tab hint, or an empty string
       *                    to use the default. The default is the same as
       *                    the HINT of the text specified with the TEXT
       *                    parameter of the specified dialog resource.
       *                    The string can be a text-resource ID.
       * @param   showHint  True if we are to display the hint text area of 
       *                    the new page, or else false in order to hide it.
       * @return  A reference to the newly created dialog panel.
       * @throws  GRuntimeException in case of any error, for instance if the
       *                            specified dialog resource does not exist.
       * @see     #addTab
       */
      GDialogPanel& addDialogTab ( const GString& dlgID, 
                                   GDialogMessageHandler* msgProc = null, 
                                   const GString& tabID = GString::Empty, 
                                   const GString& tabText = GString::Empty, 
                                   const GString& tabHint = GString::Empty,
                                   bool showHint = true );

      /**
       * Create a new tab and associate it with the specified window.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @param   win  The window of which to show when the new tab is 
       *               activated. This window should be created with either
       *               the GTabbedPanel it self or the parent of the 
       *               GTabbedPanel as its parent. We will change the 
       *               parent window of the specified window to be the 
       *               system dependent peer of the GTabbedPanel.
       * @throws  GIllegalArgumentException if the specified tab-ID is
       *                                    already in use by another tab.
       * @throws  GSystemErrorLevelException if the system fails on creating
       *                                     the tab. This is not normal
       *                                     behaviour, and will probably
       *                                     never happen, but if it does
       *                                     then this exception is thrown.
       * @see     #addDialogTab
       */
      void addTab ( const GString& tabID, 
                    const GString& textID, 
                    const GString& hintID, 
                    bool showHint,
                    const GString& iconName,
                    GWindow* win, 
                    bool autoDelWin );

      /**
       * Get the index of which is the current active tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      int getActiveTabIndex () const;

      /**
       * Get a reference to the indexed window tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @throws  GArrayIndexOutOfBoundsException if the specified index
       *                                          is out of bounds.
       */
      GWindow& getIndexedTab ( int index ) const;

      /**
       * Get a reference to the indexed page.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       * @throws  GArrayIndexOutOfBoundsException if the specified index
       *                                          is out of bounds.
       */
      const Page& getIndexedTabPage ( int index ) const;

      /**
       * Get the window panel of the specified tab.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.29
       * @throws  GIllegalArgumentException if the specified tab-ID
       *                                    does not exist.
       */
      GWindow& getTab ( const GString& tabID );

      /**
       * Get the number of tabs currently contained in the component.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      int getTabCount () const;

      /**
       * Get the index of the specified tab, or -1 if the tab does not exist.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       */
      int getTabIndex ( const GString& tabID ) const;

      /**
       * Remove the tab which ID equals the specified tab-ID.
       *
       * If the tab does not exist then we will do nothing but
       * return false.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @return  True if the specified tab did in fact exist and we
       *          removed it, or else false if the specified tab
       *          does not exist.
       */
      bool removeTab ( const GString& tabID );

      /**
       * Remove the indexed tab.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.30
       * @throws  GArrayIndexOutOfBoundsException if the specified index
       *                                          is out of bounds.
       */
      void removeTabAt ( int index );

      /**
       * Makes the tabs bar of the tabs component visible or invisible.
       * By default the tabs bar is visible, but for some cases it might 
       * be usable to make the tabs bar invisible in order to get more 
       * free screen space for the contained page face area.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.10
       */
      void setTabsVisible ( bool visible );
};

#endif
