/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_STATUSBAR
#define __GLIB_STATUSBAR

#include "glib/gui/GAbstractToolbarWindow.h"
#include "glib/gui/GStaticText.h"

/**
 * This is the statusbar class of the Genious Library.
 *
 * @author  Leif Erik Larsen
 * @since   1996.06.10
 */
class GStatusbar : public GAbstractToolbarWindow
{
   public:

      enum CellPos { LEFT = 0, RIGHT = 1 };

   private:

      /**
       * A Cell acts as a container for a application defined window,
       * and draws cell border and insets (if any).
       * 
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       */
      class Cell : public GWindow
      {
         friend class GStatusbar;

         private:

            GStatusbar& sbar;
            CellPos pos;
            GWindow* win; // Application defined window to show in the cell.
            bool autoDelWin;

         private:

            Cell ( GStatusbar& sbar, 
                   CellPos pos = LEFT, 
                   GWindow* win = null, 
                   bool autoDelWin = false,
                   int winStyle2 = WS2_OS2Y );

            virtual ~Cell ();

            /** Disable the copy constructor. */
            Cell ( const Cell& src ) : sbar(sbar), pos(LEFT), win(null), autoDelWin(false) {}

            /** Disable the assignment operator. */
            Cell& operator= ( const Cell& ) { return *this; }
      };

      GArray<Cell> cells;

      /** 
       * True if the hint-cell should be shown and hidden in a 
       * synchronized way. This property is false by default.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.10.15
       * @see     #setSynchronizedHint
       */
      bool synchHint;

      /**
       * The component that is wrapped by {@link #hintCell}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       */
      GStaticText hintText;

      /**
       * The component that is wrapped by {@link #mainCell}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       */
      GStaticText mainText;

      /** 
       * The hint cell is visible only when it contains some text. 
       * The text is usually non-blank only temporarily, typically 
       * when some menu command or toolbar button is currently 
       * selected and the application want to display some hint text
       * for that menu command to the user. When the hint cell is 
       * visible, all other statusbar cells are hidden. It is up to the 
       * application when the hint cell is visible. It is made 
       * visible by calling {@link #setHint} with a non-empty 
       * string. Call {@link #setHint} again, with an empty string,
       * in order to hide the hint cell an make the other cells visible.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       */
      Cell hintCell;

      /** 
       * The main cell automatically fills up the rest of the 
       * statusbar area after all application defined cells are 
       * layed out.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       */
      Cell mainCell;

   public:

      GStatusbar ( const GString& name,
                   const GString& constraints,
                   GWindow& parentWin );
      
      virtual ~GStatusbar ();

   private:

      /** Disable the copy constructor. */
      GStatusbar ( const GStatusbar& src ) 
         : GAbstractToolbarWindow(GString::Empty, GString::Empty, *src.getParentWindow()),
           hintText(GString::Empty, GString::Empty, *src.getParentWindow()), 
           mainText(GString::Empty, GString::Empty, *src.getParentWindow()),
           hintCell(*this), mainCell(*this) {}

      /** Disable the assignment operator. */
      GStatusbar& operator= ( const GStatusbar& ) { return *this; }

   protected:

      /**
       * This method gets automatically called when someone change the font 
       * the statusbar. We will automatically update all text cells of the 
       * statusbar so their respective fonts are also set correspondingly.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.29
       */
      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.10.06
       */
      virtual bool onTimer ( const GString& timerID, GObject* userData );

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @param   name   The name of the cell. 
       *                 This name can be used as argument to other cell 
       *                 manipulation methods of GStatusbar in order to 
       *                 identify which cell to handle. Thus, all cells 
       *                 added must have unique names.
       * @param   cmdID  The button command id. This is the ID used to 
       *                 identify which command to execute when the user 
       *                 clicks on the button. If this string is specified 
       *                 as en ampty string then we will use the same 
       *                 ID as specified with the <i>name</i> parameter.
       * @param   iconID The resource name of which icon to show on the 
       *                 button face, or an empty string to use no icon.
       * @param   pos    {@link #LEFT to add the cell to the left edge of the
       *                 statusbar, or {@link #RIGHT} to add it to the rigth.
       * @return  A reference to the added button cell component.
       * @param   GIllegalArgumentException if the specified name is 
       *                 already in use by some other cell on the statusbar.
       */
      class GToolbarButton& addButtonCell ( const GString& name, 
                                            const GString& cmdID = GString::Empty,
                                            const GString& iconID = GString::Empty,
                                            CellPos pos = RIGHT );

      /**
       * Add another cell to the statusbar, and assign the specified 
       * window to the cell. The specified window will change its 
       * own parent window to its associated cell, so the application 
       * should usually only give this method a newly created window 
       * that was created without any parent.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.19
       * @param   win         The window to put into the new statusbar cell.
       *                      The {@link GWindow#getPreferredWidth} method
       *                      will be used to determine the width of the
       *                      statusbar cell that will contain this window.
       * @param   pos         LEFT to add the cell to the left edge of the
       *                      statusbar, or RIGHT to add it to the rigth edge.
       * @param   autoDelete  True if we shall automatically destroy the
       *                      specified window when the statusbar is
       *                      destroyed, or else false.
       * @param   GIllegalArgumentException if the name of the specified 
       *                      window is already in use by some other cell
       *                      on the statusbar.
       */
      void addCell ( GWindow* win, 
                     CellPos pos = RIGHT, 
                     bool autoDelete = true );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @param   name  The name of the cell. This name can be used as
       *                argument to other cell manipulation methods of 
       *                GStatusbar in order to identify which cell to 
       *                handle. Thus, all cells added must have unique 
       *                names.
       * @param   width Preferred width of the cell, measured in character 
       *                units in the same way as for component positioning 
       *                in dialog panels. If a zero width is specified then 
       *                we will use the default preferred width, as is 
       *                calculated by the new GStaticText component it self,
       *                via method {GStaticText#getPreferredWidth}.
       * @param   pos   {@link #LEFT to add the cell to the left edge of the
       *                statusbar, or {@link #RIGHT} to add it to the rigth.
       * @return  A reference to the added text cell component.
       * @param   GIllegalArgumentException if the specified name is 
       *                already in use by some other cell on the statusbar.
       */
      class GStaticText& addTextCell ( const GString& name, 
                                       double width = 0, 
                                       CellPos pos = RIGHT );

      /**
       * Get a cell recently added by {@link #addCell}.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.19
       * @param   name    Name of which cell to get.
       * @return  A pointer to the named cell, or null if there is no
       *          cell added with the specified name.
       */
      GWindow* getCell ( const GString& name );

      /**
       * Get the content of the hint cell.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       * @see     #setHint
       * @see     #getText
       */
      virtual GString getHint ();

      /**
       * Get a reference to the window used as the cell for the hint text.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.15
       */
      GWindow& getHintWindow ();

      /**
       * Get the preferred height of the window, in pixels.
       */
      virtual int getPreferredHeight () const;

      /**
       * Get the content of the main text cell.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       * @see     #setText
       * @see     #getHint
       */
      virtual GString getText ();

      /**
       * Get a reference to the window used as the cell for the main text.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.15
       */
      GWindow& getTextWindow ();

      /**
       * Lays out all cells and components of the statusbar.
       */
      virtual void layout ();

      /**
       * Load and activate the profile variables for the statusbar
       * window, from the specified section name.
       * @see #writeProfile
       */
      virtual void queryProfile ( const GString& sectName );

      /**
       * Set the title text of the named cell. If the statusbar has no
       * cell with the specified name then we will silently ignore it
       * and do nothing but return.
       *
       * @author  Leif Erik Larsen
       * @since   2005.11.23
       * @param   name The name of which statusbar cell to update.
       * @param   txt  The new title text of the statusbar cell.
       * @see     #setCellValue
       */
      virtual void setCellText ( const GString& name, const GString& txt );

      /**
       * Set the content of the named cell. If the statusbar has no
       * cell with the specified name then we will silently ignore it
       * and do nothing but return.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @param   name The name of which statusbar cell to update.
       * @param   val  The new value of the statusbar cell.
       * @see     #setCellText
       */
      virtual void setCellValue ( const GString& name, const GString& val );

      /**
       * Set the content of the hint cell. If the content is 
       * empty then we will hide the hint cell and show all the other 
       * statusbar cells. If it is not empty then we will show the hint 
       * cell and hide all the other statusbar cells.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @see     #getHint
       * @see     #setText
       * @see     GFrameWindow#setStatusbarHint
       */
      virtual void setHint ( const GString& hint );

      /** 
       * Set whether or not the hint-cell should be shown and hidden in 
       * a synchronized way. This property is false by default.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.10.15
       */
      void setSynchronizedHint ( bool synch );

      /**
       * Set the content of the main text cell.
       * The main text cell is usually the largest statusbar cell, and 
       * automatically fills up the rest of the statusbar area after all 
       * application defined cells (added by {@link #addCell}) have been 
       * layed out.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @see     #getText
       * @see     #setHint
       * @see     GFrameWindow#setStatusbarText
       */
      virtual void setText ( const GString& text );

      /**
       * Write the profile variables for the statusbar window, under the
       * specified section name.
       * @see #queryProfile
       */
      virtual void writeProfile ( const GString& sectName, bool force );
};

#endif
