/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GStatusbar.h"
#include "glib/gui/GFrameWindow.h"
#include "glib/gui/GColor.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GStaticText.h"
#include "glib/gui/GToolbarButton.h"
#include "glib/gui/GTooltip.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/util/GMath.h"
#include "glib/GProgram.h"

GStatusbar::Cell::Cell ( GStatusbar& sbar, 
                         CellPos pos, 
                         GWindow* win, 
                         bool autoDelWin,
                         int winStyle2 ) 
                  :GWindow(win->getName(),
                           GString::Empty,
                           &sbar, 
                           &sbar,
                           WS_VISIBLE,
                           WS2_OS2Y | winStyle2),
                   sbar(sbar),
                   pos(pos), 
                   win(win),
                   autoDelWin(autoDelWin)
{
   setInsets(new GInsets(2, 2, 2, 2), true);
   setBorder(new GLineBorder(), true);
   setLayoutManager(new GBorderLayout(), true);
   win->setParentWindow(this);
}

GStatusbar::Cell::~Cell ()
{ 
   win->setParentWindow(&GWindow::GetHiddenDummy());
   if (autoDelWin)
      delete win; 
}

GStatusbar::GStatusbar ( const GString& name,
                         const GString& constraints,
                         GWindow& parentWin )
           :GAbstractToolbarWindow(name, 
                                   constraints, 
                                   parentWin, 
                                   false, // clipChildren
                                   WS_VISIBLE, // winStyle
                                   WS2_OS2Y | WS2_AUTO_PAINT_BACKGROUND), // winStyle2
            cells(8),
            synchHint(false),
            hintText("HintText", 
                     GString::Empty, 
                     GWindow::GetHiddenDummy(), 
                     WS_VISIBLE, 
                     WS2_IGNORE_FONT_PROFILE, 
                     GStaticText::LEFTMARGINE, 
                     false, 
                     false),
            mainText("MainText", 
                     GString::Empty, 
                     GWindow::GetHiddenDummy(), 
                     WS_VISIBLE, 
                     WS2_IGNORE_FONT_PROFILE, 
                     GStaticText::LEFTMARGINE, 
                     false, 
                     false),
            hintCell(*this, LEFT, &hintText, false),
            mainCell(*this, LEFT, &mainText, false)
{
   setInsets(new GInsets(2, 0, 0, 0), true);
   setBorder(new GLineBorder(), true);
   setBackgroundColor(defaultBackgroundColor = parentWin.getBackgroundColor());
   hintText.setBackgroundColor(hintText.defaultBackgroundColor = GTooltip::DefBckColor);
   hintText.setForegroundColor(hintText.defaultForegroundColor = GTooltip::DefFrgColor);
}

GStatusbar::~GStatusbar ()
{
}

void GStatusbar::writeProfile ( const GString& sectName, bool force )
{
   /*
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   ini.putBool(sectName, "Visible", isVisible(), force || prg.isAutoSaveVisibilityStateOptions());
   */
   GAbstractToolbarWindow::writeProfile(sectName, force);
}

void GStatusbar::queryProfile ( const GString& sectName )
{
   /*
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   bool visible = ini.getBool(sectName, "Visible", isVisible());
   setVisible(visible);
   */
   GAbstractToolbarWindow::queryProfile(sectName);
}

void GStatusbar::setSynchronizedHint ( bool synch )
{
   synchHint = synch;
}

GWindow& GStatusbar::getHintWindow ()
{
   return hintText;
}

GString GStatusbar::getHint ()
{
   return hintText.getText();
}

void GStatusbar::setHint ( const GString& hint )
{
   GString prevHint = getHint();
   if (hint == "")
   {
      if (isTimerStarted("ShowHint"))
         stopTimer("ShowHint");
      if (prevHint == "")
         return;
      if (synchHint)
      {
         hintText.setText(hint);
         layout();
         return;
      }
      // Hide the hint cell after some milliseconds delay, in order 
      // to prevent som uggly flashing when user drags the mouse 
      // over several toolbar buttons in sequence.
      if (!isTimerStarted("ClearHint"))
         startTimer("ClearHint", 500);
      // Don't set the (empty-) hint here. We will clear it on timer event.
      // hintText.setText(hint);
   }
   else
   {
      if (isTimerStarted("ClearHint"))
         stopTimer("ClearHint");
      // Update the hint cell, and show (layout) it after some time.
      hintText.setText(hint);
      if (!isTimerStarted("ShowHint"))
         startTimer("ShowHint", 500);
   }
}

GWindow& GStatusbar::getTextWindow ()
{
   return mainText;
}

GString GStatusbar::getText ()
{
   return mainText.getText();
}

void GStatusbar::setText ( const GString& text )
{
   GString prevText = getText();
   if (text == prevText)
      return;
   mainText.setText(text);
   mainText.updateWindow();
}

void GStatusbar::setCellText ( const GString& name, const GString& txt )
{
   GWindow* cell = getCell(name);
   if (cell == null)
      return;
   GString prevTxt = cell->getText();
   if (txt == prevTxt)
      return;
   int prevPrefWidth = cell->getPreferredWidth();
   cell->setText(txt);
   int newPrefWidth = cell->getPreferredWidth();
   if (newPrefWidth != prevPrefWidth)
   {
      // Don't paint until cells has been layed out.
      cell->validateAll(true); 
      postLayout();
   }
}

void GStatusbar::setCellValue ( const GString& name, const GString& val )
{
   GWindow* cell = getCell(name);
   if (cell == null)
      return;
   GString prevVal = cell->queryValue();
   if (val == prevVal)
      return;
   int prevPrefWidth = cell->getPreferredWidth();
   cell->changeValue(val);
   int newPrefWidth = cell->getPreferredWidth();
   if (newPrefWidth != prevPrefWidth)
   {
      // Don't paint until cells has been layed out.
      cell->validateAll(true); 
      postLayout();
   }
}

GStaticText& GStatusbar::addTextCell ( const GString& name, double width, CellPos pos )
{
   GStaticText* cell;
   cell = new GStaticText(name, GString::Empty, GWindow::GetHiddenDummy(), WS_VISIBLE, 0, GStaticText::CENTER);
   if (width > 0)
   {
      int w = int(width * (getWidthOfString(GWindow::FontDimTemplateStr) / GWindow::FontDimTemplateStr.length()));
      cell->setPreferredSize(new GDimension(w, 0), true);
   }
   cell->setBackgroundColor(cell->defaultBackgroundColor = getBackgroundColor().getDarker(32));
   cell->setForegroundColor(cell->defaultForegroundColor = GColor::BLACK);
   addCell(cell, pos, true);
   return *cell;
}

GToolbarButton& GStatusbar::addButtonCell ( const GString& name, 
                                            const GString& cmdID,
                                            const GString& iconID,
                                            CellPos pos )
{
   GToolbarButton* cell;
   cell = new GToolbarButton(name,
                             GString::Empty,
                             GWindow::GetHiddenDummy(), // parentWin, is set upon addCell()
                             cmdID == GString::Empty ? name : cmdID, // id
                             GString::Empty, // text
                             iconID);
   cell->setBackgroundColor(cell->defaultBackgroundColor = getBackgroundColor().getDarker(32));
   cell->setForegroundColor(cell->defaultForegroundColor = GColor::BLACK);
   addCell(cell, pos, true);
   return *cell;
}

void GStatusbar::addCell ( GWindow* win, CellPos pos, bool autoDelete )
{
   GString winName = win->getName();
   GWindow* tmp = getCell(winName);
   if (tmp != null)
      gthrow_(GIllegalArgumentException(GString("Name of specified window already in use in the statusbar: '%s'", GVArgs(winName))));
   win->setBackgroundColor(win->defaultBackgroundColor = getBackgroundColor().getDarker(32));
   win->setForegroundColor(win->defaultForegroundColor = GColor::BLACK);
   Cell* p = new Cell(*this, pos, win, autoDelete);
   cells.add(p);
   layout();
}

GWindow* GStatusbar::getCell ( const GString& name )
{
   for (int i=0, num=cells.getCount(); i<num; i++)
   {
      Cell& p = cells[i];
      const GString& pn = p.win->getName();
      if (pn == name)
         return p.win;
   }
   return null;
}

void GStatusbar::layout ()
{
   GInsets ins = getInsets();
   GRectangle r = getWindowRect();
   int y = r.y + ins.bottom;
   int h = r.height - ins.bottom - ins.top;

   GString hint = hintText.getText();
   if (hint != "")
   {
      // Hide all cells, except the cell for the hint.
      mainCell.setVisible(false);
      for (int i=cells.getCount()-1; i>=0; i--)
      {
         Cell& p = cells[i];
         p.setVisible(false);
      }
      hintCell.setWindowBounds(0, y, r.width, h);
      hintCell.setVisible(true);
      hintCell.updateWindow();
   }
   else
   {
      // Hide the cell for the hint, and show all other cells.
      int leftX = r.x;
      int rightX = r.x + r.width;
      hintCell.setVisible(false);
      for (int i=cells.getCount()-1; i>=0; i--)
      {
         Cell& p = cells[i];
         int prefWidth = p.win->getPreferredWidth() + 4;
         if (p.pos == LEFT)
         {
            int w = GMath::Max(0, prefWidth);
            p.setWindowBounds(leftX, y, w, h);
            leftX += prefWidth;
         }
         else // if (p.pos == RIGHT)
         {
            rightX -= prefWidth;
            int w = GMath::Max(0, prefWidth);
            p.setWindowBounds(rightX, y, w, h);
         }
         p.setVisible(true);
         p.updateWindow();
      }
      mainCell.setWindowBounds(leftX, y, GMath::Max(0, rightX - leftX), h);
      mainCell.setVisible(true);
      mainCell.updateWindow();
   }
}

int GStatusbar::getPreferredHeight () const
{
   if (!isVisible())
      return 0;
   const GInsets& ins = getInsets();
   return getHeightOfString("X") + 8 + ins.top + ins.bottom;
}

bool GStatusbar::onFontNameSizeChanged ( const GString& fontNameSize )
{
   for (int i=0, num=cells.getCount(); i<num; i++)
   {
      Cell& p = cells[i];
      p.win->setFontNameSize(p.win->defaultFontNameSize = fontNameSize);
   }
   hintText.setFontNameSize(hintText.defaultFontNameSize = fontNameSize);
   mainText.setFontNameSize(mainText.defaultFontNameSize = fontNameSize);
   GWindow::onFontNameSizeChanged(fontNameSize);
   return true;
}

bool GStatusbar::onTimer ( const GString& timerID, GObject* userData )
{
   if (timerID == "ShowHint")
   {
      // Show the hint cell.
      layout();
      return true;
   }
   else
   if (timerID == "ClearHint")
   {
      stopTimer("ClearHint");
      hintText.setText(GString::Empty);
      layout();
      return true;
   }
   else
   {
      return GWindow::onTimer(timerID, userData);
   }
}
