/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SCROLLBAR
#define __GLIB_SCROLLBAR

#include "glib/gui/GWindow.h"

/**
 * This is the Scrollbar Component of G-Lib.
 * It is probably to be used in conjunction with <i>GDecoratedWindow</i>
 * only.
 *
 * This scrollbar class uses 32 bit integers to represent the scroll
 * range and thumb position, even if the underlying system actually only
 * supports 16 bit integers for this purpose. This means that applications can
 * set larger scroll ranges just as easily as if the native API's did support
 * it directly.
 *
 * @author Leif Erik Larsen
 * @since  1999.09.27
 */
class GScrollbar : public GWindow
{
   friend class GContainerWindow;

   private:

      const int MAX_SYSTEM_SCROLL_RANGE;
      int realWorldScrolRange;
      int thumbLen;

   public:

      GScrollbar ( const GString& name, 
                   const GString& constraints, 
                   GWindow& parentWin,
                   bool vertical,
                   long winStyle = WS_VISIBLE, 
                   long winStyle2 = 0 );

      virtual ~GScrollbar ();

   private:

      /** Disable the copy constructor. */
      GScrollbar ( const GScrollbar& src ) : MAX_SYSTEM_SCROLL_RANGE(0) {}

      /** Disable the assignment operator. */
      GScrollbar& operator= ( const GScrollbar& ) { return *this; }

   public:

      static int GetDefaultVScrollbarWidth ();
      static int GetDefaultHScrollbarHeight ();
      static int GetDefaultScrollbarColor ();

   public:

      // Methods used to communicate with 32 bits scroll position and range
      // values as used in the application.

      int getScrollPos () const;
      int getScrollRange () const;
      int getThumbLength () const;
      void setScrollPos ( int pos );
      void setScrollPosAndRange ( int pos, int scrolLen, int thumbLen );
      void setScrollRange ( int scrolLen, int thumbLen );

      /**
       * Convert the specified 32 bit scroll position (as seen by the
       * application) to the corresponding 16 bit scroll position (as
       * seen by the underlying system).
       */
      int convertFromSysPos ( int syspos ) const;

      /**
       * Convert the specified 16 bit scroll position (as seen by the
       * underlying system) to the corresponding 32 bit scroll position (as
       * seen by the application).
       */
      int convertToSysPos ( int pos ) const;

      // Methods used to communicate with 16 bits scroll position and range
      // values as used in the OS/2 system.

      /**
       * @author  Leif Erik Larsen
       * @since   2006.04.21
       * @param   currentThumbTrackPos True if we are called from code that 
       *                  handles the thumb tracking event, and needs the 
       *                  current thumb track position and not the actually 
       *                  current scroll position. 
       */
      int getSysScrollPos ( bool currentThumbTrackPos = false ) const;

      void setSysScrollPos ( int pos );
      void setSysScrollPosAndRange ( int pos, int scrolLen, int thumbLen );
      void setSysScrollRange ( int scrolLen, int thumbLen );
};

#endif

