/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_PUSHBUTTON
#define __GLIB_PUSHBUTTON

#include "glib/gui/GWindow.h"
#include "glib/gui/event/GAbstractCommand.h"

/**
 * This is the low level window class that implements a push button control.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.29
 */
class GPushButton : public GWindow                    
{
   public:

      /**
       * This string contains the platform dependent character that is used
       * by the underlying system to mark which character in the button face
       * text is to be drawn with a mnemonic style (usually an underscore).
       */
      static const char MNEMONIC_MARKER_CHAR;

   public:

      /** Flags used to identify the icon position with respect to the button text, if any. */
      enum ICONPOS
      {
         /** Center icon and don't draw the text even if the text is not empty. */
         IP_CENTER,

         /** Draw the icon below the text. */
         IP_BOTTOM,

         /** Draw the icon above the text. */
         IP_TOP,

         /** Draw the icon to the left of the text. */
         IP_LEFT,

         /** Draw the icon to the right of the text. */
         IP_RIGHT
      };

   private:

      /**
       * This class is used to represent our own "command".
       */
      class Command : public GAbstractCommand
      {
         public:

            explicit Command ( const GString& id = GString::Empty );
            virtual ~Command ();
      };

      /**
       * This class is used to wrap the system dependent push-button window,
       * which is contained as a child window of GPushButton in order to hide
       * it from the user code.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.02
       */
      class Peer : public GWindow
      {
         friend class GPushButton;

         private:

            GPushButton& button;

         public:

            /** The owner push button. */
            explicit Peer ( GPushButton& button );
            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) : button(src.button) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }

         private:

            virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
            virtual void onFocus ( bool set );
      };

      friend class Peer;

   private:

      GString iconID;
      ICONPOS iconPos;
      bool isDefault;
      int hotKeyPos;
      bool useHotKey;
      bool noFocusPaint;
      bool noBoldBorder;
      mutable Peer peer;
      Command command;

   public:

      GPushButton ( const GString& name,
                    const GString& constraints,
                    GWindow& parentWin,
                    long winStyle = WS_VISIBLE,
                    long winStyle2 = 0,
                    const GString& iconID = GString::Empty,
                    bool isDefault = false,
                    bool noFocusPaint = false,
                    ICONPOS iconPos = IP_LEFT,
                    bool noBoldBorder = false,
                    bool useHotKey = true );

      virtual ~GPushButton ();

   private:

      /** Disable the copy constructor. */
      GPushButton ( const GPushButton& src ) : peer(*this) {}

      /** Disable the assignment operator. */
      GPushButton& operator= ( const GPushButton& ) { return *this; }

   private:

      void repaintImpl ( GGraphics& g, bool pressed );

   protected:

      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
      virtual bool onCommand ( int cmd );
      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      virtual void doClick ();
      GWindow& getPeer () { return peer; }
      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;
      virtual void grabFocus ( bool force = false );
      void setDefault ( bool flag );
      virtual void setEnabled ( bool flag = true, bool repaint = true );
      virtual void setOily ( bool flag );
      void setIconID ( const GString& iconID );
      virtual void setText ( const GString& text );

      /**
       * This method is called when the user has pressed the mnemonic
       * keyboard code (Alt+X, where X is the hot mnemonic character).
       *
       * We will perform just as if the push button has been clicked,
       * and this is achieved by calling {@link #doClick}.
       */
      virtual void hotKeyAction ();
};

#endif
