/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GProgressBar.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/util/GMath.h"

GProgressBar::GProgressBar ( const GString& name, 
                             const GString& constraints, 
                             GWindow& parentWin, 
                             long winStyle, 
                             long winStyle2, 
                             int minValue, 
                             int maxValue, 
                             bool horizontal )
             :GWindow(name, 
                      constraints, 
                      &parentWin, 
                      &parentWin, 
                      winStyle, 
                      winStyle2 | WS2_OS2Y,
                      GWindowClass::GENERIC,
                      GColor::WHITE,
                      GColor::DRED,
                      GPROGRESSBAR_DEFAULT_FONT),
              horizontal(horizontal),
              minValue(GMath::Min(minValue, maxValue)),
              maxValue(GMath::Max(minValue, maxValue)),
              curValue(GMath::Min(minValue, maxValue))
{
   setBorder(new GLineBorder(), true);
   setInsets(new GInsets(1, 1, 1, 1), true);
}

GProgressBar::~GProgressBar ()
{
}

bool GProgressBar::onPaint ( GGraphics& g, const GRectangle& /*rect*/ )
{
   GColor frg = getForegroundColor();
   GColor bck = getBackgroundColor();
   GRectangle r = getWindowRect();
   r.inflateRect(-1); // Don't include area of our border.

   int total = maxValue - minValue;
   if (total > 0)
   {
      int width = (horizontal ? r.width : r.height);

      // Use a factor of 200 rather than 100 to get a smoother
      // and more accurate meter bar.
      int percentFinished = (curValue * 200) / total;
      int widthOfRed = (width * percentFinished) / 200;
      int widthOfGray = width - widthOfRed - 1;
      if (widthOfGray < 0)
         widthOfGray = 0;

      // Paint the Red (relative finished) area
      if (horizontal)
         r.width -= widthOfGray;
      else
         r.height -= widthOfGray;
      g.drawFilledRectangle(r, frg);

      // Paint the Gray (relative remaining) area
      if (widthOfGray > 0)
      {
         if (horizontal)
         {
            r.x += widthOfRed;
            r.width = widthOfGray;
         }
         else
         {
            r.y += widthOfRed;
            r.height = widthOfGray;
         }
         g.drawFilledRectangle(r, bck);
      }
   }
   else
   {
      if (horizontal)
         r.width -= 1;
      else
         r.height -= 1;
      g.drawFilledRectangle(r, frg);
   }

   return true;
}

void GProgressBar::setCurrentValue ( int cur )
{
   curValue = GMath::Max(minValue, GMath::Min(maxValue, cur));
   invalidateAll(false);
}

void GProgressBar::setMaxValue ( int max )
{
   maxValue = max;
   invalidateAll(false);
}

void GProgressBar::setMinValue ( int min )
{
   minValue = min;
   invalidateAll(false);
}

void GProgressBar::setHorisontal ( bool flag )
{
   horizontal = flag;
   invalidateAll(false);
}

bool GProgressBar::isEmpty () const
{
   return false;
}

void GProgressBar::changeValue ( const GString& newValue, bool notify )
{
   int newval;
   try {
      newval = GInteger::ParseInt(newValue);
   } catch (GNumberFormatException& /*e*/) {
      return;
   }

   enterCtrlChangedFilter();
   setCurrentValue(newval);
   exitCtrlChangedFilter();

   if (notify)
      fireValueChangeListeners();
}

GString GProgressBar::queryValue () const
{
   int val = getCurrentValue();
   return GInteger::ToString(val);
}
