/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GListBox.h"
#include "glib/gui/GDropList.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/util/GMath.h"
#include "glib/exceptions/GIllegalArgumentException.h"

GListBox::Peer::Peer ( GListBox& listb )
               :GWindow("Peer",
                        GBorderLayout::CENTER,
                        &listb,
                        &listb,
                        WS_VISIBLE | LS_OWNERDRAW | LS_HORZSCROLL | LS_NOADJUSTPOS | (listb.multisel ? LS_MULTIPLESEL : 0),
                        WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                        GWindowClass::LISTBOX),
                listb(listb)
{
   setFocusable(true);
}

GListBox::Peer::~Peer ()
{
}

void GListBox::Peer::onFocusKill ()
{
   GWindow::onFocusKill();
   listb.peerHasLostFocus();
}

bool GListBox::Peer::onKeyDown ( const GKeyMessage& key )
{
   // Because list boxes are typically used in GDialogPanel's and such 
   // dialogs typically "grabs" the dialog keyboard navigation keys 
   // (such as UP, DOWN, TAB, ENTER, ESCAPE, etc.) we must explicitly 
   // handle those dialog navigation keys that are of interest to us.
   int count = listb.getItemCount();
   if (count > 0)
   {
      int curidx = listb.getSelectedIndex();
      switch (key.getCode())
      {
         case GKey::KEY_UP:
            listb.setSelectedIndex(GMath::Max(0, curidx-1));
            return true;
         case GKey::KEY_DOWN:
            listb.setSelectedIndex(GMath::Min(count-1, curidx+1));
            return true;
         default:
            // The other listbox navigation keys (e.g. HOME and PAGEUP)
            // can be ignored here because they won't be eaten up by the 
            // dialog keyboard handler and will therefore go to the 
            // system dependent keyboard handler so that the user can 
            // use the system default listbox keyboard navigation keys.
            break;
      }
   }

   // If the key is a character then select the next item in the list 
   // that match the entered character.
   if (key.isPureCharacter())
   {
      char chr = key.getCharacter();
      listb.selectNextCharMatch(chr); // Implicitly sends GM_CTRLCHANGED.
   }

   return GWindow::onKeyDown(key);
}

GListBox::GListBox ( const GString& name, 
                     const GString& constraints, 
                     GWindow& parentWin, 
                     bool multisel, 
                     long winStyle, 
                     long winStyle2 )
         :GWindow(name,
                  constraints,
                  &parentWin,
                  &parentWin,
                  winStyle,
                  winStyle2 | WS2_OS2Y | WS2_NOT_STATIC,
                  GWindowClass::GENERIC,
                  GColor::WHITE, 
                  GColor::BLACK),
          selectionListeners(3),
          heighestIcon(0),
          widestItemInPixels(0),
          multisel(multisel),
          prevIdx(-1),
          tglIconOn(null),
          tglIconOff(null),
          tglIconWidth(0),
          tglIconHeight(0),
          peer(*this)
{
   setLayoutManager(new GBorderLayout(), true);

   // Look up the toggle-icons that are used for multiple selection
   // list-boxes only. And precalculate the size of the screen area
   // that will be needed by them within each list box item.
   if (multisel)
   {
      tglIconOn = GIcon::GetIcon("%STDICON_LISTBOX_TOGGLEON");
      tglIconOff = GIcon::GetIcon("%STDICON_LISTBOX_TOGGLEOFF");
      if (tglIconOn == null || tglIconOff == null)
      {
         tglIconOn = tglIconOff = null; // Make sure both are null.
         tglIconWidth = 14;
         tglIconHeight = 14;
      }
      else
      {
         tglIconWidth = GMath::Max(tglIconOn->getWidth(), tglIconOff->getWidth());
         tglIconHeight = GMath::Max(tglIconOn->getHeight(), tglIconOff->getHeight());
      }
   }
}

GListBox::~GListBox ( void )
{
}

void GListBox::grabFocus ( bool force )
{
   peer.grabFocus(force);
}

void GListBox::setEnabled ( bool flag, bool repaint )
{
   peer.setEnabled(flag, false);
   GWindow::setEnabled(flag, repaint);
}

void GListBox::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

void GListBox::peerHasLostFocus ()
{
   GDropList::DropContainer::ListBox* drop = dynamic_cast<GDropList::DropContainer::ListBox*>(this);
   if (drop != null)
   {
      GDropList& dlist = drop->dlist;
      dlist.setDropListVisible(false, false);
      GWindow& top = dlist.getTopLevelWindow();
      GWindow* focus = top.getFocusedWindow();
      if (focus == null || !focus->isFocusable())
         dlist.grabFocus(false);
   }
}

int GListBox::getDefaultItemHeight () const
{
   int fonth = peer.getFontHeight();
   int iconh = GMath::Max(heighestIcon, tglIconHeight);
   if (iconh > 0)
      iconh += 2;
   return GMath::Max(fonth, iconh);
}

int GListBox::getItemCount () const
{
   return items.getCount();
}

void GListBox::addListBoxSelectionListener ( GListSelectionListener* l )
{
   if (l == null)
      gthrow_(GIllegalArgumentException("l == null"));
   if (!selectionListeners.contains(l)) // If not already registered
      selectionListeners.add(l);
}

int GListBox::getPreferredHeight () const
{
   return (6 * peer.getHeightOfString("X")) + 12;
}

int GListBox::getPreferredWidth () const
{
   return (16 * peer.getWidthOfString("X")) + 36;
}

void GListBox::updateItemHeight ()
{
   int height = getDefaultItemHeight();
   peer.sendMessage(LM_SETITEMHEIGHT, MPARAM(height), MPARAM(0));
}

void GListBox::addItem ( const GString& text, 
                         const GString& iconName, 
                         GObject* userData, 
                         bool autoDelUD )
{
   int end = getItemCount();
   insertItem(end, text, iconName, userData, autoDelUD);
}

void GListBox::insertItem ( int index, 
                            const GString& text, 
                            const GString& iconName, 
                            GObject* userData, 
                            bool autoDelUD )
{
   if (index == -1)
      index = getSelectedIndex();

   int count = getItemCount();
   if (index < 0 || index > count)
      index = count;

   GListBoxItem* lbi = null;

   if (iconName != "")
   {
      const GIcon* icn = GIcon::GetIcon(iconName);
      if (icn != null)
      {
         int iconHeight = icn->getHeight();
         if (iconHeight > getHeighestIcon())
            setHeighestIcon(iconHeight);
         lbi = new GListBoxItem(text, icn);
      }
   }

   if (lbi == null)
      lbi = new GListBoxItem(text);

   lbi->setUserData(userData, autoDelUD);

   items.insert(lbi, index);

   GGraphics g(peer);
   int width = measureItemWidth(g, *lbi);
   bool widestItemChanged = false;
   if (width > widestItemInPixels)
   {
      widestItemInPixels = width;
      widestItemChanged = true;
   }

   int pos = (index == count ? LIT_END : index);
   peer.sendMessage(LM_INSERTITEM, MPARAM(pos), MPARAM(text.cstring()));
   peer.sendMessage(LM_SETITEMHANDLE, MPARAM(index), MPARAM(lbi));

   updateItemHeight();

   if (prevIdx >= 0 && index <= prevIdx)
      prevIdx = -1;
}

int GListBox::getSelectedIndex () const
{
   int ret;
   ret = int(peer.sendMessage(LM_QUERYSELECTION, MPARAM(LIT_CURSOR)));
   return ret < 0 ? -1 : ret; // Make sure to return -1 if no item is selected!
}

const GListBoxItem& GListBox::getItem ( int index ) const
{
   if (index == -1)
      index = getSelectedIndex();
   return items[index];
}

void GListBox::setItemIcon ( int index, const GString& iconName )
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return;

   const GIcon* icn = GIcon::GetIcon(iconName);
   if (icn != null)
   {
      int iconHeight = icn->getHeight();
      if (iconHeight > getHeighestIcon())
         setHeighestIcon(iconHeight);
      updateItemHeight();
      GListBoxItem& item = items.get(index);
      item.setIcon(icn);
      peer.sendMessage(LM_SETITEMHANDLE, MPARAM(index), MPARAM(&item));
      repaintItem(index);
   }
}

void GListBox::setItemTextAndIcon ( int index, 
                                    const GString& text, 
                                    const GString& iconName )
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return;

   GListBoxItem& item = items.get(index);
   item.setText(text);
   const GIcon* icn = GIcon::GetIcon(iconName);
   if (icn != null)
   {
      int iconHeight = icn->getHeight();
      if (iconHeight > getHeighestIcon())
         setHeighestIcon(iconHeight);
      item.setIcon(icn);
   }

   updateItemHeight();
   peer.sendMessage(LM_SETITEMHANDLE, MPARAM(index), MPARAM(&item));
   repaintItem(index);
}

void GListBox::setItemText ( int index, const GString& text )
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return;

   GListBoxItem& item = items.get(index);
   item.setText(text);

   updateItemHeight();
   peer.sendMessage(LM_SETITEMHANDLE, MPARAM(index), MPARAM(&item));
   repaintItem(index);
}

GString GListBox::getItemText ( int index ) const
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return GString::Empty;

   return items[index].getText();
}

void GListBox::setItemUserData ( int index, GObject* userData, bool autoDelete )
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return;

   GListBoxItem& item = items.get(index);
   item.setUserData(userData, autoDelete);

   peer.sendMessage(LM_SETITEMHANDLE, MPARAM(index), MPARAM(&item));
   repaintItem(index);
}

GObject* GListBox::getItemUserData ( int index ) const
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return null;

   GListBoxItem& item = items.get(index);
   return item.getUserData();
}

bool GListBox::removeItem ( int index )
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return false;

   items.remove(index);
   peer.sendMessage(LM_DELETEITEM, MPARAM(index));
   if (prevIdx >= 0 && index <= prevIdx)
      prevIdx = -1;
   return true;
}

void GListBox::removeAllItems ()
{
   items.removeAll();
   peer.sendMessage(LM_DELETEALL);
   prevIdx = -1;
}

void GListBox::setSelectedIndex ( int index )
{
   int curidx = getSelectedIndex();
   if (curidx == index)
      return;
   if (index >= 0 && index < getItemCount())
      peer.sendMessage(LM_SELECTITEM, MPFROMSHORT(index), MPFROMSHORT(true));
   else
      peer.sendMessage(LM_SELECTITEM, MPARAM(LIT_NONE));
}

bool GListBox::isItemSelected ( int index )
{
   if (index <= -1)
      return false;

   if (multisel)
   {
      short idx = short(index);
      short nextSelectedIndex = short(int(peer.sendMessage(LM_QUERYSELECTION, MPFROMSHORT(idx-1))));
      if (nextSelectedIndex == idx)
         return true;
      else
         return false;
   }
   else
   {
      int sel = getSelectedIndex();
      if (index >= 0 && index == sel)
         return true;
      else
         return false;
   }
}

void GListBox::setItemSelected ( int index, bool sel )
{
   if (index <= -1 || index >= getItemCount())
      return;

   if (multisel)
   {
      peer.sendMessage(LM_SELECTITEM, MPFROMSHORT(index), MPFROMSHORT(sel));
   }
   else
   {
      if (sel)
         setSelectedIndex(index);
      else
      if (getSelectedIndex() == index)
         setSelectedIndex(-1); // Deselect current selection.
   }
}

void GListBox::repaintItem ( int index ) const
{
   if (index == -1)
      index = getSelectedIndex();

   if (index < 0 || index >= getItemCount())
      return;

   // Trick to get the indexed item repainted without too much work.
   GListBoxItem& item = items.get(index);
   const char* itemStr = item.getText().cstring();
   peer.sendMessage(LM_SETITEMTEXT, MPARAM(index), MPARAM(itemStr));
}

bool GListBox::selectNextCharMatch ( char chr )
{
   const int num = getItemCount();
   int idx = getSelectedIndex();
   for (int i=0; i<num; i++)
   {
      if (++idx >= num)
         idx = 0;
      const GListBoxItem& item = getItem(idx);
      const GString& itemText = item.getText();
      int itemTextLen = itemText.length();
      if (itemTextLen <= 0)
         continue;
      char itemFirstChar = itemText[0];
      if (chr != ' ')
      {
         // Find the first character in the item text that is not a space.
         for (int stridx=0; stridx<itemTextLen; stridx++)
         {
            itemFirstChar = itemText[stridx];
            if (itemFirstChar != ' ')
               break;
         }
      }
      if (toupper(chr) == toupper(itemFirstChar))
      {
         // Set selection, and implicitly send GM_CTRLCHANGED.
         setSelectedIndex(idx);

         // Get a working copy of the listener array, in case some listers are 
         // added or removed by the code being notifyed.
         const int count = selectionListeners.getCount();
         GVector<GListSelectionListener*> llist(count);
         for (int i=0; i<count; i++)
            llist.add(selectionListeners[i]);

         // Invoke the listeners.
         for (int i=0; i<count; i++)
         {
            GListSelectionListener* l = llist[i];
            l->onListBoxSelectionChangedByCharKey(*this);
         }
         return true;
      }
   }
   return false;
}

void GListBox::sendCtrlChanged ()
{
   GWindow::sendCtrlChanged();

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = selectionListeners.getCount();
   GVector<GListSelectionListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(selectionListeners[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GListSelectionListener* l = llist[i];
      l->onListBoxSelectionChanged(*this);
   }
}

int GListBox::getHeighestIcon () const 
{ 
   return heighestIcon; 
}

void GListBox::setHeighestIcon ( int height ) 
{ 
   heighestIcon = height; 
}

bool GListBox::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case LN_SELECT:
      // An item is being selected (or deselected).
      {
         int newIdx = getSelectedIndex();
         if (newIdx == prevIdx)
            break;
         prevIdx = newIdx;
         fireValueChangeListeners();
         break;
      }

      case LN_ENTER:
      // User has double-clicked or pressed enter on the current selected item.
      {
         // Ignore this event if caused by enter-key on keyboard.
         if (GWindow::IsAboutToHandleKey())
            break; 
         
         // Send the GM_LISTBOX_DBLCLICK message.
         GDialogPanel* dlg = getOwnerDialogPanel();
         if (dlg != null)
         {
            GString compID = getName();
            int curIdx = getSelectedIndex();
            GString curIdxStr = GInteger::ToString(curIdx);
            dlg->sendDialogMessage(GDialogMessageHandler::GM_LISTBOX_DBLCLICK, compID, curIdxStr);
         }

         // Get a working copy of the listener array, in case some listers are 
         // added or removed by the code being notifyed.
         const int count = selectionListeners.getCount();
         GVector<GListSelectionListener*> llist(count);
         for (int i=0; i<count; i++)
            llist.add(selectionListeners[i]);

         // Invoke the listeners.
         for (int i=0; i<count; i++)
         {
            GListSelectionListener* l = llist[i];
            l->onListBoxSelectionDblClick(*this);
         }
         break;
      }
   }

   return true;
}

int GListBox::measureItemWidth ( GGraphics& g, const GListBoxItem& item  ) const
{
   const GString& text = item.getText();
   int width = g.getWidthOfString(text) + 4;
   const GIcon* icon = item.getIcon();
   if (icon != null)
   {
      int iconWidth = icon->getWidth();
      width += iconWidth + 6;
   }
   if (tglIconWidth > 0)
      width += tglIconWidth + 6;
   return width;
}

GWindowMessage::Answer GListBox::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_DRAWITEM: 
      {
         if (msg.getParam1LoUShort() != peer.getWindowID())
            return GWindow::handleWindowMessage(msg);
         OWNERITEM* oi = (OWNERITEM*) msg.getParam2();
         if (oi == null) // Should never happen, but in case.
            return GWindow::handleWindowMessage(msg);
         int index = oi->idItem;
         if (index < 0 || index >= getItemCount())
            return GWindow::handleWindowMessage(msg);
         GGraphics g(oi->hps, peer, false, false);
         GRectangle rect = GWindow::MakeLibRect(oi->rclItem, g);
         bool selected = oi->fsState;
         drawItemImpl(g, index, rect, selected);
         // Make sure to clear the left margine. This is maybe needed due
         // to a bug in OS/2 Warp. Needed only when listbox has a horizontal
         // scrollbar which thumb is somewhere to the right. But it doesn't
         // hurt to do this anyway, so don't test on the horizontal thumb
         // position for speed optimizing reasons.
         rect.x = oi->rclItem.xLeft - 2;
         rect.width = 2;
         GColor defbck = peer.getBackgroundColor();
         g.drawFilledRectangle(rect, defbck);
         // Clear, so PM won't highlight after we have returned.
         oi->fsState = oi->fsStateOld = false;
         return MRESULT(TRUE);
      }

      case WM_MEASUREITEM: 
      {
         if (msg.getParam1LoUShort() != peer.getWindowID())
            return MRFROM2SHORT(GMath::Max(peer.getFontHeight(), getHeighestIcon()), 0);
         int height = getDefaultItemHeight();
         int index = msg.getParam2LoShort();
         if (index < 0 || index >= getItemCount())
            return MRFROM2SHORT(height, 0);
         return MRFROM2SHORT(height, widestItemInPixels); 
      }
   
      default:
         return GWindow::handleWindowMessage(msg);
   }
}

void GListBox::drawItemImpl ( GGraphics& g,
                              int index, 
                              const GRectangle& rect,
                              bool selected )
{
   const GListBoxItem& item = getItem(index);

   // Get the foreground and background colors of which to use.
   GColor defbck = peer.getBackgroundColor();
   GColor bck = ((selected && !multisel) ? GColor::DCYAN : defbck);
   GColor frg = ((selected && !multisel) ? GColor::WHITE : peer.getForegroundColor());

   // Fill the item rectangle with the selection or background color.
   GRectangle r = rect;
   g.drawFilledRectangle(r, bck);
   if (selected && !multisel)
   {
      g.setColor(GColor::BLACK);
      g.drawRectangle(r);
   }

   // Draw the toggle state, if this is for a multiple selection
   // list box.
   if (multisel)
   {
      int xpos = r.x + 2;
      const GIcon* icon = (selected ? tglIconOn : tglIconOff);
      if (icon != null)
      {
         int ypos = r.y + r.height/2 - icon->getHeight()/2;
         g.drawIcon(xpos, ypos, *icon);
      }
      else
      {
         // The icons used to paint the toggle state of the item is
         // not available for some reason. Do a manual drawing then, 
         // in order to at least make the list-box usable to the user.
         int ypos = r.y + r.height/2 - tglIconHeight/2;
         g.setColor(frg);
         g.drawRectangle(xpos, ypos, tglIconWidth, tglIconHeight);
         if (selected)
         {
            g.drawLine(xpos, ypos, xpos+tglIconWidth-1, ypos+tglIconHeight-1);
            g.drawLine(xpos, ypos+tglIconHeight-1, xpos+tglIconWidth-1, ypos);
         }
      }
      r.nudgeX(tglIconWidth + 5);
   }

   // Draw the icon (if any).
   const GIcon* icon = item.getIcon();
   if (icon != null)
   {
      int iconWidth = icon->getWidth();
      int xpos = r.x + 1;
      int ypos = r.y + r.height/2 - icon->getHeight()/2;
      g.drawIcon(xpos, ypos, *icon);
      r.nudgeX(iconWidth + 5);
   }

   // Draw the item text.
   r.nudgeX(1);
   const GString& txt = item.getText();
   g.drawText(txt, r, frg);
}

bool GListBox::isEmpty () const
{
   int count = getItemCount();
   return count <= 0;
}

void GListBox::changeValue ( const GString& newValue, bool notify )
{
   enterCtrlChangedFilter();
   int index;
   try {
      index = GInteger::ParseInt(newValue);
   } catch (GNumberFormatException& /*e*/) {
      index = -1;
   }
   setSelectedIndex(index);
   exitCtrlChangedFilter();
   if (notify)
      fireValueChangeListeners();
}

GString GListBox::queryValue () const
{
   int index = getSelectedIndex();
   return GInteger::ToString(index);
}
