/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GLinePlotter.h"
#include "glib/gui/GGraphics.h"
#include "glib/util/GMath.h"

GLinePlotter::Pt::Pt () 
                 :val(0.0), 
                  ypos(0) 
{
}

GLinePlotter::Pt::~Pt ()
{
}

void GLinePlotter::Pt::recalcYPos ( int graphY, int graphH, double peak ) 
{ 
   ypos = (int) (graphY + graphH * (val / peak)); 
}

GLinePlotter::GLinePlotter ( const GString& name, 
                             const GString& constraints, 
                             GWindow& parentWin, 
                             long winStyle, 
                             long winStyle2 )
             :GWindow(name, 
                      constraints, 
                      &parentWin, 
                      &parentWin, 
                      winStyle, 
                      winStyle2 | WS2_OS2Y),
              w(0),
              h(0),
              graphX(23),
              graphW(0),
              cur(0.0),
              peak(1.0),
              columnInc(0),
              ptNum(0),
              graphColor(46, 139, 87),
              mfColor(0, 100, 0)
{
   setForegroundColor(GColor::GREEN);
   setBackgroundColor(GColor::BLACK);
}

GLinePlotter::~GLinePlotter ()
{
}

aptr< GArray<GLinePlotter::Pt> > GLinePlotter::makePts ( const int num )
{
   aptr< GArray<GLinePlotter::Pt> > ret(new GArray<GLinePlotter::Pt>(num));
   for (int i=0; i<num; i++)
      ret->add(new GLinePlotter::Pt());
   return ret;
}

void GLinePlotter::clearAllSamples ()
{
   h = 0;
   w = 0;
   cur = 0.0;
   peak = 1.0;
   columnInc = 0;
   ptNum = 0;
   if (pts.get() != null)
      pts->removeAll();
}

void GLinePlotter::addSample ( double sample )
{
   cur = sample;

   GDimension d = getWindowSize();
   if (d.width <= 0)
      d.width = 1;
   if (d.height <= 0)
      d.height = 1;

   bool recalcAll = (d.height != h);
   if (d.width != w || recalcAll)
   {
      w = d.width;
      h = d.height;
      graphW = w - graphX - 5;
      if (graphW <= 0)
         return; // Probably because we are still not visible on screen

      if (pts.get() == null)
      {
         pts = makePts(graphW);
         ptNum = 0;
      }
      else
      if (pts->getCount() != graphW)
      {
         aptr< GArray<GLinePlotter::Pt> > old = pts;
         pts = makePts(graphW);
         const int num = GMath::Min(ptNum, GMath::Min(old->getCount(), graphW));
         for (int i=0; i<num; i++)
            pts->add(new GLinePlotter::Pt(old->get(i)));
      }
   }
   else
   {
      if (pts.get() == null)
      {
         pts = makePts(graphW);
         ptNum = 0;
      }
   }

   double ssH = getFontHeight();
   double remainingHeight = double(h - (ssH*2) - 0.5);
   int graphY = (int) ssH;
   int graphH = (int) remainingHeight;

   const int numSamples = pts->getCount();

   if (cur > peak)
   {
      peak = cur;
      recalcAll = true;
   }

   if (recalcAll)
   {
      for (int i=0; i<numSamples; i++)
         pts->get(i).recalcYPos(graphY, graphH, peak);
   }

   const int ptsLength = pts->getAllocatedSize();
   if (ptNum >= ptsLength)
   {
      // Throw out oldest point.
      for (int i=1; i<ptsLength; i++)
      {
         pts->get(i-1).val = pts->get(i).val;
         pts->get(i-1).ypos = pts->get(i).ypos;
      }
      ptNum = numSamples - 1;
   }

   if (ptNum >= 0)
   {
      pts->get(ptNum).val = cur;
      pts->get(ptNum).recalcYPos(graphY, graphH, peak);
   }

   ptNum++;

   // ---
   invalidateAll(false);
}

bool GLinePlotter::onPaint ( GGraphics& g, const GRectangle& /*rect*/ )
{
   GRectangle rect = getWindowRect();
   int w = rect.width;
   int h = rect.height;

   GColor fg = getForegroundColor();
   GColor bg = getBackgroundColor();

   // Clear the window, to make up the background.
   g.drawFilledRectangle(rect, bg);

   // .. Draw allocated and used strings ..
   const int border = 4;

   // ---
   int graphH = h - (border*2);
   int blockHeight = graphH/10;
   int blockHeightRest = graphH%10;
   const int blockWidth = 15;

   // Draw the vertical bar.
   const int usage = int(cur / (peak / 10.0));
   for (int i=0, ypos=border; i<10; i++, ypos+=blockHeight)
   {
      int addh = i<blockHeightRest ? 1 : 0;
      GRectangle r(border, ypos, blockWidth-1, blockHeight+addh-1);
      g.drawFilledRectangle(r, i<=usage ? fg : mfColor);
      ypos += addh;
   }

   // Draw the thin rectangle around the whole graph area.
   int graphY = border;
   int graphW = w - graphX - 5;
   g.setColor(graphColor);
   g.drawRectangle(graphX, graphY, graphW, graphH);

   // Draw the row lines to form the graph cells.
   double graphRow = graphH/10;
   for (double j=graphY; j<=graphH+graphY; j+=graphRow)
      g.drawLine(graphX, int(j), graphX+graphW-1, int(j));

   // Draw animated column movement.
   double graphColumn = graphW/15;
   if (columnInc == 0)
      columnInc = int(graphColumn);
   for (double j=graphX+columnInc; j<graphW+graphX; j+=graphColumn)
      g.drawLine(int(j), graphY, int(j), graphY+graphH-1);

   columnInc--;

   // Finally, draw the line it self.
   if (pts.get() != null)
   {
      g.setColor(GColor::YELLOW);
      int k = 1;
      for (int j=graphX+graphW-ptNum+1; k<ptNum; k++, j++)
         g.drawLine(j-1, pts->get(k-1).ypos, j, pts->get(k).ypos);
   }

   return true;
}

