/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GGroupBox.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GGraphics.h"
#include "glib/util/GMath.h"

GGroupBox::GGroupBox ( const GString& name,
                       const GString& constraints,
                       GWindow& parentWin,
                       long winStyle,
                       long winStyle2,
                       bool hotKey,
                       bool noLeft,
                       bool noRight,
                       bool noTop,
                       bool noBottom  )
          :GWindow(name,
                   constraints,
                   &parentWin,
                   &parentWin,
                   winStyle,
                   winStyle2 | WS2_OS2Y),
           useHotKey(hotKey),
           hotKeyPos(-1),
           noLeft(noLeft),
           noRight(noRight),
           noTop(noTop),
           noBottom(noBottom)
{
   setForegroundColor(GColor::DBLUE);
   // Background: Use same color as the parent window.
}

GGroupBox::~GGroupBox ()
{
}

bool GGroupBox::isEmpty () const
{
   return text == GString::Empty;
}

void GGroupBox::changeValue ( const GString& newValue, bool /*notify*/ )
{
   setText(newValue);
}

GString GGroupBox::queryValue () const
{
   return text;
}

GString GGroupBox::getText () const
{
   return text;
}

void GGroupBox::setText ( const GString& txt_ )
{
   if (text == txt_)
      return;

   GWindow::setText(txt_);
   text = GWindow::getText();
   if (useHotKey)
      hotKeyPos = autoAssocAHotKey(text);
   invalidateAll(true);
}

bool GGroupBox::onPaint ( GGraphics& g, const GRectangle& rect )
{
   GColor bck = getBackgroundColor();
   g.drawFilledRectangle(rect, bck);

   int fontW = g.getFontWidth();
   int fontH = g.getFontHeight() + GDialogPanel::COMPONENT_EXTRAFONTH;

   GDimension dm = getWindowSize();

   int pixXLeft = fontW / 2;
   int pixYBottom = (fontH / 2) + (GDialogPanel::COMPONENT_EXTRAFONTH / 2);
   int pixXRight = pixXLeft + dm.width - 1 - fontW;
   int pixYTop = pixYBottom + dm.height - 1 - fontH - (GDialogPanel::COMPONENT_EXTRAFONTH / 2);

   int textYPos = pixYTop - (fontH / 2) + (GDialogPanel::COMPONENT_EXTRAFONTH / 2) + 1;
   int textXPos = fontW + (fontW / 2);
   int textWidth = 0;

   // Paint the text string.
   GString textToDraw(text.length() + 2);
   if (text != "")
   {
      int hkpos = hotKeyPos;
      if (hkpos >= 0)
         hkpos += 1; // Count for the inserted prefix space character.
      textToDraw += ' ';
      textToDraw += text;
      textToDraw += ' ';
      textWidth = getWidthOfString(textToDraw);
      if (isEnabled())
      {
         g.setColor(getForegroundColor());
         g.drawTextMnemonic(textXPos, textYPos, textToDraw, hkpos);
      }
      else
      {
         g.setColor(GColor::WHITE);
         g.drawTextMnemonic(textXPos+1, textYPos+1, textToDraw, hkpos);
         g.setColor(GColor::DGRAY);
         g.drawTextMnemonic(textXPos+0, textYPos+0, textToDraw, hkpos);
      }
   }

   // Paint the 3D-border, except on the Text String area.
   g.setColor(bck.getDarker());
   if (!noLeft)
      g.drawLine(pixXLeft, pixYBottom, pixXLeft, pixYTop+1);
   if (!noTop)
   {
      if (text == "")
      {
         g.drawLine(pixXLeft, pixYTop+1, pixXRight, pixYTop+1);
      }
      else
      {
         g.drawLine(pixXLeft, pixYTop+1, textXPos-1, pixYTop+1);
         int textx = GMath::Min(textXPos+textWidth+1, pixXRight - (fontW/2));
         g.drawLine(textx, pixYTop+1, pixXRight, pixYTop+1);
      }
   }
   if (!noRight)
      g.drawLine(pixXRight-1, pixYTop, pixXRight-1, pixYBottom-1);
   if (!noBottom)
      g.drawLine(pixXRight-1, pixYBottom, pixXLeft, pixYBottom);

   g.setColor(bck.getLighter());
   if (!noLeft)
      g.drawLine(pixXLeft+1, pixYBottom+1, pixXLeft+1, pixYTop);
   if (!noTop)
   {
      int rightx = pixXRight-1;
      if (text == "")
      {
         g.drawLine(pixXLeft+1, pixYTop, rightx, pixYTop);
      }
      else
      {
         g.drawLine(pixXLeft+1, pixYTop, textXPos-1, pixYTop);
         int textx = GMath::Min(textXPos+textWidth+1, rightx - (fontW/2));
         g.drawLine(textx, pixYTop, rightx, pixYTop);
      }
   }
   if (!noRight)
      g.drawLine(pixXRight, pixYTop, pixXRight, pixYBottom-1);
   if (!noBottom)
      g.drawLine(pixXRight, pixYBottom-1, pixXLeft+1, pixYBottom-1);

   return true;
}

