/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_GRAPHICS
#define __GLIB_GRAPHICS

#include "glib/gui/GColor.h"
#include "glib/util/GArray.h"
#include "glib/util/GKeyBag.h"
#include "glib/util/GSystemYConverter.h"
#include "glib/primitives/GPoint.h"
#include "glib/primitives/GDimension.h"

/**
 * Class used to represent some drawing area, typically on a screen 
 * window, a memory bitmap or on a printer. In most cases an instance of 
 * this class can be seen as a wrapper for a system dependent painting
 * handle, sunch as a "Graphics Device" (HDC) on Windows or 
 * a "Presentation Space" (HPS) on OS/2 Warp.
 *
 * Only RGB painting mode is supported by this class, 
 * to keep things simple.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GGraphics : public GObject, public GSystemYConverter
{
   friend class GIcon;

   public:

      enum HAlign {
         LEFT = 0,
         RIGHT,
         HCENTER
      };

      enum VAlign {
         TOP = 0,
         BOTTOM,
         VCENTER
      };

      enum LineType {
         LTSolid = 0,
         LTDotted,
         LTDashDotted
      };

      typedef HPS Handle;

      /** 
       * Special color instance used to identify colors that should be 
       * ugnored. That is; specifying this color is treated as if the 
       * color parameter is not specified at all, and the already active 
       * current color will be used in the draw operation, instead of 
       * the dummy color specified.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.24
       */
      static const GColor DUMMY_COLOR;

      /** 
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       */
      class Font : public GObject
      {
         friend class GGraphics;

         private:

            int referenceCount;

         public:

            const GString fontNameSize;

         private:

            Font ( const GString& fontNameSize );
            virtual ~Font ();
      };

      /** 
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       */
      class Brush : public GObject
      {
         friend class GGraphics;

         private:

            int referenceCount;
            const GString name;

         public:

            const GColor color;

         private:

            Brush ( const GColor& color );
            virtual ~Brush ();
      };

   private:

      /** The system dependent Presentation Space or Device Context handle. */
      Handle handle;

      enum HandleReleaseType
      {
         HRTDontRelease,
         HRTRelease,
         HRTDestroy
      };

      /** 
       * True if we shall release the graphics handle upon destroy, 
       * or else false if we shall leave it without destroying it. 
       */
      HandleReleaseType releaseHandleOnDestroy;

      /** 
       * Reference to owner window of where the graphics object belongs. 
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.21
       */
      const class GWindow& ownerWin;

      /**
       * True if the {@link #ownerWin} variable references a "substitute".
       * That is; the owner window is not actually the owner of this 
       * graphics context object, but is provided just in order to be
       * able to get certain properties such as the width and the height 
       * of the drawing area, in a platform independent manner.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.21
       */
      bool winIsSubstitute;

      /** True if PS is associated with a bitmap in memory (off-screen). */
      bool isMemory;

      /** True if we uses OS/2'ish Y-coordinates. */
      bool os2y;

      /** Size of the drawing area represented by this graphics object. */
      GDimension size;

      /** This variable is used to cache some frequently used font information. */
      mutable class GFontMetrics* fm;

      /** The current line type. */
      LineType lineType;

      /** Device context. Can be null. */
      HDC hDC;

      /** The common font pool of ours. */
      static GKeyBag<GGraphics::Font> FontPool;

      /** The common brush pool of ours. */
      static GKeyBag<GGraphics::Brush> BrushPool;

   public:

      /**
       * Create a graphics object that will paint on the specified window.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.09
       * @param   win  The window for which the new 
       *               graphics object is associated.
       */
      explicit GGraphics ( const class GWindow& win );

      /**
       * Create a graphics object that will paint on the specified window.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.09
       * @param   win  The window for which the new 
       *               graphics object is associated.
       */
      explicit GGraphics ( const class GWindow* win );

      /**
       * Create a graphics object that will paint on the specified bitmap.
       *
       * The new graphics object will be compatible with the specified
       * Handle, or the screen in case the specified Handle is null.
       */
      GGraphics ( const class GBitmap& bm, 
                  const class GWindow& win, 
                  Handle hcomp = null );

      /**
       * Create a graphics object that will wrap and paint on the
       * specified system dependent graphics handle.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.10
       * @param   handle  The system dependent graphics handle of 
       *                  where to paint.
       * @param   win     The window of where the handle is associated.
       *                  Can be another window (e.g. the parent) than 
       *                  the one actually used to create the handle.
       * @param   winIsSubstitute True if the specified window is a 
       *                  "substitute", that is if the window specified 
       *                  is not the actual window on which the specified 
       *                  handle will paint. In that case we will not 
       *                  initialize the graphics object with the current 
       *                  font of the specified window, but use the already 
       *                  active font of the handle "as is" instead. There 
       *                  might also be some other properties that we will 
       *                  leave on the specified handle "as is" if this 
       *                  parameter is true.
       * @param   isScreen True if the specified handle is created for 
       *                  drawing on the "whole screen". This should 
       *                  probably never be true if <i>winIsSubstitute</i>
       *                  is false.
       */
      GGraphics ( Handle handle,
                  class GWindow& win, 
                  bool winIsSubstitute,
                  bool isScreen );

      /**
       * Destroy the wrapped Presentations Space Handle if needed.
       */
      virtual ~GGraphics ();

   private:

      /**
       * Common construction code.
       */
      void init ();

      /** 
       * Initialize the size of our drawing area.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.10
       */
      void initDrawingAreaSize ( bool isScreen );

   public:

      /**
       * Return true if this graphics object uses OS/2'ish y-coordinates.
       * Part of {@link GSystemYConverter}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.03
       * @see     #setOS2Y
       */
      virtual bool isOS2Y () const;

      /**
       * Get the system dependent Graphics Handle.
       * This is a HDC on Windows, or a HPS on OS/2 Warp.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.14
       */
      Handle getHandle () const;

      /** 
       * Check the specified font into the common font pool, and decrement 
       * the reference count of the font. If the reference count is back 
       * to zero we might possibly delete the font object to free system
       * resources that is associated with the font.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @return  Always returns null. 
       * @see     #CheckOutFont
       */
      static Font* CheckInFont ( Font* font );

      /** 
       * Check the specified font out of the common font pool, and increment
       * the reference count of the font. If the font is not already in the 
       * pool we will create it and add it to the pool automatically.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @return  A pointer to the checked out font. We keeps the ownership.
       * @see     #CheckInFont
       */
      static Font* CheckOutFont ( const GString& fontNameSize );

      /** 
       * Check the specified brush into the common brush pool, and decrement 
       * the reference count of the brush. If the reference count is back 
       * to zero we might possibly delete the brush object to free system
       * resources that is associated with the brush.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @return  Always returns null. 
       * @see     #CheckOutBrush
       */
      static Brush* CheckInBrush ( Brush* brush );

      /** 
       * Check the specified brush out of the common brush pool, and increment
       * the reference count of the brush. If the brush is not already in the 
       * pool we will create it and add it to the pool automatically.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.17
       * @return  A pointer to the checked out brush. We keeps the ownership.
       * @see     #CheckInBrush
       */
      static Brush* CheckOutBrush ( const GColor& color );

      /**
       * Draw the specified bitmap using the current mix mode.
       */
      void drawBitmap ( int xpos,
                        int ypos,
                        class GBitmap& bm,
                        const GColor& frgColor = GColor::BLACK,
                        const GColor& bckColor = GColor::BLACK );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      void drawEmphasiseBox ( const class GRectangle& rect,
                              const GColor& color,
                              int width,
                              bool invert );

      /**
       * Draw the specified icon resource on this graphics object,
       * at the specified position.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.07
       * @param   xpos      Left edge of the icon.
       * @param   ypos      Bottom edge of the icon.
       * @param   icon      The icon of which to draw.
       * @param   smallIcon Some icons actually contains several alternative 
       *                    bitmaps in various sizes. For such icons, this 
       *                    parameters should be true if the calling code 
       *                    prefers a small size icon to be painted. Else 
       *                    this parameter should be false.
       */
      void drawIcon ( int xpos,
                      int ypos,
                      const class GIcon& icon,
                      bool smallIcon = false ) const;

      void drawLine ( int xpos,
                      int ypos,
                      int endXPos,
                      int endYPos );

      void drawLineTo ( int xpos,
                        int ypos );

      void drawFilledRectangle ( int xpos,
                                 int ypos,
                                 int width,
                                 int height,
                                 const GColor& color = DUMMY_COLOR );

      void drawFilledRectangle ( const class GRectangle& rect, 
                                 const GColor& color = DUMMY_COLOR );

      void drawRectangle ( int xpos,
                           int ypos,
                           int width,
                           int height );

      void drawRectangle ( const class GRectangle& rect );

      void drawFilledPolygon ( const GArray<GPoint>& polygon );

      /**
       * Draw a single pixel at the specified position using the
       * current color.
       */
      void drawPixel ( int xpos, int ypos );

      /**
       * Draw the specified text at the specified position using the
       * current color, without clearing the background. The position is
       * the lower left corner of the text if {@link #isOS2Y} is true,
       * else the position is the upper left corner.
       */
      void drawText ( int xpos,
                      int ypos,
                      const GString& text );
      void drawText ( const GString& text,
                      const class GRectangle& clipRect,
                      const GColor& frgColor = DUMMY_COLOR,
                      const GColor& bckColor = DUMMY_COLOR,
                      GGraphics::HAlign halign = LEFT,
                      GGraphics::VAlign valign = VCENTER );

      /**
       * Draw the specified text at the specified position using the
       * current color, without clearing the background. The position is
       * the lower left corner of the text.
       *
       * If the specified mnemonic position is not negative then we will
       * also paint an underscore under the indexed character to
       * form a visual mnemonic character on screen.
       */
      void drawTextMnemonic ( int xpos,
                              int ypos,
                              const GString& text,
                              int mnemonicIndex );

      /**
       * Add the specified rectangle are to the clipping region of the 
       * graphics drawing area. The specified rectangle will be added 
       * to the existing clipping region (if any). Any attempt to paint
       * on the clipping region of the graphics object will not affect 
       * the screen or graphics output device.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      void excludeClipRectangle ( const class GRectangle& rect );

      GColor getColor () const;

      /**
       * Get a reference to the font metrics for this graphics object.
       */
      const class GFontMetrics& getFontMetrics () const;

      /**
       * Get the height of the current font, in pixels.
       */
      int getFontHeight () const;

      /**
       * Get the average character width of the current font, in pixels.
       */
      int getFontWidth () const;

      /**
       * Shorten the specified text by replacing a number of its characters 
       * by the specified dots-string.
       *
       * If the maximum width is to narrow for any text to fit we will
       * just return a copy of {@link GString#ThreeDots} directly.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.04
       * @param   txt  The text of which to shorten.
       * @param   maxPixelWidth The pixel width of which area the shortened
       *               text is to fit.
       * @param   cutTail True if we shall cut the tail only, or else we 
       *               will cut the middle part of the text.
       * @param   dots The string to insert where we cut characters.
       */
      GString getShortenedText ( const GString& txt, 
                                 int maxPixelWidth,
                                 bool cutTail = true,
                                 const GString& dots = GString::ThreeDots );

      /**
       * Get the size of the graphics object draw area. 
       *
       * @author  Leif Erik Larsen
       * @since   2003.09.29
       */
      const GDimension& getSize () const;

      /**
       * Calculate how many vertical pixels the specified string need to be
       * fully visible when painted on the graphics object, using the
       * current font and color.
       */
      int getHeightOfString ( const GString& str ) const;

      GDimension getTextDim ( const GString& text ) const;

      /**
       * Calculate how many horizontal pixels the specified string need to be
       * fully visible when painted on the graphics object, using the
       * current font and color.
       */
      int getWidthOfString ( const GString& str ) const;

      /**
       * Word wrap the specified text so that none of the resulting lines
       * do exceed the specified display width with respect to the
       * current font of the graphics object graphics.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.14
       * @param   text      The text of which to word wrap.
       * @param   maxWidth  The width (in pixels) of the available area of where
       *                    which the wrapped text should not exceed.
       * @param   strings   The array of where to return the text after it has
       *                    been split into word wrapped lines.
       * @return  A reference to the <i>strings</i> parameter.
       */
      GArray<GString>& getWordWrappedText ( const GString& text,
                                            int maxWidth,
                                            GArray<GString>& strings ) const;

      void setColor ( const GColor& color );

      /**
       * Set and activate the specified fontNameSize on this graphics 
       * context. This method is usually not called directly by client 
       * code, because the graphics object constructor automatically 
       * selects the fontNameSize of the owner window. But the font 
       * can be changed by application code, by using this method.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.21
       */
      void setFontNameSize ( const GString& fontNameSize );

      void setLineType ( LineType lt );

      /**
       * Set whether or not this graphics object should paint using 
       * OS/2'ish y-corrdinates. The default is that the graphics object 
       * inherits the OS2Y mode of the window specified to the constructor,
       * but this can be changed by calling this method.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.11
       * @see     #isOS2Y
       */
      void setOS2Y ( bool b );

      void setPattern ( int patternID );

      void setPosition ( int xpos, int ypos );
      void setPosition ( const GPoint& pt );

      /** Convert to system dependent Y-coordinate, if needed. */
      virtual int sysY ( int y ) const;
};

#endif
