/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <limits.h>
#include "glib/gui/GGraphics.h"
#include "glib/gui/GWindow.h"
#include "glib/gui/GBitmap.h"
#include "glib/gui/GFontMetrics.h"
#include "glib/gui/GIcon.h"
#include "glib/sys/GSystem.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/util/GLog.h"

const GColor GGraphics::DUMMY_COLOR;
GKeyBag<GGraphics::Font> GGraphics::FontPool;
GKeyBag<GGraphics::Brush> GGraphics::BrushPool;

GGraphics::Font::Font ( const GString& fontNameSize )
                :referenceCount(0),
                 fontNameSize(fontNameSize)
{
}

GGraphics::Font::~Font ()
{
}

GGraphics::Brush::Brush ( const GColor& color )
                 :referenceCount(0),
                  name(color.toString()),
                  color(color)
{
}

GGraphics::Brush::~Brush ()
{
}

GGraphics::GGraphics ( const GWindow& win )
          :ownerWin(win),
           winIsSubstitute(false)
{
   init();
}

GGraphics::GGraphics ( const GWindow* win )
          :ownerWin(*win),
           winIsSubstitute(false)
{
   init();
}

void GGraphics::init ()
{
   isMemory = false;
   os2y = ownerWin.isOS2Y();
   fm = null;
   lineType = LTSolid;
   initDrawingAreaSize(false);
   releaseHandleOnDestroy = HRTRelease;
   hDC = null;
   handle = ::WinGetPS(ownerWin.getHWND());
   ::GpiCreateLogColorTable(handle, 0, LCOLF_RGB, 0, 0, null);
   setColor(GColor::BLACK);
}

GGraphics::GGraphics ( Handle handle, 
                       GWindow& win, 
                       bool winIsSubstitute,
                       bool isScreen )
           :handle(handle),
            ownerWin(win),
            releaseHandleOnDestroy(HRTDontRelease),
            winIsSubstitute(winIsSubstitute),
            isMemory(false)
{
   os2y = win.isOS2Y();
   fm = null;
   hDC = null;
   ::GpiCreateLogColorTable(handle, 0, LCOLF_RGB, 0, 0, null);
   initDrawingAreaSize(isScreen);
   setColor(GColor::BLACK);
}

GGraphics::GGraphics ( const GBitmap& bm,
                       const GWindow& win,
                       Handle hcomp )
          :ownerWin(win),
           winIsSubstitute(false),
           isMemory(true),
           os2y(win.isOS2Y()),
           fm(null),
           releaseHandleOnDestroy(HRTDestroy),
           size(bm.getSize())
{
   HDC hdcComp = hcomp ? ::GpiQueryDevice(hcomp) : null;
   hDC = ::DevOpenDC(GProgram::hAB, OD_MEMORY, "*", 0, null, hdcComp);
   SIZEL sizlPage = {0, 0}; // Use the same page size as device
   handle = ::GpiCreatePS(GProgram::hAB, hDC, &sizlPage, GPIF_DEFAULT | PU_PELS | GPIT_MICRO | GPIA_ASSOC);
   ::GpiCreateLogColorTable(handle, 0, LCOLF_RGB, 0, 0, null);
   ::GpiSetBitmap(handle, bm.getHBM()); // Make this HPS draw into the specified bitmap
   setColor(GColor::BLACK);
}

GGraphics::~GGraphics ()
{
   delete fm;

   if (handle != null && isMemory)
      ::GpiSetBitmap(handle, null);

   if (handle != null && releaseHandleOnDestroy != HRTDontRelease)
   {
      switch (releaseHandleOnDestroy)
      {
         case HRTRelease:
            ::WinReleasePS(handle);
            break;

         case HRTDestroy:
            ::GpiDestroyPS(handle);
            break;

         default:
            GLog::Log(this, "Error: Unknown HandleReleaseType");
            break;
      }
   }

   if (hDC != null)
      ::DevCloseDC(hDC);
}

void GGraphics::initDrawingAreaSize ( bool isScreen )
{
   if (isScreen)
   {
      size = GSystem::GetScreenSize();
      return;
   }
   size = ownerWin.getWindowSize();
}

GGraphics::Handle GGraphics::getHandle () const 
{ 
   return handle; 
}

bool GGraphics::isOS2Y () const 
{ 
   return os2y; 
}

void GGraphics::setOS2Y ( bool b )
{
   this->os2y = b;
}

int GGraphics::sysY ( int y ) const
{
   if (os2y)
      return y;
   else
      return size.height - y - 1;
}

GGraphics::Font* GGraphics::CheckOutFont ( const GString& fontNameSize )
{
   Font* f = FontPool.get(fontNameSize);
   if (f == null)
   {
      f = new Font(fontNameSize);
      FontPool.put(fontNameSize, f);
   }
   f->referenceCount++;
   return f;
}

GGraphics::Font* GGraphics::CheckInFont ( Font* font )
{
   if (font != null)
   {
      if (font->referenceCount <= 0)
         gthrow_(GIllegalStateException("font->referenceCount <= 0"));
      font->referenceCount--;
      if (font->referenceCount == 0)
      {
         /* TODO: Don't delete the font. Keep it cached in the pool. 
         const GString& fontNameSize = font->getName();
         FontPool.remove(fontNameSize);
         */
      }
   }
   return null;
}

GGraphics::Brush* GGraphics::CheckOutBrush ( const GColor& color )
{
   GString name = color.toString();
   Brush* b = BrushPool.get(name);
   if (b == null)
   {
      b = new Brush(color);
      BrushPool.put(name, b);
   }
   b->referenceCount++;
   return b;
}

GGraphics::Brush* GGraphics::CheckInBrush ( Brush* brush )
{
   if (brush != null)
   {
      if (brush->referenceCount <= 0)
         gthrow_(GIllegalStateException("referenceCount <= 0"));
      brush->referenceCount--;
      if (brush->referenceCount == 0)
      {
         /* TODO: Don't delete the brush. Keep it cached in the pool.
         const GString& name = brush->getName();
         BrushPool.remove(name);
         */
      }
   }
   return null;
}

void GGraphics::drawPixel ( int xpos, int ypos )
{
   POINTL pt = { xpos, sysY(ypos) };
   ::GpiSetPel(handle, &pt);
}

void GGraphics::drawBitmap ( int xpos,
                             int ypos,
                             GBitmap& bm,
                             const GColor& frgColor,
                             const GColor& bckColor )
{
   POINTL pt = { xpos, sysY(ypos) };
   HBITMAP hbm = bm.getHBM();
   ::WinDrawBitmap(handle, hbm, null, &pt, frgColor, bckColor, DBM_NORMAL);
}

void GGraphics::drawIcon ( int xpos,
                           int ypos,
                           const GIcon& icon,
                           bool smallIcon ) const
{
   // Icons can be in two forms:
   //
   // A) Two bitmaps that originates from an icon resource in the
   //    resource table of the application.
   //
   // B) A system dependent icon handle for an icon that originates
   //    from the shell of the underlying system (file icons, etc.).
   //
   // Each of these icon forms recuires their own way of painting...

   if (icon.a != null)
   {
      // Paint the icon in its A-form.
      int width = icon.getWidth();
      int height = icon.getHeight();
      POINTL aptlPoints[3];
      if (isOS2Y())
      {
         aptlPoints[0].x = xpos; // Target bottom left corner, X.
         aptlPoints[0].y = ypos; // Ditto, Y.
         aptlPoints[1].x = xpos + width - 1;  // Target top right corner, X.
         aptlPoints[1].y = ypos + height - 1; // Ditto, Y.
         aptlPoints[2].x = 0; // Source bottom left corner, X.
         aptlPoints[2].y = 0; // Ditto, Y.
      }
      else
      {
         aptlPoints[0].x = xpos;
         aptlPoints[0].y = sysY(ypos) - height + 1;
         aptlPoints[1].x = xpos + width - 1;
         aptlPoints[1].y = sysY(ypos);
         aptlPoints[2].x = 0;
         aptlPoints[2].y = 0;
      }
      ::GpiWCBitBlt(handle, icon.a->bmMask->getHBM(), 3, aptlPoints, ROP_SRCAND, BBO_IGNORE);
      ::GpiWCBitBlt(handle, icon.a->bmImage->getHBM(), 3, aptlPoints, ROP_SRCPAINT, BBO_IGNORE);
   }
   else
   if (icon.b != null)
   {
      // Paint the icon in its B-form.
      HPOINTER hicon = icon.b->hicon;
      int options = (smallIcon ? DP_MINIICON : DP_NORMAL);
      int y = sysY(ypos);
      ::WinDrawPointer(handle, xpos, ypos, hicon, options);
   }
}

void GGraphics::setFontNameSize ( const GString& fontNameSize )
{
   // TODO: ???
}

GDimension GGraphics::getTextDim ( const GString& text ) const
{
   RECTL r = { 0, 0, INT_MAX, INT_MAX };
   char *str = (char *) text.cstring();
   ::WinDrawText(handle, text.length(), str, &r, 0, 0, DT_QUERYEXTENT | DT_LEFT | DT_TOP | DT_TEXTATTRS);
   return GDimension(r.xRight - r.xLeft, r.yTop - r.yBottom);
}

int GGraphics::getWidthOfString ( const GString& str ) const
{
   GDimension dim = getTextDim(str);
   return dim.width;
}

int GGraphics::getHeightOfString ( const GString& str ) const
{
   GDimension dim = getTextDim(str);
   return dim.height;
}

int GGraphics::getFontHeight () const 
{ 
   const GFontMetrics& fm_ = getFontMetrics();
   return fm_.getHeight(); 
}

int GGraphics::getFontWidth () const 
{ 
   const GFontMetrics& fm_ = getFontMetrics();
   return fm_.getAverageWidth(); 
}

const GDimension& GGraphics::getSize () const 
{ 
   return size; 
}

void GGraphics::drawEmphasiseBox ( const GRectangle& rect,
                                   const GColor& color,
                                   int width,
                                   bool invert )
{
   RECTL r;
   r.xLeft = rect.x;
   r.xRight = rect.x + rect.width - 1;
   r.yTop = sysY(rect.y + rect.height - 1);
   r.yBottom = sysY(rect.y);
   int flags = (invert ? DB_DESTINVERT : 0);
   ::WinDrawBorder(handle, &r, width, width, color, 0, flags);
}

void GGraphics::drawLine ( int xpos, int ypos, int endXPos, int endYPos )
{
   if (xpos == endXPos && ypos == endYPos)
      return;

   // OS/2 draws "inclusive" end-point, but we should not (like Windows).
   if (endXPos > xpos)
      endXPos -= 1; 
   else
   if (endXPos < xpos)
      endXPos += 1;

   // ---
   if (endYPos > ypos)
      endYPos -= 1; 
   else
   if (endYPos < ypos)
      endYPos += 1; 

   setPosition(xpos, ypos);
   drawLineTo(endXPos, endYPos);
}

void GGraphics::drawLineTo ( int xpos, int ypos )
{
   POINTL pt = { xpos, sysY(ypos) };
   ::GpiLine(handle, &pt);
}

void GGraphics::drawFilledRectangle ( int xpos,
                                      int ypos,
                                      int width,
                                      int height,
                                      const GColor& color )
{
   GRectangle rect(xpos, ypos, width, height);
   drawFilledRectangle(rect, color);
}

void GGraphics::drawFilledRectangle ( const GRectangle& rect,
                                      const GColor& color )
{
   setColor(color);
   setPosition(rect.x, rect.y);
   POINTL pt = { rect.x + rect.width - 1, sysY(rect.y + rect.height - 1) };
   ::GpiBox(handle, DRO_OUTLINEFILL, &pt, 0, 0);
}

void GGraphics::drawRectangle ( int xpos, int ypos, int width, int height )
{
   setPosition(xpos, ypos); // Bottom left.
   drawLineTo(xpos + width - 1, ypos);
   drawLineTo(xpos + width - 1, ypos + height - 1);
   drawLineTo(xpos, ypos + height - 1);
   drawLineTo(xpos, ypos);
}

void GGraphics::drawRectangle ( const GRectangle& rect )
{
   drawRectangle(rect.x, rect.y, rect.width, rect.height);
}

void GGraphics::drawText ( const GString& text,
                           const GRectangle& clipRect,
                           const GColor& frgColor,
                           const GColor& bckColor,
                           HAlign halign,
                           VAlign valign )
{
   int osflags = 0;
   switch (halign)
   {
      case LEFT:
         osflags |= DT_LEFT;
         break;
      case RIGHT:
         osflags |= DT_RIGHT;
         break;
      default:
         osflags |= DT_CENTER;
         break;
   }
   switch (valign)
   {
      case TOP:
         osflags |= DT_TOP;
         break;
      case BOTTOM:
         osflags |= DT_BOTTOM;
         break;
      default:
         osflags |= DT_VCENTER;
         break;
   }
   int frg = (&frgColor == &DUMMY_COLOR ? getColor() : frgColor);
   int bck = 0;
   if (&bckColor != &DUMMY_COLOR)
   {
      bck = bckColor;
      osflags |= DT_ERASERECT;
   }
   int len = text.length();
   const char* str = text.cstring();
   RECTL r = GWindow::MakeSysRect(clipRect, *this, false);
   ::WinDrawText(handle, len, (char*) str, &r, frg, bck, osflags);
}

void GGraphics::drawText ( int xpos,
                           int ypos,
                           const GString& text )
{
   GRectangle r;
   r.x = xpos;
   r.width = 32000;
   r.height = getFontHeight();
   r.y = ypos;
   drawText(text, r);
}

void GGraphics::drawTextMnemonic ( int xpos,
                                   int ypos,
                                   const GString& text,
                                   int mnemonicIndex )
{
   // Draw the text.
   drawText(xpos, ypos, text);

   // Draw the mnemonic underscore if requested.
   if (mnemonicIndex >= 0 && mnemonicIndex < text.length())
   {
      GString hotChar(1);
      hotChar += text[mnemonicIndex];
      int mnemonicWidth = getWidthOfString(hotChar);
      if (mnemonicWidth > 0)
      {
         int mnemonicXPos = xpos;
         int mnemonicYPos = ypos + 1;
         if (mnemonicIndex > 0)
         {
            GString prefix = text.substring(0, mnemonicIndex);
            int prefixWidth = getWidthOfString(prefix);
            mnemonicXPos += prefixWidth;
         }
         int endx = mnemonicXPos + mnemonicWidth;
         if (mnemonicWidth <= 2)
            endx += 1;
         drawLine(mnemonicXPos, mnemonicYPos, endx, mnemonicYPos);
      }
   }
}

void GGraphics::drawFilledPolygon ( const GArray<GPoint>& polygon )
{
   const int count = polygon.getCount();
   if (count < 3)
      return;

   setPosition(polygon[0]);
   POINTL* points = new POINTL[count];
   for (int i=0; i<count-1; i++)
   {
      const GPoint& pt = polygon[i+1];
      points[i].x = pt.x;
      points[i].y = sysY(pt.y);
   }
   POLYGON poly;
   poly.ulPoints = count-1;
   poly.aPointl = points;
   GpiPolygons(handle, 1, &poly, POLYGON_BOUNDARY, POLYGON_INCL);
   delete [] points;
}

GColor GGraphics::getColor () const
{
   int c = GpiQueryColor(handle);
   return GColor(c);
}

void GGraphics::excludeClipRectangle ( const class GRectangle& rect )
{
   GWindow::SysRect sr = GWindow::MakeSysRect(rect, *this, true);
   ::GpiExcludeClipRectangle(handle, &sr);
}

const GFontMetrics& GGraphics::getFontMetrics () const
{
   if (fm == null)
      fm = new GFontMetrics(*this);
   return *fm;
}

void GGraphics::setColor ( const GColor& color )
{
   if (&color == &DUMMY_COLOR)
      return;

   int rgb = color.getRGB();
   ::GpiSetColor(handle, rgb);
}

void GGraphics::setPosition ( int xpos, int ypos )
{
   POINTL pt = { xpos, sysY(ypos) };
   GpiMove(handle, &pt);
}

void GGraphics::setPosition ( const GPoint& pt ) 
{ 
   setPosition(pt.x, pt.y); 
}

void GGraphics::setLineType ( LineType lt )
{
   if (lt == lineType)
      return;

   lineType = lt;
   int sysID;
   switch (lt)
   {
      case LTSolid: sysID = LINETYPE_SOLID; break;
      case LTDotted: sysID = LINETYPE_DOT; break;
      case LTDashDotted: sysID = LINETYPE_DASHDOT; break;
      default: sysID = LINETYPE_SOLID; break;
   }
   ::GpiSetLineType(handle, sysID);
}

void GGraphics::setPattern ( int patternID )
{
   GpiSetPattern(handle, patternID);
}

GString GGraphics::getShortenedText ( const GString& txt, 
                                      int maxPixelWidth,
                                      bool cutTail,
                                      const GString& dots )
{
   int txtLen = txt.length();
   GString ret(txtLen + 3);

   if (cutTail)
   {
      ret = txt;
      ret += dots;
      for (int pos=txtLen-1; pos>=0; pos--)
      {
         ret.removeCharAt(pos);
         if (getWidthOfString(ret) <= maxPixelWidth)
            return ret;
      }
   }
   else
   {
      int mid = txtLen / 2;
      GString prefix = txt.substring(0, mid);
      GString postfix = txt.substring(mid);
      for (;;)
      {
         prefix.removeLastChar();
         postfix.removeFirstChar();
         ret = prefix;
         ret += dots;
         ret += postfix;
         if (getWidthOfString(ret) <= maxPixelWidth)
            return ret;
         if (prefix == "" && postfix == "")
            break;
      }
   }

   return GString::ThreeDots;
}

GArray<GString>& GGraphics::getWordWrappedText ( const GString& text,
                                                 int maxWidth,
                                                 GArray<GString>& strings ) const
{
   strings.removeAll();

   int startPos = 0;
   int lastSpacePos = 0;
   bool newLine = false;
   const int textLength = text.length();
   int currentPos = 0;
   for (;;)
   {
      if (currentPos > textLength)
         break;

      char nextChar;
      if (currentPos == textLength)
         nextChar = '\n';
      else
         nextChar = text.charAt(currentPos);

      if (nextChar == '\n')
      {
         newLine = true;
      }
      else
      {
         GString testStr = text.substring(startPos, currentPos);
         testStr.trim(GString::TrimEndsOnly, GString::EOL);
         if (getWidthOfString(testStr) >= maxWidth)
         {
            newLine = true;
            if (lastSpacePos > 0)
               currentPos = lastSpacePos;
            else
               currentPos -= 1;
         }
      }

      if (newLine)
      {
         newLine = false;
         GString* nextLine = new GString(text.substring(startPos, currentPos));
         nextLine->trim(GString::TrimEndsOnly, GString::EOL);
         strings.add(nextLine);
         startPos = currentPos + 1;
         lastSpacePos = 0;
      }
      else
      if (nextChar == ' ' || nextChar == '\t')
         lastSpacePos = currentPos;

      currentPos++;
   }

   return strings;
}
