/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/GProgram.h"
#include "glib/gui/GDropList.h"
#include "glib/gui/GScrollbar.h"
#include "glib/gui/GFrameWindow.h"
#include "glib/gui/event/GCommandMap.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/util/GMath.h"
#include "glib/primitives/GBoolean.h"
#include "glib/sys/GSystem.h"

DEFINE_COMMAND_TABLE(GDropList);
   ADD_COMMAND("cmdDrop", cmdDrop);
END_COMMAND_TABLE;

GDropList::DropContainer::ListBox::ListBox ( GDropList& dlist, GWindow& parentWin )
          :GListBox(GString::Empty,
                    GBorderLayout::CENTER,
                    parentWin,
                    false),
           dlist(dlist)
{
   // We want mouse events from the listbox wrapper so that we get
   // a notification when user releases the mouse button after 
   // he or she has clicked on some item in the droplist. In that 
   // case we will hide the droplist window.
   GWindow& peer = getPeer(); 
   peer.addMouseListener(this);
}

GDropList::DropContainer::ListBox::~ListBox ()
{
}

bool GDropList::DropContainer::ListBox::mouseReleased ( const GMouseEvent& /*ev*/ )
{
   // User has released mouse button after he or she has clicked on
   // some droplist item. Hide the droplist window then.
   dlist.setDropListVisible(false);
   return true;
}

bool GDropList::DropContainer::ListBox::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_ALT_UP:
      case GKey::KEY_ALT_DOWN:
      case GKey::KEY_ESCAPE:
      case GKey::KEY_ENTER:
           dlist.setDropListVisible(false);
           dlist.grabFocus(true);
           return true;

      default:
           return GListBox::onKeyDown(key);
   }
}

GDropList::DropContainer::DropContainer ( GDropList& dlist, GWindow& ownerWin )
                         :GWindow(GString::Empty,
                                  GString::Empty,
                                  null,
                                  &ownerWin,
                                  WS_SAVEBITS,
                                  WS2_OS2Y),
                          list(dlist, *this)
{
   setFontNameSize(dlist.getFontNameSize());
   setLayoutManager(new GBorderLayout(), true);
}

GDropList::DropContainer::~DropContainer ()
{
}

GDropList::TextEntry::TextEntry ( GDropList& dlist )
                     :GTextEntry("Entry",
                                 GString::Empty,
                                 dlist,
                                 WS_VISIBLE,
                                 WS2_OS2Y | WS2_IGNORE_COLORS_AND_FONT_PROFILE,
                                 0,
                                 GTextEntry::LEFT,
                                 true,
                                 false,
                                 true),
                      dlist(dlist)
{
}

GDropList::TextEntry::~TextEntry ()
{
}

bool GDropList::TextEntry::onKeyDown ( const GKeyMessage& key )
{
   char chr = key.getCharacter();
   if (chr != '\0' && !key.isAltDown() && !key.isCtrlDown())
   {
      if (dlist.drop.list.selectNextCharMatch(chr))
         return true;
      return GTextEntry::onKeyDown(key);
   }

   switch (key.getCode())
   {
      case GKey::KEY_ALT_UP:
      case GKey::KEY_ALT_DOWN:
      {
         dlist.cmdDrop();
         return true;
      }

      case GKey::KEY_CTRL_UP: 
      {
         int idx = dlist.getSelectedIndex();
         if (idx > 0)
            dlist.setSelectedIndex(idx - 1);
         return true; 
      }

      case GKey::KEY_CTRL_DOWN: 
      {
         int idx = dlist.getSelectedIndex();
         if (idx < dlist.getItemCount() - 1)
            dlist.setSelectedIndex(idx + 1);
         return true; 
      }

      default:
         return GTextEntry::onKeyDown(key);
   }
}

GDropList::Button::Button ( GDropList& dlist )
                  :GPushButton("cmdDrop", 
                               GString::Empty, 
                               dlist, 
                               WS_VISIBLE, 
                               WS2_DEFAULTPAINT, 
                               "%STDICON_DROPLIST_BUTTON", 
                               false, 
                               true, 
                               GPushButton::IP_CENTER, 
                               true),
                   dlist(dlist)
{
}

GDropList::Button::~Button ()
{
}

GDropList::GDropList ( const GString& name,
                       const GString& constraints,
                       GWindow& parentWin,
                       long winStyle,
                       long winStyle2 )
          :GWindow(name,
                   constraints,
                   &parentWin,
                   &parentWin,
                   winStyle,
                   winStyle2 | WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC),
           selectionListeners(3),
           selectListenerWasCalled(false),
           drop(*this, parentWin.getTopLevelWindow()),
           entry(*this),
           button(*this)
{
   button.setOily(true); // Don't stop on button upon UP/DOWN/etc.
   button.getPeer().setFocusable(false); // Don't stop on button upon TAB either.
   drop.list.addListBoxSelectionListener(this);
}

GDropList::~GDropList ()
{
}

void GDropList::addListBoxSelectionListener ( GListSelectionListener* l )
{
   if (l == null)
      gthrow_(GIllegalArgumentException("l == null"));
   if (!selectionListeners.contains(l)) // If not already registered
      selectionListeners.add(l);
}

void GDropList::setEnabled ( bool flag, bool repaint )
{
   entry.setEnabled(flag, false);
   button.setEnabled(flag, false);
   GWindow::setEnabled(flag, repaint);
}

void GDropList::setOily ( bool flag )
{
   entry.setOily(flag);
   GWindow::setOily(flag);
}

void GDropList::setSelectedIndex ( int index ) 
{ 
   selectListenerWasCalled = false;
   drop.list.setSelectedIndex(index); 
   if (!selectListenerWasCalled)
      onListBoxSelectionChanged(*this);
}

int GDropList::getItemCount () const 
{ 
   return drop.list.getItemCount(); 
}

GObject* GDropList::getItemUserData ( int index ) const 
{ 
   return drop.list.getItemUserData(index); 
}

void GDropList::addItem ( const GString& text,
                          const GString& iconName,
                          GObject* userData,
                          bool autoDelUD )
{
   drop.list.addItem(text, iconName, userData, autoDelUD);
   if (getItemCount() == 1)
      setSelectedIndex(0);
}

void GDropList::insertItem ( int index,
                             const GString& text,
                             const GString& iconName,
                             GObject* userData,
                             bool autoDelUD )
{
   drop.list.insertItem(index, text, iconName, userData, autoDelUD);
   if (getItemCount() == 1)
      setSelectedIndex(0);
}

bool GDropList::removeItem ( int index )
{
   int newindex = -1;
   if (index == getSelectedIndex())
   {
      const int count = getItemCount();
      if (index == count - 1)
         newindex = count - 2;
   }
   bool ret = drop.list.removeItem(index);
   if (newindex >= 0)
      setSelectedIndex(newindex);
   else
      entry.setText(GString::Empty);
   return ret;
}

void GDropList::removeAllItems ()
{
   drop.list.removeAllItems();
   entry.setText(GString::Empty);
}

void GDropList::onListBoxSelectionChanged ( GAbstractListBox& /*listb*/ )
{
   selectListenerWasCalled = true;

   GString txt;
   int idx = getSelectedIndex();
   if (idx >= 0)
      txt = getItemText(idx);
   entry.setText(txt);

   // Send GM_CTRLCHANGED.
   fireValueChangeListeners();

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = selectionListeners.getCount();
   GVector<GListSelectionListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(selectionListeners[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++) 
   {
      GListSelectionListener* l = llist[i];
      l->onListBoxSelectionChanged(*this);
   }
}

void GDropList::onListBoxSelectionChangedByCharKey ( GAbstractListBox& /*listb*/ )
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = selectionListeners.getCount();
   GVector<GListSelectionListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(selectionListeners[i]);

   // Invoke the listeners.
   for (int i=0; i<count; i++) 
   {
      GListSelectionListener* l = llist[i];
      l->onListBoxSelectionChangedByCharKey(*this);
   }
}

void GDropList::onListBoxSelectionDblClick ( GAbstractListBox& /*listb*/ )
{
   setDropListVisible(false);
}

void GDropList::setDropListVisible ( bool show, bool grabEntryFocus )
{
   if (!show && !isDropListVisible())
      return; // Nothing to do.

   // Make sure that the title bar of the containing frame window
   // is painted in "active color" even if the drop down list is active.
   GWindow& top = getTopLevelWindow();
   GFrameWindow* frameWin = dynamic_cast<GFrameWindow*>(&top);
   if (frameWin != null)
      frameWin->setKeepCaptionActive(show);

   // ---
   if (show)
   {
      GDimension screenSize = GSystem::GetScreenSize();
      GPoint pos = getWindowPosOnScreen();
      GDimension dim = getWindowSize();

      int itemHeight = drop.list.getDefaultItemHeight();
      int hscrlHeight = GScrollbar::GetDefaultHScrollbarHeight();

      int width = dim.width;
      int height = (getItemCount() * itemHeight) + (itemHeight / 2);
      height = GMath::Min(height, screenSize.height / 2);
      height = GMath::Max(height, itemHeight);
      height += hscrlHeight + 6;

      int xpos = pos.x;
      int ypos = pos.y - height;

      // Try to keep the window inside the physical screen area.
      if (xpos + width > screenSize.width)
         xpos = screenSize.width - width;
      if (xpos < 0)
         xpos = 0;
      if (ypos < 0)
         ypos = pos.y + dim.height;
      if (ypos + height > screenSize.height)
         ypos = pos.y - height;

      // See comment below, regarding posting user message "hideDropList".
      drop.setWindowBounds(xpos, ypos, width, height);
      postUserMessage("showDropList");
   }
   else
   {
      // We must do this asynchronously in order to prevent possible 
      // problems on some systems (e.g. OS/2) because of some limitations
      // on what we are allowed to do upon handling synchronous focus-change
      // events from the system, and we are often called upon such 
      // system related focus change events.
      GBoolean* b = new GBoolean(grabEntryFocus);
      bool autoDeleteB = true;
      postUserMessage("hideDropList", b, autoDeleteB);
   }

   // Request the button to repaint it self in order to remove
   // its bold border, if needed.
   button.invalidateAll(true);
}

bool GDropList::onUserMessage ( GUserMessage& msg )
{
   GString id = msg.getParam1String();
   if (id == "showDropList")
   {
      drop.setVisible(true);
      drop.list.grabFocus(true);
      return true;
   }
   else
   if (id == "hideDropList")
   {
      drop.setVisible(false);
      bool grabEntryFocus = msg.getParam2Bool();
      if (grabEntryFocus)
         entry.grabFocusAsynchronously();
      return true;
   }
   return GWindow::onUserMessage(msg);
}

bool GDropList::isDropListVisible ()
{
   return drop.isVisible();
}

void GDropList::cmdDrop ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isDropListVisible();
   setDropListVisible(show);
}

void GDropList::layout ()
{
   GDimension dim = getWindowSize();
   int vscrlWidth = GScrollbar::GetDefaultVScrollbarWidth();
   int buttonWidth = GMath::Min(dim.width/2, vscrlWidth + 1);
   int entryWidth = GMath::Max(0, dim.width - buttonWidth);
   entry.setWindowBounds(0, 0, entryWidth, dim.height);
   button.setWindowBounds(entryWidth, 0, buttonWidth, dim.height);
}

int GDropList::getPreferredHeight () const
{
   int strh = getHeightOfString("X");
   return (6 * strh) + 12;
}

int GDropList::getPreferredWidth () const
{
   int strw = getWidthOfString("X");
   return (16 * strw) + 36;
}

bool GDropList::isEmpty () const
{
   return getItemCount() <= 0;
}

void GDropList::changeValue ( const GString& newValue, bool notify )
{
   enterCtrlChangedFilter();
   try {
      int index = GInteger::ParseInt(newValue);
      setSelectedIndex(index);
   } catch (GNumberFormatException& /*e*/) {
   }
   exitCtrlChangedFilter();
   if (notify)
      fireValueChangeListeners();
}

GString GDropList::queryValue () const
{
   int idx = getSelectedIndex();
   return GInteger::ToString(idx);
}

void GDropList::grabFocus ( bool force ) 
{ 
   entry.grabFocus(force); 
}

int GDropList::getSelectedIndex () const 
{ 
   return drop.list.getSelectedIndex(); 
}

const GListBoxItem& GDropList::getItem ( int index ) const 
{ 
   return drop.list.getItem(index); 
}

void GDropList::setItemIcon ( int index, const GString& iconName ) 
{ 
   drop.list.setItemIcon(index, iconName); 
}

void GDropList::setItemTextAndIcon ( int index, 
                                     const GString& text, 
                                     const GString& iconName ) 
{ 
   drop.list.setItemTextAndIcon(index, text, iconName); 
}

void GDropList::setItemText ( int index, const GString& text ) 
{ 
   drop.list.setItemText(index, text); 
}

GString GDropList::getItemText ( int index ) const 
{ 
   return drop.list.getItemText(index); 
}

void GDropList::setItemUserData ( int index, GObject* userData, bool autoDelete ) 
{ 
   drop.list.setItemUserData(index, userData, autoDelete); 
}
