/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ABSTRACTTOOLBARWINDOW
#define __GLIB_ABSTRACTTOOLBARWINDOW

#include "glib/gui/GToolbarElement.h"

/**
 * This class forms the abstract and general base functionality of any
 * toolbar type of windows.
 *
 * Examples of usage in Larsen Commander are 1) the main toolbar window,
 * 2) the file panel header bar window (which contain some push buttons) and
 * 3) the keybar window at bottom of the Larsen Commander main window.
 *
 * @author  Leif Erik Larsen
 * @since   1998.11.07
 */
class GAbstractToolbarWindow : public GWindow
{
   public:

      /** The default font for all kinds of toolbar windows. */
      static const GString DefaultFont;

      /** True if we shall paint flat button borders rather than as 3D. */
      bool flatButtons;

      /** True if we shall show and maintain the tooltip. */
      bool showTooltip;

   private:

      /** The vector of toolbar elements (buttons, separators, etc.) that are currently contained in the toolbar window. */
      GArray<GToolbarElement> buttons;

   protected:

      GAbstractToolbarWindow ( const GString& name,
                               const GString& constraints,
                               GWindow& parentWin,
                               bool clipChildren = false,
                               int winStyle = WS_VISIBLE,
                               int winStyle2 = 0 );

   public:

      virtual ~GAbstractToolbarWindow ();

   private:

      /** Disable the copy constructor. */
      GAbstractToolbarWindow ( const GAbstractToolbarWindow& src ) {}

      /** Disable the assignment operator. */
      GAbstractToolbarWindow& operator= ( const GAbstractToolbarWindow& ) { return *this; }

   protected:

      /**
       * This is the inner implementation of all methods used to add 
       * buttons or toolbar elements of any type. It can be overridden 
       * by the subclass, typically in order to keep track of all 
       * elements added.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       */
      virtual class GToolbarElement* addElementImpl ( class GToolbarElement* elm, bool autoDelete = true );

   public:

      virtual bool onPaintBackground ( GGraphics& g, const GRectangle& rect );
      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onForegroundColorChanged ( const GColor& color );

      void setFlatButtonBorders ( bool flag );
      virtual bool isFlatButtonBorders () const;
      void setShowTooltip ( bool flag );
      virtual bool isShowTooltip () const;

      void addButton ( class GToolbarButton& butt );
      class GToolbarButton* addButton ( class GToolbarButton* butt, bool autoDelete = true );
      class GToolbarButton* addButton ( const GString& id, const GString& text, const GString& iconName, const GString& userData1, const GString& constraints = GString::Empty );
      class GToolbarElement* addElement ( class GToolbarElement* elm, bool autoDelete = true );
      class GToolbarSeparator* addSeparator ( const GString& constraints = GString::Empty );

      /**
       * Get a reference to the indexed toolbar element.
       */
      class GToolbarElement& getElement ( int index );

      /**
       * Get the number of toolbar elements (buttons, separators, etc.) that
       * are currently contained in the toolbar window.
       */
      int getElementCount () const { return buttons.getCount(); }

      /**
       * Load and activate the profile variables for the toolbar
       * window, from the specified section name.
       * @see #writeProfile
       */
      virtual void queryProfile ( const GString& sectName );

      /**
       * Change the font of all the toolbar elements in one operation.
       */
      void setFontNameSizeOnAllElements ( const GString& fontNameSize );

      /**
       * Write the profile variables for the toolbar window, under the
       * specified section name.
       * @see #queryProfile
       */
      virtual void writeProfile ( const GString& sectName, bool force );
};

#endif
