/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GAbstractToolbarWindow.h"
#include "glib/gui/GToolbarSeparator.h"
#include "glib/gui/GToolbarButton.h"
#include "glib/gui/GGraphics.h"
#include "glib/util/GSectionBag.h"
#include "glib/GProgram.h"

const GString GAbstractToolbarWindow::DefaultFont = "8.Helv";

GAbstractToolbarWindow::GAbstractToolbarWindow ( const GString& name,
                                                 const GString& constraints,
                                                 GWindow& parentWin,
                                                 bool clipChildren,
                                                 int winStyle,
                                                 int winStyle2 )
                       :GWindow(name,
                                constraints,
                                &parentWin,
                                &parentWin,
                                winStyle | (clipChildren ? WS_CLIPCHILDREN : 0),
                                winStyle2 | WS2_OS2Y,
                                GWindowClass::GENERIC,
                                GSystem::GetSystemColor(GSystem::SCID_DIALOGBCK),
                                GSystem::GetSystemColor(GSystem::SCID_DIALOGTXT),
                                GAbstractToolbarWindow::DefaultFont),
                        flatButtons(true),
                        showTooltip(true),
                        buttons(32)
{
   setTooltipPosition(GTooltip::PosBelow);
}

GAbstractToolbarWindow::~GAbstractToolbarWindow ()
{
}

GToolbarElement& GAbstractToolbarWindow::getElement ( int index ) 
{ 
   return buttons.get(index);
}

void GAbstractToolbarWindow::addButton ( GToolbarButton& butt )
{
   addElement(&butt, false);
}

GToolbarButton* GAbstractToolbarWindow::addButton ( GToolbarButton* butt, bool autoDelete )
{
   addElement(butt, autoDelete);
   return butt;
}

GToolbarButton* GAbstractToolbarWindow::addButton ( const GString& id, const GString& text, const GString& iconName, const GString& userData1, const GString& constraints )
{
   GToolbarButton* butt = new GToolbarButton(id, constraints, *this, id, text, iconName, userData1);
   addButton(butt);
   return butt;
}

GToolbarElement* GAbstractToolbarWindow::addElement ( GToolbarElement* elm, bool autoDelete )
{
   return addElementImpl(elm, autoDelete);
}

GToolbarElement* GAbstractToolbarWindow::addElementImpl ( GToolbarElement* elm, bool autoDelete )
{
   buttons.add(elm, autoDelete);
   return elm;
}

GToolbarSeparator* GAbstractToolbarWindow::addSeparator ( const GString& constraints )
{
   GToolbarSeparator* sep = new GToolbarSeparator(*this, constraints);
   addElement(sep);
   return sep;
}

void GAbstractToolbarWindow::writeProfile ( const GString& sectName, bool force )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   ini.putBool(sectName, "Visible", isVisible(), force || prg.isAutoSaveVisibilityStateOptions());
   ini.putBool(sectName, "FlatButtons", flatButtons, force || prg.isAutoSaveVisibilityStateOptions());
   ini.putBool(sectName, "ShowTooltip", showTooltip, force || prg.isAutoSaveVisibilityStateOptions());
   GWindow::writeProfile(sectName, force);
}

void GAbstractToolbarWindow::queryProfile ( const GString& sectName )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   bool visible = ini.getBool(sectName, "Visible", isVisible());
   setVisible(visible);
   flatButtons = ini.getBool(sectName, "FlatButtons", flatButtons);
   showTooltip = ini.getBool(sectName, "ShowTooltip", showTooltip);
   GWindow::queryProfile(sectName);
}

void GAbstractToolbarWindow::setFontNameSizeOnAllElements ( const GString& fontNameSize )
{
   int num = getElementCount();
   for (int i=0; i<num; i++)
      getElement(i).setFontNameSize(fontNameSize);
}

void GAbstractToolbarWindow::setFlatButtonBorders ( bool flag )
{
   if (flag == flatButtons)
      return;

   flatButtons = flag;
   invalidateAll(true);
}

bool GAbstractToolbarWindow::isFlatButtonBorders () const
{
   return flatButtons;
}

void GAbstractToolbarWindow::setShowTooltip ( bool flag )
{
   showTooltip = flag;
}

bool GAbstractToolbarWindow::isShowTooltip () const
{
   return showTooltip;
}

bool GAbstractToolbarWindow::onBackgroundColorChanged ( const GColor& /*color*/ )
{
   // The background color of the toolbar has changed. This means that all 
   // child windows (typically separators and buttons) must get a chance to 
   // repaint them selves, because they might use the same background color 
   // as of their owning toolbar.
   invalidateAll(true);
   return true;
}

bool GAbstractToolbarWindow::onForegroundColorChanged ( const GColor& /*color*/ )
{
   invalidateAll(true);
   return true;
}

bool GAbstractToolbarWindow::onPaintBackground ( GGraphics& g, const GRectangle& rect )
{
   const GColor& bck = getBackgroundColor();
   g.drawFilledRectangle(rect, bck);
   return true;
}
