/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ABSTRACTLISTBOX
#define __GLIB_ABSTRACTLISTBOX

#include "glib/gui/GListBoxItem.h"

/**
 * Interface used to declare the common behaviour of various types of
 * List Box components, such as {@link GListBox} and {@link GDropList}.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 */
class GAbstractListBox
{
   public:

      /**
       * Add a new item to the bottom of the List Box.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   text      Text of the new item of which to add.
       * @param   iconName  Name of which Icon Resource to paint next to the
       *                    item text, or an empty string if no icon. The
       *                    icon must exist in the specified resource table.
       * @param   userData  A reference to any Program Defined User Data
       *                    Object to be associated with the item.
       * @param   autoDelUD True if we shall automatically delete the
       *                    specified user data object when it is no longer
       *                    referenced by the list box, or else false.
       */
      virtual void addItem ( const GString& text, 
                             const GString& iconName = GString::Empty, 
                             GObject* userData = null, 
                             bool autoDelUD = false ) = 0;

      /**
       * Insert a new item into the List Box at the indexed position.
       *
       * The new item will be inserted above the indexed item, or it will
       * be added to the bottom of the List if the specified index is equal
       * to, or larger than, the current number of items in the list.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index     Index of where to insert the specified item. The
       *                    new item will be inserted just above the current
       *                    selected item if you specify -1. If you specify
       *                    -1 and there are no current selected item then
       *                    the new item will be added to the bottom of the
       *                    List.
       * @param   text      Text of the new item of which to insert.
       * @param   iconName  Name of which Icon Resource to paint next to the
       *                    item text, or an empty string if no icon. The
       *                    icon must exist in the specified resource table.
       * @param   userData  A reference to any Program Defined User Data
       *                    Object to be associated with the item.
       * @param   autoDelUD True if we shall automatically delete the
       *                    specified user data object when it is no longer
       *                    referenced by the list box, or else false.
       */
      virtual void insertItem ( int index, 
                                const GString& text, 
                                const GString& iconName = GString::Empty, 
                                GObject* userData = null, 
                                bool autoDelUD = false ) = 0;

      /**
       * Get a read-only reference to the indexed List Box item object.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @param   index     Index of which item to get, or -1 to get
       *                    the current selected item.
       */
      virtual const GListBoxItem& getItem ( int index ) const = 0;

      /**
       * Set the current List Box selection so that the indexed item
       * is the current selected item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index   Index of which item to select.
       */
      virtual void setSelectedIndex ( int index ) = 0;

      /**
       * Get the index of the current selected item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The index of the current selected item, or -1 if there
       *          is no item currently selected or if the List Box is
       *          currently empty.
       */
      virtual int getSelectedIndex () const = 0;

      /**
       * Change the text of the indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index  Index of which item to set, or -1 to set the
       *                 current selected item.
       * @param   text   The new text of which to change to.
       */
      virtual void setItemText ( int index, const GString& text ) = 0;

      /**
       * Change the icon of the indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index     Index of which item to set, or -1 to set the
       *                    current selected item.
       * @param   iconName  Name of which Icon Resource to paint next to the
       *                    item text, or an empty string if no icon. The
       *                    icon must exist in the specified resource table.
       */
      virtual void setItemIcon ( int index, const GString& iconName ) = 0;

      /**
       * Change the text and the icon of the indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index     Index of which item to set, or -1 to set the
       *                    current selected item.
       * @param   text      The new text of which to change to.
       * @param   iconName  Name of which Icon Resource to paint next to the
       *                    item text, or an empty string if no icon. The
       *                    icon must exist in the specified resource table.
       */
      virtual void setItemTextAndIcon ( int index, 
                                        const GString& text, 
                                        const GString& iconName ) = 0;

      /**
       * Get a copy of the current text of the indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index     Index of which item to get, or -1 to get
       *                    the current selected item.
       */
      virtual GString getItemText ( int index ) const = 0;

      /**
       * Set the Program Defined User Data Object of the indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index      Index of which item to set, or -1 to set the
       *                     current selected item.
       * @param   userData   Program Defined User Data Object to assign to
       *                     the indexed item. Can be null.
       * @param   autoDelete True if we shall automatically delete the
       *                     specified user data object when it is no longer
       *                     referenced by the list box, or else false.
       */
      virtual void setItemUserData ( int index, GObject* userData, bool autoDelete ) = 0;

      /**
       * Get a reference to the Program Defined User Data Object of the
       * indexed item.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index   Index of which item to query, or -1 to query the
       *                  current selected item.
       * @return  A pointer to the Program Defined User Data Object
       *          of the item.
       */
      virtual GObject* getItemUserData ( int index ) const = 0;

      /**
       * Remove the indexed item from the List Box.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   index   Index of which item to remove, or -1 to remove
       *                  the current selected item.
       * @return  True if the indexed item was successfully removed, or else
       *          false. Returns false when specified index is out of bounds.
       */
      virtual bool removeItem ( int index ) = 0;

      /**
       * Remove all items from the List Box.
       *
       * This will make the List Box empty.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual void removeAllItems () = 0;

      /**
       * Get the current number of items in the List Box.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual int getItemCount () const = 0;
};

#endif // #ifndef __GLIB_ABSTRACTLISTBOX

