/* --------------------------------------------------------------------------
 *
 * This file was part of the "More for C++" library.
 * Copyright (c) 1999-2003 by Thorsten Goertz.
 * The "More for C++" library is free software; you can
 * redistribute it and/or modify it under the terms of the license
 * that comes with http://www.morefor.org/.
 *
 * ------------------------------------------------------------------------ */

#include "glib/primitives/GObject.h"
#include "glib/gc/create.h"
#include "glib/gc/bitmask.h"
#include "glib/gc/gcimpl.h"
#include "glib/util/GLog.h"

typedef unsigned char byte;

static GCImpl gc;
static size_t magicKey = 0x22021969;
static char magicPattern[4] = { 0x55, 0x55, 0x55, 0x55 };

Handle GCImpl::NilHandle = { 0, 0, 0, 0, 0, 0, 0, magicKey };

void GCImpl::Init ( HeapManager& heapManager )
{
  gc.m_pHeapManager = &heapManager;
  gc.m_pLastObject = 0;
  gc.m_nNoOfObjects = 0;
  gc.m_nNoOfTareObjects = 0;
  gc.m_bCollecting = false;
  gc.m_pMarkStack = new ObjectStack();
  gc.m_ppCollectorThread = new p<GCThread>;
  *gc.m_ppCollectorThread = CREATE GCThread();
}

void GCImpl::ShutDown ()
{
   *gc.m_ppCollectorThread = 0;

   while (gc.m_nNoOfObjects > 0)
   {
      GCImpl::CollectObjects();
   }

   delete gc.m_ppCollectorThread;
   delete gc.m_pMarkStack;

   gc.m_ppCollectorThread = 0;
   gc.m_pMarkStack = 0;
   gc.m_bCollecting = false;
   gc.m_nNoOfTareObjects = 0;
   gc.m_nNoOfObjects = 0;
   gc.m_pLastObject = 0;
   gc.m_pHeapManager = 0;
}

void* GCImpl::CreateObject ( size_t nSizeOfObject, bool bMayBeNode )
{
   Handle* pHandle;
   size_t  nConcreteSize = sizeof( Handle ) + nSizeOfObject;
   size_t  nNoOfBytesForBits = 0;

   if (bMayBeNode)
   {
      nNoOfBytesForBits = BitMask::calcSize( nSizeOfObject );
      nConcreteSize += nNoOfBytesForBits;
   }

   synchronized (gc)
   {
      pHandle = ( Handle* ) gc.m_pHeapManager -> createObject( nConcreteSize );
   } end_synchronized

   pHandle -> m_nFlags = 0x00000000;
   pHandle -> m_pPrevObject = 0;
   pHandle -> m_pFinalizable = 0;
   pHandle -> m_nSizeOfObject = nSizeOfObject;
   pHandle -> m_nNoOfRootPointers = 1; // !!!
   pHandle -> m_pForRootPointers = 0;
   pHandle -> m_pForPointers = 0;
   pHandle -> m_pMagicKey = magicKey;

   if( bMayBeNode )
   {
      byte*   pBytesForBits = ( byte* ) ( pHandle + 1 ) + nSizeOfObject;
      BitMask bits( pBytesForBits, nNoOfBytesForBits, nSizeOfObject );

      memset( pHandle + 1, magicPattern[0], nSizeOfObject );
      gc.setNodeFlag( pHandle, true );
      bits.clear( );
   }

   gc.addObject( pHandle );

   return pHandle + 1;
}

bool GCImpl::IsRootPointer ( void** ppObject )
{
  size_t* pKey = ( size_t* ) ppObject;
  return pKey[1] != magicKey;
}

Handle* GCImpl::ResolveObject ( void* pObject )
{
   size_t* pKey = ( size_t* ) pObject;
   while( pKey[-1] != magicKey )
      pKey--;
   return ( ( Handle* ) pKey ) - 1;
}

void GCImpl::InitPointer ( void**& rppObject,
                           void* pObject,
                           GObject* pFinalizable )
{
  bool    bIsRootPointer = true;
  byte*   pFirstByteOfObject = 0;
  size_t  nSizeOfObject = 0;
  size_t  nDistance = 0;

  if( memcmp( &rppObject, magicPattern, 4 ) == 0 )
  {
    synchronized (gc)
    {
      Handle* pHandle = gc.getLastObject( );

      while( bIsRootPointer == true && pHandle != 0 )
      {
        byte* pFirstByteOfPointer = ( byte* ) &rppObject;

        pFirstByteOfObject = ( byte* ) ( pHandle + 1 );
        nSizeOfObject = pHandle -> m_nSizeOfObject;
        nDistance = size_t( pFirstByteOfPointer - pFirstByteOfObject );

        if( nDistance < nSizeOfObject )
        {
          bIsRootPointer = false;
        }

        pHandle = gc.getPreviousObject( pHandle );
      }

    } end_synchronized
  }

  GCImpl::AssignObject( rppObject, pObject, pFinalizable, bIsRootPointer );

  if (bIsRootPointer == false)
  {
     byte*   pBytesForBits = pFirstByteOfObject + nSizeOfObject;
     size_t  nNoOfBytesForBits = BitMask::calcSize( nSizeOfObject );
     BitMask bitMask( pBytesForBits, nNoOfBytesForBits, nSizeOfObject );
     bitMask.setBit( nDistance, true );
  }
}

void GCImpl::AssignObject ( void**& rppObject,
                            void* pObject,
                            GObject* pFinalizable,
                            bool bIsRootPointer )
{
   Handle* pHandle;
   bool    bUnlockUnassignedObject = false;
 
   if( pObject == 0 )
   {
       pHandle = &GCImpl::NilHandle;
   }
   else
   {
      pHandle = GCImpl::ResolveObject(pObject);
      if( pHandle -> m_pForPointers == 0 )
      {
         bUnlockUnassignedObject = true;
         pHandle -> m_pFinalizable = pFinalizable;
         pHandle -> m_pForRootPointers = pObject;
         pHandle -> m_pForPointers = pObject;
      }
   }

   if (bIsRootPointer)
   {
      rppObject = &pHandle -> m_pForRootPointers;
      pHandle -> m_nNoOfRootPointers++;
   }
   else
   {
      rppObject = &pHandle -> m_pForPointers;
   }

   if( bUnlockUnassignedObject )
   {
      pHandle -> m_nNoOfRootPointers--; // !!!
   }
}

void GCImpl::ForgetObject ( void**& rppObject, bool bIsRootPointer )
{
  Handle* pHandle;

  if( *rppObject == 0 )
  {
    pHandle = &GCImpl::NilHandle;
  }
  else
  {
    pHandle = GCImpl::ResolveObject( *rppObject );
  }

  if( bIsRootPointer )
  {
    rppObject = &GCImpl::NilHandle.m_pForRootPointers;
    pHandle -> m_nNoOfRootPointers--;
  }
  else
  {
    rppObject = &GCImpl::NilHandle.m_pForPointers;
  }
}

void GCImpl::ForgetUnassignedObject ( void* pObject )
{
  Handle* pHandle = GCImpl::ResolveObject(pObject);
  if (pHandle -> m_pForPointers == 0)
  {
    pHandle -> m_pFinalizable = 0;
    pHandle -> m_pForRootPointers = pObject;
    pHandle -> m_pForPointers = pObject;
    pHandle -> m_nNoOfRootPointers--;
  }
}

GCImpl::GCThread::GCThread ()
                 :GThread("GarbageCollector"),
                  requestStop(false)
{
}

GCImpl::GCThread::~GCThread ()
{
   requestStop = true;
}

void GCImpl::GCThread::run ()
{
   while (!requestStop)
   {
      try {
         GThread::Sleep(1000);
         GCImpl::CollectObjects();
      } catch (GException& e) {
         GString msg("Unhandled exception in GC-thread. Message is: %s", GVArgs(e.toString()));
         GString stackTrace = e.getStackTrace(msg);
         GLog::Log(this, stackTrace);
      } catch (std::exception& e) {
         GString msg("Unhandled exception in GC-thread. Message is: %s", GVArgs(e.what()));
         GLog::Log(this, msg);
      } catch (...) {
         GString msg("Unhandled exception of unknown type in GC-thread.");
         GLog::Log(this, msg);
      }
   }
}

void GCImpl::StartCollectorThread ()
{
   if ((*gc.m_ppCollectorThread)->isRunning())
      return;
   synchronized (gc)
   {
      if (!(*gc.m_ppCollectorThread)->isRunning())
      {
         (*gc.m_ppCollectorThread)->requestStop = false;
         (*gc.m_ppCollectorThread)->start();
      }
   } end_synchronized
}

void GCImpl::StopCollectorThread ()
{
   if (!(*gc.m_ppCollectorThread)->isRunning())
      return;
   synchronized (gc)
   {
      (*gc.m_ppCollectorThread)->requestStop = true;
   } end_synchronized
   (*gc.m_ppCollectorThread)->waitUntilTheThreadHasExited();
}

void GCImpl::DeclareObjectsTare ()
{
   gc.m_nNoOfTareObjects = gc.m_nNoOfObjects;
}

size_t GCImpl::GetNoOfObjects ()
{
   return gc.m_nNoOfObjects - gc.m_nNoOfTareObjects;
}

void GCImpl::CollectObjects ( bool force )
{
   for (;;)
   {
      if (gc.m_bCollecting == false)
      {
         synchronized (gc) 
         {
            if( gc.m_bCollecting == false ) 
            {
               gc.m_bCollecting = true;
               break;
            }
         } end_synchronized
      }
      if (!force)
         return;
      GThread::Sleep(25);
   }

   try {
      gc.markObjects();
      gc.unmarkRootObjectTree();
      gc.finalizeMarkedObjects();
      gc.sweepMarkedObjects();
      synchronized (gc) {
         gc.m_bCollecting = false;
      } end_synchronized
   } catch (...) {
      synchronized (gc) {
         gc.m_bCollecting = false;
      } end_synchronized
      throw;
   }
}

void GCImpl::addObject ( Handle* pObject )
{
   synchronized (*this)
   {
      pObject->m_pPrevObject = m_pLastObject;
      m_pLastObject = pObject;
      m_nNoOfObjects++;

   } end_synchronized
}

Handle* GCImpl::getLastObject () const
{
   Handle* pResult;

   synchronized (*this)
   {
      pResult = m_pLastObject;

   } end_synchronized

   return pResult;
}

Handle* GCImpl::getPreviousObject ( Handle *pHandle ) const
{
   Handle* pResult;

   synchronized (*this)
   {
      pResult = pHandle -> m_pPrevObject;

   } end_synchronized

   return pResult;
}

void GCImpl::removeObject ( Handle* pObject, Handle* pNextObject )
{
   synchronized (*this)
   {
      m_nNoOfObjects--;
      if (pObject == m_pLastObject)
      {
         m_pLastObject = pObject->m_pPrevObject;
      }
      else
      {
         pNextObject->m_pPrevObject = pObject->m_pPrevObject;
      }

   } end_synchronized
}

void GCImpl::markObjects ()
{
  Handle* pHandle = getLastObject( );

  while( pHandle != 0 )
  {
    setGarbageFlag( pHandle, true );

    if( pHandle -> m_nNoOfRootPointers > 0 )
    {
      m_pMarkStack -> pushObject( pHandle );
    }

    pHandle = getPreviousObject( pHandle );
  }
}

void GCImpl::unmarkRootObjectTree ()
{
  while( m_pMarkStack -> isEmpty( ) == false )
  {
    Handle* pHandle = ( Handle* ) m_pMarkStack -> popObject( );

    if( objectIsGarbage( pHandle ) )
    {
      setGarbageFlag( pHandle, false );

      if( objectMayBeNode( pHandle ) )
      {
        unmarkChildObjects( pHandle );
      }
    }
  }
}

void GCImpl::unmarkChildObjects ( Handle* pHandle )
{
  size_t  i = 0;
  byte*   pObject = ( byte* ) ( pHandle + 1 );
  size_t  nSizeOfObject = pHandle -> m_nSizeOfObject;
  byte*   pBytesForBits = pObject + nSizeOfObject;
  size_t  nNoOfBytesForBits = BitMask::calcSize( nSizeOfObject );
  BitMask bitMask( pBytesForBits, nNoOfBytesForBits, nSizeOfObject );

  while( i < nSizeOfObject )
  {
    if( bitMask.getBit( i ) )
    {
      void*** pppChildObject = ( void*** ) ( pObject + i );
      void**  ppChildObject = *pppChildObject;
      void*   pChildObject = *ppChildObject;

      if (pChildObject != 0)
      {
        pHandle = GCImpl::ResolveObject( pChildObject );
        if( pHandle != 0 && objectIsGarbage( pHandle ) )
        {
           m_pMarkStack -> pushObject( pHandle );
        }
      }

      i += sizeof( pppChildObject );
    }
    else
    {
      i += 2;
    }
  }
}

void GCImpl::finalizeMarkedObjects ()
{
   Handle* obj = getLastObject();
   while (obj != 0)
   {
      if (objectIsGarbage(obj))
      {
         GObject* finalizable = obj->m_pFinalizable;
         if (finalizable != null)
         {
            finalizable->finalize();
            // Set the finalizable pointer to null, so we will not call 
            // it again. Regardless of what happens.
            obj->m_pFinalizable = null;
         }
      }
      obj = getPreviousObject(obj);
   }
}

void GCImpl::sweepMarkedObjects ()
{
  ObjectStack garbage;

  synchronized (*this)
  {
    Handle* pPreviousObject = 0;
    Handle* pObject = getLastObject( );

    while( pObject != 0 )
    {
      if( objectIsGarbage( pObject ) )
      {
        Handle* pGarbage = pObject;

        garbage.pushObject( pGarbage );
        pObject = getPreviousObject( pGarbage );
        removeObject( pGarbage, pPreviousObject );
      }
      else
      {
        pPreviousObject = pObject;
        pObject = getPreviousObject( pObject );
      }
    }

    while( !garbage.isEmpty( ) )
    {
      m_pHeapManager -> destroyObject( garbage.popObject( ) );
    }

    m_pHeapManager -> compactHeap( );

  } end_synchronized
}

void GCImpl::setNodeFlag ( Handle* pHandle, bool    bObjectMayBeNode )
{
  if( bObjectMayBeNode )
  {
    pHandle -> m_nFlags |= 0x00000001;
  }
  else
  {
    pHandle -> m_nFlags &= 0xFFFFFFFE;
  }
}

bool GCImpl::objectMayBeNode ( Handle* pHandle ) const
{
  return ( pHandle -> m_nFlags & 0x00000001 ) == 0x00000001;
}

void GCImpl::setGarbageFlag ( Handle* pHandle, bool bGarbage )
{
  if( bGarbage )
  {
    pHandle -> m_nFlags |= 0x00000002;
  }
  else
  {
    pHandle -> m_nFlags &= 0xFFFFFFFD;
  }
}

bool GCImpl::objectIsGarbage ( Handle* pHandle ) const
{
   return (pHandle->m_nFlags & 0x00000002) == 0x00000002;
}
