/* --------------------------------------------------------------------------
 *
 * This file was part of the "More for C++" library.
 * Copyright (c) 1999-2003 by Thorsten Goertz.
 * The "More for C++" library is free software; you can
 * redistribute it and/or modify it under the terms of the license
 * that comes with http://www.morefor.org/.
 *
 * ------------------------------------------------------------------------ */

#include <malloc.h>
#include "glib/gc/aligningheapmanager.h"

////////////////////////////////////////////////////////////////////////////////

typedef unsigned char byte;

static const size_t nSize_t = 2 * sizeof( size_t );

static const size_t nAlignment0 = 48 - nSize_t;
static const size_t nAlignment1 = 64 - nSize_t;
static const size_t nAlignment2 = 80 - nSize_t;
static const size_t nAlignment3 = 96 - nSize_t;
static const size_t nAlignment4 = 128 - nSize_t;
static const size_t nAlignment5 = 160 - nSize_t;
static const size_t nLargestAlignment = nAlignment5;

////////////////////////////////////////////////////////////////////////////////

inline void* getObject
(
  void*   pAlignedObjects,
  size_t  nAlignment,
  size_t  nIndex
)
{
  return ( byte* ) pAlignedObjects + nSize_t + nIndex * ( nSize_t + nAlignment );
}

////////////////////////////////////////////////////////////////////////////////

inline void setSizeOfObject
(
  void*   pObject,
  size_t  nSizeOfObject
)
{
  ( ( size_t* ) pObject )[-1] = nSizeOfObject;
}

////////////////////////////////////////////////////////////////////////////////

inline size_t getSizeOfObject
(
  void* pObject
)
{
  return ( ( size_t* ) pObject )[-1];
}

////////////////////////////////////////////////////////////////////////////////

AligningHeapManager::AligningHeapManager( ):
  m_alignedObjects0( nAlignment0 ),
  m_alignedObjects1( nAlignment1 ),
  m_alignedObjects2( nAlignment2 ),
  m_alignedObjects3( nAlignment3 ),
  m_alignedObjects4( nAlignment4 ),
  m_alignedObjects5( nAlignment5 )
{
}

////////////////////////////////////////////////////////////////////////////////

AligningHeapManager::~AligningHeapManager( )
{
}

////////////////////////////////////////////////////////////////////////////////

void* AligningHeapManager::createObject
(
  size_t nSizeOfObject
)
{
  void* pObject;

  if( nSizeOfObject <= nAlignment0 )
  {
    pObject = m_alignedObjects0.createObject( nSizeOfObject );
  }
  else if( nSizeOfObject <= nAlignment1 )
  {
    pObject = m_alignedObjects1.createObject( nSizeOfObject );
  }
  else if( nSizeOfObject <= nAlignment2 )
  {
    pObject = m_alignedObjects2.createObject( nSizeOfObject );
  }
  else if( nSizeOfObject <= nAlignment3 )
  {
    pObject = m_alignedObjects3.createObject( nSizeOfObject );
  }
  else if( nSizeOfObject <= nAlignment4 )
  {
    pObject = m_alignedObjects4.createObject( nSizeOfObject );
  }
  else if( nSizeOfObject <= nAlignment5 )
  {
    pObject = m_alignedObjects5.createObject( nSizeOfObject );
  }
  else
  {
    pObject = ( byte* ) malloc( nSize_t + nSizeOfObject ) + nSize_t;

    setSizeOfObject( pObject, nSizeOfObject );
  }

  return pObject;
}

////////////////////////////////////////////////////////////////////////////////

void AligningHeapManager::destroyObject
(
  void* pObject
)
{
  size_t nSizeOfObject = getSizeOfObject( pObject );

  if( nSizeOfObject <= nLargestAlignment )
  {
    setSizeOfObject( pObject, 0 );
  }
  else
  {
    free( ( byte* ) pObject - nSize_t );
  }
}

////////////////////////////////////////////////////////////////////////////////

void AligningHeapManager::compactHeap( )
{
  m_alignedObjects5.compact( );
  m_alignedObjects4.compact( );
  m_alignedObjects3.compact( );
  m_alignedObjects2.compact( );
  m_alignedObjects1.compact( );
  m_alignedObjects0.compact( );
}

////////////////////////////////////////////////////////////////////////////////

AligningHeapManager::AlignedObjects::AlignedObjects
(
  size_t nAlignment
)
: m_nAlignment( nAlignment ),
  m_nCapacity( 0x4000 / ( nSize_t + m_nAlignment ) ),
  m_nNextObject( 0 ),
  m_pObjects( malloc( m_nCapacity * ( nSize_t + m_nAlignment ) ) ),
  m_pNextAlignedObjects( 0 )
{
  for( size_t i = 0; i < m_nCapacity; i++ )
  {
    setSizeOfObject( getObject( m_pObjects, m_nAlignment, i ), 0 );
  }
}

////////////////////////////////////////////////////////////////////////////////

AligningHeapManager::AlignedObjects::~AlignedObjects( )
{
  delete m_pNextAlignedObjects;
  free( m_pObjects );
}

////////////////////////////////////////////////////////////////////////////////

void* AligningHeapManager::AlignedObjects::createObject
(
  size_t nSizeOfObject
)
{
  void* pObject;

  if( m_nNextObject < m_nCapacity )
  {
    void* pNextObject;

    pObject = getObject( m_pObjects, m_nAlignment, m_nNextObject );
    pNextObject = getObject( m_pObjects, m_nAlignment, ++m_nNextObject );

    while( m_nNextObject < m_nCapacity && getSizeOfObject( pNextObject ) > 0 )
    {
      pNextObject = getObject( m_pObjects, m_nAlignment, ++m_nNextObject );
    }
  }
  else
  {
    if( m_pNextAlignedObjects == 0 )
    {
      m_pNextAlignedObjects = new AlignedObjects( m_nAlignment );
    }

    pObject = m_pNextAlignedObjects -> createObject( nSizeOfObject );
  }

  setSizeOfObject( pObject, nSizeOfObject );

  return pObject;
}

////////////////////////////////////////////////////////////////////////////////

size_t AligningHeapManager::AlignedObjects::compact( )
{
  size_t nNoOfObjects = 0;

  for( size_t i = 0; i < m_nCapacity; i++ )
  {
    void* pObject = getObject( m_pObjects, m_nAlignment, i );

    if( getSizeOfObject( pObject ) > 0 )
    {
      nNoOfObjects++;
    }
    else if( m_nNextObject > i )
    {
      m_nNextObject = i;
    }
  }

  if( m_pNextAlignedObjects != 0 )
  {
    if( m_pNextAlignedObjects -> compact( ) == 0 )
    {
      AlignedObjects* pNextAlignedObjects = m_pNextAlignedObjects;

      m_pNextAlignedObjects = pNextAlignedObjects -> m_pNextAlignedObjects;
      pNextAlignedObjects -> m_pNextAlignedObjects = 0;
      delete pNextAlignedObjects;
    }
  }

  return nNoOfObjects;
}

////////////////////////////////////////////////////////////////////////////////
