/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_PROGRAM
#define __GLIB_PROGRAM

#include "glib/primitives/aptr.h"
#include "glib/sys/GSystem.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/util/GSectionBag.h"
#include "glib/util/GLocaleData.h"
#include "glib/util/GLog.h"
#include "glib/resource/GRcException.h"
#include "glib/resource/GResourceTable.h"
#include "glib/GThread.h"
#include "glib/GEnvironment.h"

/**
 * This is the base class for G-Lib based programs.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.07
 */
class GProgram : public GObject
{
   private:

      /** The Application Name, used to define the location of configurations files, etc. */
      const GString applName;

      /** The environment variables of the program. */
      GEnvironment envVars;

      /** Various Country and Locale specific formating params. */
      GLocaleData countryData;

      /** Application specified Pre-Processor-Defines for the Resource Compiler. */
      GKeyBag<GString> ppdefinesForRcCompiler;

      /** Full path of the program (exe-file). */
      GString fullPath;

      /** The root directory of the program. */
      GString rootDirectory;

      /** The name of the executable program file (exe-file). */
      GString exeFileName;

      /** the name of the executable program file (exe-file) without extension. */
      GString exeModName;

      /** The full path to the resource file used by this program instance. */
      GString resourcePath;

      /** The full path to the locale specific resource file used by this program instance. */
      GString resourcePathLocale;

      /** The main resource table. */
      aptr<GResourceTable> resources;

      /** The ini-file manager object. */
      GSectionBag* ini;

      /** True while the run() method is on the stack. */
      bool isInsideRun;

      /** The result as returned by <i>mainStart</i>. */
      int result;

      /** Number of milliseconds used to load the main resource table. */
      ulonglong millisSpentOnMainResourceTableLoad;

      /** Number of milliseconds used to load the locale resource table. */
      ulonglong millisSpentOnLocaleResourceTableLoad;

      /** The system dependent PID of the application. */
      GSystem::ProcessID pid;

      /**
       * This variable will point to the program object of the current
       * application context.
       * @see #getProgram
       */
      static GProgram* Program;

      /** 
       * The set of parameters given by command line. 
       * Ownership responsibility is ours. 
       * @see #getCmdLineParameters
       */
      static class GCmdLineParameters* Params;

   public:

      /** The time millis when the application did initially start-up (at GLib-level). */
      static const ulonglong TimeMillisEarlyStartup;

   public:

      // These two variables are initialized by ::main().
      /** Anchor block. */
      static HAB hAB;
      /** Handle of the message queue. */
      static HMQ hMQ;

   protected:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       * @param   applName  The name of the application, as used in e.g. the 
       *                    default profile (INI-file) directory (at least 
       *                    on Windows).
       * @throws  GRcException if the resource table of the program can 
       *                       not be loaded.
       */
      GProgram ( const GString& applName = GString::Empty,
                 bool loadIni = true,
                 bool loadRc = true,
                 bool loadRcLocale = true,
                 const GKeyBag<GString>* ppdefinesForRcCompiler = null );

   public:

      virtual ~GProgram ();

   private:

      /** Disable the copy constructor. */
      GProgram ( const GProgram& );

      /** Disable the assignment operator. */
      void operator= ( const GProgram& );

   private:

      void checkLocaleTexts ( GKeyBag<GTextResource>& mainTexts,
                              GKeyBag<GTextResource>& localeTexts );

      virtual void run ();

   public:

      /**
       * Typically called by <i>mainStart()</i> after main window has been
       * created, to enter the main message loop of where to receive and
       * dispatch GUI messages.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.11
       * @param   breakSem  The message loop will check this boolean before
       *                    every attempt to get a new message and if it is 
       *                    true we will break the message loop and return.
       *                    If null is specified we will never check it and 
       *                    the message loop will not break until the 
       *                    system dependent "WM_QUIT" type of message 
       *                    occurs in the message queue.
       */
      void enterModalMessageLoop ( bool* breakSem = null );

      /**
       * Create and execute the specified dialog resource as
       * a modal dialog box.
       *
       * @author  Leif Erik Larsen
       * @since   2000.04.19
       * @param   ownerWin  Pointer to the owner window object, or null.
       * @param   dlgID     The ID of which Dialog Resource to execute.
       * @param   msgProc   Pointer to which Message Handler Object that
       *                    the dialog system shall call in order to let
       *                    the program handle dialog messages manually.
       * @return  A copy of the dialog dismiss argument string.
       * @throws  GExecuteDialogException if we fail to execute the dialog.
       *                    The dialog execution is considered to fail if
       *                    for instance the specified dialog does not exist,
       *                    or if any of the components of the dialog did call
       *                    {@link GDialogPanel#dismiss} during the handling
       *                    of the {@link GDialogMessageHandler#GM_INIDIALOG} 
       *                    message. However, if the dialog was dismissed by 
       *                    the program defined message handler of the dialog 
       *                    panel it self then we will not throw an exception 
       *                    because this is not considered an error but rather 
       *                    a cancelation.
       * @see     #makeDialog
       */
      GString executeDialog ( GWindow* ownerWin, 
                              const GString& dlgID, 
                              GDialogMessageHandler* msgProc );

      /**
       * Get a reference to a copy of the Application Name as 
       * was specified to the constructor of this class.
       * Note that the Application Name can be empty, if not 
       * defined by the application upon program object construction.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.02
       */
      const GString& getApplicationName () const;

      /**
       * Get a reference to the parsed command line parameter set that was 
       * given to the program from the operating system.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       */
      const class GCmdLineParameters& getCmdLineParameters () const;

      /**
       * Get a reference to the variable string of the specified
       * environment variable.
       */
      GString getEnvironmentVar ( const GString& varName, 
                                  const GString& defValue = GString::Empty ) const;

      GEnvironment& getEnvironmentVars ();

      /**
       * Get the fully qualified name of the application executable module.
       * This is usually the EXE-file, on Windows and OS/2.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.30
       */
      const GString& getFullRootPath () const;

      GSectionBag& getIniProperties () const;

      const GLocaleData& getLocaleData () const;

      /**
       * The subclass must override this method and return a reference
       * to the main application window.
       *
       * It is up to the application not to call this method until the
       * main window has been created and initialized.
       */
      virtual GWindow& getMainWindow () = 0;

      /**
       * Get the number of milliseconds used to load the locale 
       * resource table.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.30
       * @see     #getMillisSpentOnMainResourceTableLoad
       */
      ulonglong getMillisSpentOnLocaleResourceTableLoad () const;

      /**
       * Get the number of milliseconds used to load the main resource table.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.30
       * @see     #getMillisSpentOnLocaleResourceTableLoad
       */
      ulonglong getMillisSpentOnMainResourceTableLoad () const;

      /**
       * Get the system dependent process id (pid) of the application.
       * Note that on Windows this is the ProcessID, not the ProcessHandle.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.16
       */
      GSystem::ProcessID getProcessID () const;

      /**
       * Get a reference to the program object of the current
       * application context. If the program object has not yet been
       * instantiated then this method will throw a
       * {@link GIllegalStateException}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.02.09
       */
      static GProgram& GetProgram ();

      /** 
       * Get the full path to the resource file used by this program 
       * instance. The returned path is the absolute path to the resource 
       * table file that was used to load the resource table as is returned 
       * by {@link #getResourceTable}. That is; the absolute path to the 
       * file that was specified with the command line argument "-rcFile".
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.19
       */
      const GString& getResourcePath () const;

      /** 
       * Get the full path to the locale specific resource file used 
       * by this program instance. That is; the absolute path to the 
       * file that was specified with the command line 
       * argument "-rcFileLocale".
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.19
       */
      const GString& getResourcePathLocale () const;

      GResourceTable& getResourceTable () const;

      /**
       * Get the program result as returned by {@link #mainStart}.
       * If the {@link #mainStart} method has not yet returned 
       * then we will return 0.
       */
      int getResult () const;

      /**
       * Get a reference to the root directory string of the program file.
       *
       * That is the directory of where the program file it self is
       * contained and from where it was launched. The returned directory
       * will contain the terminating directory slash character as well
       * as the drive name.
       */
      const GString& getRootDirectory () const;

      /**
       * Return true if and only if the window color options are to be
       * automatically saved to the program profile upon saving options.
       *
       * The default implementation of this method simply always returns true.
       * User program code should probably override this method and return
       * a boolean that respects the user options.
       *
       * @see GWindow#writeProfile
       */
      virtual bool isAutoSaveColorOptions () const;

      /**
       * Return true if and only if the window font options are to be
       * automatically saved to the program profile upon saving options.
       *
       * The default implementation of this method simply always returns true.
       * User program code should probably override this method and return
       * a boolean that respects the user options.
       *
       * @see GWindow#writeProfile
       */
      virtual bool isAutoSaveFontOptions () const;

      /**
       * Return true if and only if the window "visibility state" options
       * are to be automatically saved to the program profile upon saving
       * options.
       *
       * The default implementation of this method simply always returns true.
       * User program code should probably override this method and return
       * a boolean that respects the user options.
       *
       * @see GWindow#writeProfile
       */
      virtual bool isAutoSaveVisibilityStateOptions () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2004.03.19
       * @see     #LoadText
       */
      static bool IsLoadableText ( const GString& textID );

      /**
       * Always return true, because the program is the main thread, 
       * and the main thread always runs as long as the program is alive.
       */
      bool isRunning () const;

      /**
       * Override this method if you want the application to use fancy
       * menubars by default. That is, menubars with icons (if any).
       * The default implementation return false by default.
       * @see #isUseFancyPopupMenues
       */
      virtual bool isUseFancyMenubars () const { return false; }

      /**
       * Override this method if you want the application to use fancy
       * popup menues by default. That is, popup menues with icons (if any).
       * The default implementation return false by default.
       * @see #isUseFancyMenubars
       */
      virtual bool isUseFancyPopupMenues () const;

      /**
       * Compile and load an INI-file and return it as a new instance
       * of <i>GSectionBag</i>.
       *
       * The INI-file must be located in the root directory of the
       * application, or in the path which is pointed to by an
       * environment variable or a command line parameter.
       *
       * We will check for the command line parameter first. If that
       * variable is not specified then we will check the environment
       * variable. If the environment variable is not defined either,
       * or if the referenced path does not exist, then we will try to
       * open the INI-file in the root directory of the program, using
       * the filename as specified in <i>iniFile.paramValue</i>.
       *
       * If the INI-file is not found there either then we do nothing
       * but return a new and empty instance of <i>GSectionBag</i>.
       */
      virtual GSectionBag* loadIniProperties ( const class GProgramParameter& iniFile );

      /**
       * Compile and load an RC-file and return it as a new instance
       * of {@link GResourceTable}.
       *
       * The RC-file must be located in the root directory of the
       * application, or in the path which is pointed to by an
       * environment variable or a command line parameter.
       *
       * We will check for the command line parameter first. If that
       * variable is not specified then we will check the environment
       * variable. If the environment variable is not defined either, or if
       * the referenced path does not exist, then we will try to open the
       * RC-file in the root directory of the program, using the filename
       * as specified in <i>pps.paramValue</i>.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.19
       * @param   pps         Specify the properties needed to find the 
       *                      rc-file of which to load. This information
       *                      typically originates from the command line 
       *                      used to start the application.
       * @param   searchPath  The semicolon separated list of directories 
       *                      of where to search for the rc-files during 
       *                      loading. This search path is also used for
       *                      loading any #include-modules.
       * @param   pathLoaded  Returns the path of the loaded rc-file.
       *                      If this argument is null we will not 
       *                      touch it at all.
       * @param   millisSpent Returns the number olf milliseconds used to 
       *                      load the resources, if != null.
       * @throws  GRcException in case of any error compiling the RC-file.
       */
      aptr<GResourceTable> loadResourceTable ( const class GProgramParameter& pps, 
                                               const class GSearchPath& searchPath,
                                               GString* pathLoaded = null,
                                               ulonglong* millisSpent = null );

      /**
       * Load the specified text from the resource table of the 
       * application. This method is just a convenient wrapper 
       * for {@link GResourceTable#loadText}
       * 
       * @author  Leif Erik Larsen
       * @since   2004.03.19
       * @see     #IsLoadableText
       * @see     GResourceTable#loadText
       */
      static GString LoadText ( const char* sourceStr, 
                                GResourceTable::LoadTextFlags flags = GResourceTable::LT_PREFER_TEXT, 
                                int maxLength = 0 );

      /**
       * This is the default main entry point of the program.
       * Most programs will override this method in order to perform
       * its task, which is typically something like a main loop.
       *
       * The default implementation of this method does nothing but
       * return a zero.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       * @see    #getCmdLineParameters
       */
      virtual int mainStart ();

      /**
       * Create a so-called dynamic dialog.
       *
       * A dynamic dialog is a dialog which definition doesn't have to
       * exist in a resource table as standard dialogs do in most cases.
       * A dynamic dialog can be dynamically defined by the program during
       * runtime. Another easy feature of a dynamic dialog returned from
       * this method is that it can be easily reexecuted after it has
       * been dismissed.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.04
       * @param   dlgID     The name of which Dialog Resource to use as 
       *                    template, or an empty string to make an empty 
       *                    dialog.
       * @param   msgProc   Reference to which message processing routine that
       *                    the dialog system shall call to let program handle
       *                    dialog messages manually.
       * @return  An auto-pointer to an allocated instance
       *          of {@link GDialogFrame} that will automatically be destryed
       *          when the auto-pointer goes out of scope.
       * @throws  GNoSuchResourceException if the specified dialog resource 
       *                    can not be found.
       * @see     #executeDialog
       * @see     GDialogPanel#executeModal
       * @see     GDialogPanel#executeModeless
       */
      aptr<GDialogFrame> makeDialog ( const GString& dlgID = GString::Empty,
                                      GDialogMessageHandler* msgProc = null );

      /**
       * Method that is probably used mostly for printing of debug 
       * text messages to some kind of output. The default implementation 
       * of this method prints it to nowhere, but subclasses can override 
       * this so that the text is printed anywhere else.
       * 
       * This method can be used as an alternative to {@link GLog}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.14
       * @param   str  The text of which to print. Can contain format tags
       *               for the optional arguments container. See 
       *               {@link GVArgs#FormatArgs} for details regarding the 
       *               formatting tags.
       * @param   args The list of variable arguments that are to be
       *               joined into the format string in a formatted way.
       * @see     GLog#Log
       */
      virtual void printF ( const GString& str, const GVArgs& args = GVArgs() );

      /**
       * This static method is public as an implementation side effect and 
       * should not be used by any user code. It is called initialle by the 
       * library startup and initialization code to parse the command line 
       * arguments into a more user friendly container object that is to be 
       * accessed by user code through {@link #getCmdLineParameters}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       * @param   params    The new command line parameters list. Must have 
       *                    been allocated from the heap with operator "new".
       *                    We takes the ownership responsibility of 
       *                    the object.
       */
      static void SetCmdLineParameters ( class GCmdLineParameters* params );

      /**
       * Call {@link #start} and return the value from {@link #getResult}.
       */
      virtual int runTheProgram ();

      /**
       * Set the specified environment variable name and value.
       *
       * For example: <i>setEnvironmentVar("VARIABLE=YES")</i>.
       */
      void setEnvironmentVar ( const GString& varNameAndValue );

      /**
       * Same as {@link GWindow#showMessageBox}, but here the message box
       * will be displayed without a modal parent window.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.28
       * @see     GWindow#showMessageBox
       * @see     GMessageBox
       */
      GMessageBox::Answer showMessageBox ( const GString& msg, 
                                           GMessageBox::Type type = GMessageBox::TYPE_NONE, 
                                           const GString& flags = GString::Empty, 
                                           const GString& title = GString::Empty, 
                                           bool monoFont = false,
                                           const GString& userText1 = GString::Empty,
                                           const GString& userText2 = GString::Empty,
                                           const GString& userText3 = GString::Empty );

      /**
       * Just call {@link GThread#run}, synchronously.
       */
      virtual void start ();

      /**
       * Overrides <i>GThread::stop()</i>, and does nothing but return.
       */
      virtual void stop ();

      GString toString () const;

      /**
       * This method is automatically called just after {@link #mainStart} 
       * has finished, but only if it finish normally (no exception).
       *
       * The default implementation of this method will write the  user
       * profile (ini-file) of the program, in case it has been modified.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       */
      virtual void writeUserProfile ();
};

#endif
