/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/GProgram.h"
#include "glib/GProgramParameter.h"
#include "glib/vfs/GFile.h"
#include "glib/util/GTokenizer.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GNoSuchResourceException.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/primitives/GLong.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GExecuteDialogException.h"
#include "glib/gui/GMessageBox.h"

GProgram::GProgram ( const GString& applName,
                     bool loadIni, 
                     bool loadRc, 
                     bool loadRcLocale,
                     const GKeyBag<GString>* ppdefinesForRcCompiler )
         :applName(applName),
          ini(null),
          isInsideRun(false),
          result(0),
          millisSpentOnMainResourceTableLoad(0),
          millisSpentOnLocaleResourceTableLoad(0),
          pid(0)
{
   if (GProgram::Program != null)
      gthrow_(GIllegalStateException("GProgram is a singleton that can be instantiated only once!"));
   GProgram::Program = this;
   if (ppdefinesForRcCompiler != null)
      this->ppdefinesForRcCompiler = *ppdefinesForRcCompiler;

   // Keep a copy of the root directory of the application.
   TIB* tib;
   PIB* pib;
   ::DosGetInfoBlocks(&tib, &pib);
   char moduleName[GFile::MAXPATH];
   ::DosQueryModuleName(pib->pib_hmte, sizeof(moduleName), moduleName);
   this->fullPath = moduleName;
   this->rootDirectory = fullPath;
   this->pid = pib->pib_ulpid;
   GFile::CutFileName(rootDirectory);

   // Also keep a copy of the program executable filename (exe-file).
   exeFileName = GFile::GetFileName(fullPath);
   exeModName = GFile::GetName(fullPath);
   // OS/2's DosQueryModuleName() gives a string where all characters are
   // uppercase. Assume most users want to see lowercase filenames instead.
   exeModName.toLowerCase();

   // Find the default location and filename for the ini-file.
   GString userHomeDir = GSystem::GetUserApplicationDataDirectory(applName);
   GString defIni("%s%s.ini", GVArgs(userHomeDir).add(exeModName));

   // The exe-filename forms the template for the properties that defines
   // the location of our ini-, rc- and log-file, either via environment
   // variable, program command line argument or a default value.
   GVArgs arg1(exeModName);
   GProgramParameter iniFile(*this, GString("%s_iniFile", arg1), "-iniFile", defIni);
   GProgramParameter rcSearchPath(*this, GString("%s_rcSearchPath", arg1), "-rcSearchPath", "");
   GProgramParameter rcFile(*this, GString("%s_rcFile", arg1), "-rcFile", GString("%s%s.rc", GVArgs(rootDirectory).add(exeModName)));
   GProgramParameter rcFileLocale(*this, GString("%s_rcFileLocale", arg1), "-rcFileLocale", GString::Empty);
   GProgramParameter logFilter(*this, GString("%s_logFilter", arg1), "-logFilter", "prod");

   // GLog: We are also responsible to initialize the static GLog context.
   // This is why we are declared to be a friend class of GLog.
   GProgramParameter logFileOpts(*this, GString("%s_logFile", arg1), "-logFile", GString("%s.log", arg1));
   GLog::OpenLogFile(logFileOpts);
   GString opt = logFilter.getValueAsString(*Params);
   if (opt.beginsWith("DEBUG", true))
      GLog::FilterLevel = GLog::DEBUG;
   else
   if (opt.beginsWith("TEST", true))
      GLog::FilterLevel = GLog::TEST;
   else
   if (opt.beginsWith("PROD", true))
      GLog::FilterLevel = GLog::PROD;
   else
   if (opt.beginsWith("OFF", true) || opt.beginsWith("MAX", true))
      GLog::FilterLevel = GLog::LOGGING_IS_OFF;
   else
      GLog::FilterLevel = GLog::PROD;

   // ---
   if (loadIni)
      ini = loadIniProperties(iniFile);
   if (ini == null)
      ini = new GSectionBag();

   // ---
   // Load the RC-file.
   GSearchPath rcspath(rcSearchPath.getValueAsString(*Params));
   if (loadRc)
      resources = loadResourceTable(rcFile, rcspath, &resourcePath, &millisSpentOnMainResourceTableLoad);
   if (resources.get() == null)
      resources = aptr<GResourceTable>(new GResourceTable());

   // ---
   // Load locale specific text resources (if any)
   // and let them override the default texts.
   if (loadRcLocale)
   {
      aptr<GResourceTable> resLocale = loadResourceTable(rcFileLocale, rcspath, &resourcePathLocale, &millisSpentOnLocaleResourceTableLoad);
      GKeyBag<GTextResource>& mainTexts = resources->getTextBag();
      GKeyBag<GTextResource>& localeTexts = resLocale->getTextBag();
      const int num = localeTexts.getCount();
      if (num > 0) // If any locale specific texts was loaded at all.
         checkLocaleTexts(mainTexts, localeTexts); // Log ID of missing locale texts, if any.
      for (int i=num-1; i>=0; i--)
      {
         GTextResource& txt = localeTexts.getIndexedItem(i);
         const GString& id = txt.getIDString();
         localeTexts.removeIndexedItem(i, false);
         mainTexts.update(id, &txt, true);
      }
   }
}

GProgram::~GProgram ()
{
   delete ini;
   GProgram::Program = null;
}

void GProgram::SetCmdLineParameters ( GCmdLineParameters* params )
{
   GProgram::Params = params;
}

const GCmdLineParameters& GProgram::getCmdLineParameters () const
{
   return *GProgram::Params;
}

const GString& GProgram::getApplicationName () const
{
   return applName;
}

ulonglong GProgram::getMillisSpentOnMainResourceTableLoad () const
{
   return millisSpentOnMainResourceTableLoad;
}

ulonglong GProgram::getMillisSpentOnLocaleResourceTableLoad () const
{
   return millisSpentOnLocaleResourceTableLoad;
}

void GProgram::checkLocaleTexts ( GKeyBag<GTextResource>& mainTexts,
                                  GKeyBag<GTextResource>& localeTexts )
{
   if (!GLog::Filter(GLog::TEST))
      return;

   // Check that all texts are overriden. Log those who are not.
   const int numMainTexts = mainTexts.getCount();
   for (int i=0; i<numMainTexts; i++)
   {
      const GString& id = mainTexts.getIndexedItem(i).getIDString();
      if (!localeTexts.containsKey(id))
         GLog::Log(this, "Text '%s' not overridden.", GVArgs(id));
   }

   // Check that all locale specific texst are also defined in standard resource table.
   const int numLocaleTexts = localeTexts.getCount();
   for (int i=0; i<numLocaleTexts; i++)
   {
      const GString& id = localeTexts.getIndexedItem(i).getIDString();
      if (!mainTexts.containsKey(id))
         GLog::Log(this, "Locale specific text '%s' is not part of the standard text resources.", GVArgs(id));
   }
}

int GProgram::runTheProgram ()
{
   start();
   return getResult();
}

void GProgram::start ()
{
   run();
}

void GProgram::stop ()
{
   return;
}

GSystem::ProcessID GProgram::getProcessID () const
{
   return pid;
}

const GLocaleData& GProgram::getLocaleData () const
{
   return countryData;
}

bool GProgram::isRunning () const
{
   return true;
}

void GProgram::writeUserProfile ()
{
   try {
      GSectionBag& ini = getIniProperties();
      ini.write();
   } catch (GIOException& e) {
      GString msg = e.toString();
      GString stackTrace = e.getStackTrace(msg);
      GLog::Log(this, stackTrace);
   }
}

void GProgram::run ()
{
   if (isInsideRun)
      return; // Should never happen, but in case

   isInsideRun = true;
   result = 0;

   // Call the main entry point of the program, to the user program point
   // of view. Protect the call in a low-level and system-dependent exception
   // handler, so that a log-statement is written in case the program
   // thread traps.
   bool ok = true;
   GTRY(e, true) {
      result = mainStart();
      writeUserProfile();
   } GCATCH(e) {
      ok = false;
   } GENDCATCH();
   isInsideRun = false;
   if (!ok)
      gthrow_(GException("System trap in application main thread."));
}

int GProgram::getResult () const
{
   if (isRunning())
      return 0;
   else
      return result;
}

GProgram& GProgram::GetProgram ()
{
   if (Program == null)
      gthrow_(GIllegalStateException("GProgram not instantiated!"));
   return *Program;
}

GString GProgram::toString () const
{
   return fullPath;
}

const GString& GProgram::getResourcePath () const
{
   return resourcePath;
}

const GString& GProgram::getResourcePathLocale () const
{
   return resourcePathLocale;
}

GResourceTable& GProgram::getResourceTable () const
{
   return *resources.get();
}

bool GProgram::isAutoSaveColorOptions () const
{
   return true;
}

bool GProgram::isAutoSaveFontOptions () const
{
   return true;
}

bool GProgram::isAutoSaveVisibilityStateOptions () const
{
   return true;
}

bool GProgram::isUseFancyPopupMenues () const
{
   return false;
}

int GProgram::mainStart ()
{
   return 0;
}

void GProgram::enterModalMessageLoop ( bool* breakSem )
{
   bool dummyFalse = false;
   bool* bsem = (breakSem == null ? &dummyFalse : breakSem);

   QMSG msg;
   while (!*bsem && ::WinGetMsg(hAB, &msg, 0, 0, 0))
   {
      ::WinDispatchMsg(hAB, &msg);
   }
}

bool GProgram::IsLoadableText ( const GString& textID )
{
   return GResourceTable::IsLoadableText(textID);
}

GString GProgram::LoadText ( const char* sourceStr,
                             GResourceTable::LoadTextFlags flags,
                             int maxLength )
{
   const GProgram& prg = GetProgram();
   const GResourceTable* res = prg.resources.get();
   if (res == null)
      return sourceStr;
   return res->loadText(sourceStr, flags, maxLength);
}

void GProgram::printF ( const GString& /*str*/, const GVArgs& /*args*/ )
{
   // Yes, the default implementation is documented to print to nowhere!
}

const GString& GProgram::getRootDirectory () const
{
   return rootDirectory;
}

const GString& GProgram::getFullRootPath () const
{
   return fullPath;
}

aptr<GResourceTable> GProgram::loadResourceTable ( const GProgramParameter& pps, 
                                                   const GSearchPath& searchPath,
                                                   GString* pathLoaded,
                                                   ulonglong* millisSpent )
{
   if (pathLoaded != null)
      *pathLoaded = "";

   GString path = pps.getValueAsString(*Params);
   if (path != "")
   {
      // Make the path "absolute".
      GString absPath;
      if (GFile::ContainsDrive(path))
      {
         absPath = path;
      }
      else
      {
         GVfsLocal vfs;
         absPath = vfs.getCurrentDirectory(false);
         if (!vfs.walkPath(absPath, path))
            absPath = path;
      }
      if (pathLoaded != null)
         *pathLoaded = absPath;

      ulonglong timeStart = GSystem::CurrentTimeMillis();
      GRcTokenizer tokenizer(absPath, true, searchPath, &ppdefinesForRcCompiler);

      // Load the resource table script.
      aptr<GResourceTable> ret = GRcCompiler::ParseResources(tokenizer);
      if (millisSpent != null)
         *millisSpent = GSystem::CurrentTimeMillis() - timeStart;

      return ret;
   }
   else
   {
      // The program does probably not require any resources, but return
      // an empty default in order to prevent null-pointer exceptions.
      if (millisSpent != null)
         *millisSpent = 0;
      GResourceTable* res = new GResourceTable();
      return aptr<GResourceTable>(res);
   }
}

GSectionBag* GProgram::loadIniProperties ( const GProgramParameter& iniFile )
{
   GString path;
   GFile ini(iniFile.getValueAsString(*Params));
   if (ini.isDirectoryFromRoot())
   {
      path = ini.getFullPath();
   }
   else
   {
      GString tmp = GFile::GetCurrentDir();
      GFile::Slash(tmp);
      tmp += ini.getFullPath();
      GVfsLocal vfs;
      vfs.getWalkedPath(tmp, path);
   }
   if (path == "")
      return new GSectionBag();
   else
      return new GSectionBag(path);
}

void GProgram::setEnvironmentVar ( const GString& varNameAndValue )
{
   if (GLog::Filter(GLog::TEST))
      GLog::Log(this, "%s: varNameAndValue=\"%s\"", GVArgs(__FUNCTION__).add(varNameAndValue));
   envVars.setEnv(varNameAndValue);
}

GString GProgram::getEnvironmentVar ( const GString& varName, const GString& defValue ) const
{
   if (varName == "LIBPATH")
      return GSystem::GetLibPath();
   return envVars.getEnv(varName, defValue);
}

GEnvironment& GProgram::getEnvironmentVars ()
{
   return envVars;
}

GSectionBag& GProgram::getIniProperties () const
{
   return *ini;
}

GString GProgram::executeDialog ( GWindow* ownerWin,
                                  const GString& dlgID,
                                  GDialogMessageHandler* msgProc )
{
   GResourceTable& rtbl = getResourceTable();
   GDialogResource* dlgRes = rtbl.getDialogResource(dlgID);
   if (dlgRes == null)
      gthrow_(GExecuteDialogException(GStringl("No such dialog resource: \"%s\"", GVArgs(dlgID))));
   try {
      GDialogFrame dlgbox(msgProc, *dlgRes);
      GString dismissArg = dlgbox.executeModal(ownerWin);
      return dismissArg;
   } catch (std::exception& e) {
      gthrow_(GExecuteDialogException(e.what()));
   }
}

aptr<GDialogFrame> GProgram::makeDialog ( const GString& dlgID,
                                          GDialogMessageHandler* msgProc )
{
   aptr<GDialogResource> dlgRes;
   if (dlgID == "")
   {
      dlgRes = aptr<GDialogResource>(new GDialogResource(GString::Empty));
   }
   else
   {
      GResourceTable& res = getResourceTable();
      dlgRes = aptr<GDialogResource>(res.getDialogResource(dlgID));
      if (dlgRes.get() == null)
         gthrow_(GNoSuchResourceException(GString("No such dialog resource: \"%s\"", GVArgs(dlgID))));
      dlgRes.release(); // Don't destroy the contained object on return!
   }
   return aptr<GDialogFrame>(new GDialogFrame(msgProc, *dlgRes));
}

GMessageBox::Answer GProgram::showMessageBox ( const GString& msg, 
                                               GMessageBox::Type type, 
                                               const GString& flags, 
                                               const GString& title, 
                                               bool monoFont,
                                               const GString& userText1,
                                               const GString& userText2,
                                               const GString& userText3 )
{
   GMessageBox mbox(type, msg, title, flags, monoFont, userText1, userText2, userText3);
   GMessageBox::Answer answ = mbox.show(null);
   return answ;
}
