package jp.co.sra.smalltalk.menu;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

/**
 * StMenuItemViewSwing class
 * 
 *  @author    nisinaka
 *  @created   2005/02/17 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StMenuItemViewSwing.java,v 8.9 2008/02/20 06:33:36 nisinaka Exp $
 */
public class StMenuItemViewSwing extends StObject implements StMenuItemView {

	protected StMenuItem menuItem;
	protected JComponent menuItemView;

	/**
	 * Create a new instance of StMenuItemViewSwing and initialize it.
	 *
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @category Instance creation
	 */
	public StMenuItemViewSwing(StMenuItem aMenuItem) {
		this.initialize();
		this.setMenuItem(aMenuItem);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		menuItem = null;
		menuItemView = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.setMenuItem(null);
		super.release();
	}

	/**
	 * Answer my current menu item.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuItem
	 * @see jp.co.sra.smalltalk.menu.StMenuItemView#menuItem()
	 * @category model accessing
	 */
	public StMenuItem getMenuItem() {
		return menuItem;
	}

	/**
	 * Set my new menu itme.
	 * 
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @category model accessing
	 */
	public void setMenuItem(StMenuItem aMenuItem) {
		if (menuItem != null) {
			menuItem.removeDependentListener(this);
		}
		menuItem = aMenuItem;
		if (menuItem != null) {
			menuItem.addDependentListener(this);
		}
	}

	/**
	 * Answer my window.
	 * If not created nor added yet, return null.
	 * 
	 * @return java.awt.Window
	 * @category accessing
	 */
	public Window getWindow() {
		return (menuItemView == null) ? null : SwingUtilities.getWindowAncestor(menuItemView);
	}

	/**
	 * Answer the menu item view.
	 * 
	 * @return javax.swing.JComponent
	 * @category user interface
	 */
	protected JComponent menuItemView() {
		if (menuItemView == null) {
			menuItemView = this.createMenuItemView();
			this.updateMenuItemView();
		}
		return menuItemView;
	}

	/**
	 * Answer the menu item view as JMenuItem.
	 * 
	 * @return javax.swing.JMenuItem
	 * @category converting
	 */
	public JMenuItem toMenuItem() {
		return (JMenuItem) this.menuItemView();
	}

	/**
	 * Update for the change notice.
	 * 
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aspect = evt.getAspect();
		if (aspect == $("enabled")) {
			this.updateEnabled();
		} else if (aspect == $("label")) {
			this.updateLabel();
		} else if (aspect == $("labelImage")) {
			this.updateLabelImage();
		}
	}

	/**
	 * Update the enabled of the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateEnabled() {
		this.toMenuItem().setEnabled(this.getMenuItem().isEnabled());
	}

	/**
	 * Update the label of the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateLabel() {
		this.toMenuItem().setText(this.getMenuItem().labelString());
	}

	/**
	 * Update the label image of the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateLabelImage() {
		if (this.getMenuItem().labelImage() == null) {
			this.toMenuItem().setIcon(null);
		} else {
			this.toMenuItem().setIcon(new ImageIcon(this.getMenuItem().labelImage()));
		}
	}

	/**
	 * Update the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateMenuItemView() {
		this.updateEnabled();
		this.updateLabel();
		this.updateLabelImage();
	}

	/**
	 * Create a menu item view.
	 * 
	 * @return javax.swing.JComponent
	 * @category private
	 */
	protected JComponent createMenuItemView() {
		JMenuItem aMenuItem = new JMenuItem();
		aMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				StMenuItemViewSwing.this.performMenu($("action"), e);
			}
		});

		if (this.getMenuItem().shortcutKeyCharacter() > 0) {
			int keyCode = this.getMenuItem().shortcutKeyCharacter();
			int modifiers = this.getMenuItem().shortcutModifiers();
			aMenuItem.setAccelerator(KeyStroke.getKeyStroke(keyCode, modifiers));
		}

		return aMenuItem;
	}

	/**
	 * Perform for the menu action.
	 * 
	 * @param aspect jp.co.sra.smalltalk.StSymbol
	 * @param parameter java.lang.Object
	 * @category private
	 */
	protected void performMenu(StSymbol aspect, Object parameter) {
		MenuEvent aMenuEvent = new MenuEvent(this, aspect, parameter);
		this.getMenuItem().performMenu(aMenuEvent);
	}

	/**
	 * Create an appropriate view for the menu item.
	 * 
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @return jp.co.sra.smalltalk.menu.StMenuItemViewSwing
	 * @category Utilities
	 */
	protected static StMenuItemViewSwing _New(StMenuItem aMenuItem) {
		if (aMenuItem == null) {
			return null;
		}

		Class aClass = null;
		try {
			aClass = Class.forName(aMenuItem.getClass().getName() + "ViewSwing");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}

		return (aClass == null) ? null : (StMenuItemViewSwing) StObject._New(aClass, aMenuItem);
	}

}
