package jp.co.sra.smalltalk;

import java.util.Collection;
import java.util.Vector;

/**
 * StSortedCollection class
 * 
 *  @author    nisinaka
 *  @created   1998/12/02 (by nisinaka)
 *  @updated   1999/08/05 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StSortedCollection.java,v 8.11 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StSortedCollection extends StObject {

	/** The block closure for comparing two elements. */
	protected StBlockClosure sortBlock = null;

	/** The collection which holds the elements in order. */
	protected Vector _elements = new Vector();

	/**
	 * Create a new instance of StSortedCollection and initialize it.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Instance creation
	 */
	public StSortedCollection(StBlockClosure aBlock) {
		this.sortBlock_(aBlock);
	}

	/**
	 * Create a new instance of StSortedCollection and initialize it.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockCosure
	 * @param anArray java.lang.Object[]
	 * @category Instance creation
	 */
	public StSortedCollection(StBlockClosure aBlock, Object[] anArray) {
		this(aBlock);

		for (int i = 0; i < anArray.length; i++) {
			this.add_(anArray[i]);
		}
	}

	/**
	 * Create a new instance of StSortedCollection with a sort block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aCollection java.util.Vector
	 * @category Instance creation
	 */
	public StSortedCollection(StBlockClosure aBlock, Vector aCollection) {
		this(aBlock, aCollection.toArray());
	}

	/**
	 * Create a new instance of StSortedCollection and initialize it.
	 * Should not be created without a sorted block.
	 *
	 * @category Instance creation
	 */
	private StSortedCollection() {
	}

	/**
	 * Set the sort block of the receiver.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void sortBlock_(StBlockClosure aBlock) {
		sortBlock = aBlock;
	}

	/**
	 * Answer the element at the specified index.
	 * 
	 * @param index int
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object at_(int index) {
		return _elements.get(index);
	}

	/**
	 * Answer the first element of the collection.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object first() {
		return _elements.firstElement();
	}

	/**
	 * Answer the last element of the collection.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object last() {
		return _elements.lastElement();
	}

	/**
	 * Answer my size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return _elements.size();
	}

	/**
	 * Add the new object as one of the receiver's element.
	 * 
	 * @param newObject java.lang.Object
	 * @return java.lang.Object
	 * @category adding
	 */
	public Object add_(Object newObject) {
		if (_elements.isEmpty()) {
			_elements.addElement(newObject);
		} else {
			int nextIndex = this.indexForInserting_(newObject);
			_elements.insertElementAt(newObject, nextIndex);
		}
		return newObject;
	}

	/**
	 * Answer the receiver's elements as array.
	 * 
	 * @return java.lang.Object[]
	 * @category converting
	 */
	public Object[] _asArray() {
		return _elements.toArray();
	}

	/**
	 * Answer the receiver's elements as a collection.
	 * 
	 * @return java.util.Collection
	 * @category converting
	 */
	public Collection _asCollection() {
		return new Vector(_elements);
	}

	/**
	 * Compare the two elements.
	 * 
	 * @param e1 java.lang.Object
	 * @param e2 java.lang.Object
	 * @return boolean
	 * @category private
	 */
	protected boolean compareElement_to_(Object e1, Object e2) {
		if (sortBlock == null) {
			throw new SmalltalkException("No sort block.");
		}
		return ((Boolean) sortBlock.value_value_(e1, e2)).booleanValue();
	}

	/**
	 * Answer the index for inserting the new object.
	 * 
	 * @param newObject java.lang.Object
	 * @return int
	 * @category private
	 */
	protected int indexForInserting_(Object newObject) {
		int low = 0;
		int high = _elements.size() - 1;
		int index = (high + low) / 2;
		while (low <= high) {
			if (this.compareElement_to_(_elements.elementAt(index), newObject)) {
				low = index + 1;
			} else {
				high = index - 1;
			}
			index = (high + low) / 2;
		}
		return low;
	}

}
