package jp.co.sra.smalltalk;

import java.io.*;

/**
 * StReadStream class
 * 
 *  @author    nisinaka
 *  @created   1999/06/21 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StReadStream.java,v 8.9 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StReadStream extends StObject {

	/** The string as a source stream. */
	protected String source = null;

	/** The current reading position within a source string. */
	protected int position;

	/** The length of the source string. */
	protected int length;

	/**
	 * The default constructor.
	 * 
	 * @category Instance creation
	 */
	public StReadStream() {
		this("");
	}

	/**
	 * Create a new instance of StReadStream with a string as a source.
	 * 
	 * @param aString java.lang.String
	 * @category Instance creation
	 */
	public StReadStream(String aString) {
		this.on_(aString);
	}

	/**
	 * Create a new instance of StReadStream with a reader.
	 * 
	 * @param aReader java.io.Reader
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public StReadStream(Reader aReader) throws IOException {
		if (aReader != null) {
			BufferedReader bufferedReader = (aReader instanceof BufferedReader) ? (BufferedReader) aReader : new BufferedReader(aReader);
			StringBuffer stringBuffer = new StringBuffer();
			int ch;
			while ((ch = bufferedReader.read()) != -1) {
				stringBuffer.append((char) ch);
			}
			this.on_(stringBuffer.toString());
		}
	}

	/**
	 * Create a new instance of StReadStream with a file.
	 * 
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public StReadStream(File aFile) throws IOException {
		this((aFile == null) ? (Reader) null : new FileReader(aFile));
	}

	/**
	 * Answer the copied contents of the receiver.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String contents() {
		return new String(source);
	}

	/**
	 * Answer the character at the current position and change the position.
	 * 
	 * @return char
	 * @category accessing
	 */
	public char next() {
		return source.charAt(position++);
	}

	/**
	 * Answer the next anInteger elements of the receiver. If there are not
	 * enough elements available, answer a collection of as many as are
	 * available.
	 * 
	 * @param anInteger int
	 * @return java.lang.String
	 * @category accessing
	 */
	public String nextAvailable_(int anInteger) {
		int to = Math.min(this.length, this.position() + anInteger);
		int size = to - this.position();
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < size; i++) {
			buffer.append(this.next());
		}
		return buffer.toString();
	}

	/**
	 * Answer the character at the current position, but not change the position.
	 * 
	 * @return char
	 * @category accessing
	 */
	public char peek() {
		if (this.atEnd()) {
			return 0;
		}
		return source.charAt(position);
	}

	/**
	 * Answer true and increment the position if the next character equals to ch.
	 * 
	 * @param ch char
	 * @return boolean
	 * @category accessing
	 */
	public boolean peekFor_(char ch) {
		if (this.atEnd()) {
			return false;
		}

		char nextChar = this.next();
		if (ch == nextChar) {
			return true;
		}

		this.skip_(-1);
		return false;
	}

	/**
	 * Answer the substring from the current position to the specified character.
	 * 
	 * @param theCharacter char
	 * @return java.lang.String
	 * @category accessing
	 */
	public String upTo_(char theCharacter) {
		StringBuffer buffer = new StringBuffer(64);
		while (this.atEnd() == false) {
			char ch = this.next();
			if (ch == theCharacter) {
				return buffer.toString();
			}
			buffer.append(ch);
		}
		return buffer.toString();
	}

	/**
	 * Answer the substring from the current position to the end.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String upToEnd() {
		return source.substring(position, length);
	}

	/**
	 * Answer true if the position is at the end.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean atEnd() {
		return (position >= length);
	}

	/**
	 * Set the status of the stream to be closed.
	 * 
	 * @category status
	 */
	public void close() {
		// nothing to do 
	}

	/**
	 * Answer the current reading position.
	 * 
	 * @return int
	 * @category positioning
	 */
	public final int position() {
		return position;
	}

	/**
	 * Set the current position.
	 * 
	 * @param newPosition int
	 * @throws SmalltalkException 
	 * @category positioning
	 */
	public void position_(int newPosition) {
		if ((0 <= newPosition) && (newPosition <= source.length())) {
			position = newPosition;
		} else {
			throw new SmalltalkException("positionOutOfBoundsError: " + newPosition);
		}
	}

	/**
	 * Set the position of the receiver to the end of its stream of elements.
	 * 
	 * @category positioning
	 */
	public void setToEnd() {
		position = length;
	}

	/**
	 * Set position to position + offset.
	 * 
	 * @param offset int
	 * @category positioning
	 */
	public void skip_(int offset) {
		this.position_(position + offset);
	}

	/**
	 * Returns a String that represents the value of this object.
	 * 
	 * @return a string representation of the receiver
	 * @category printing
	 */
	public String toString() {
		// Insert code to print the receiver here.
		// This implementation forwards the message to super. You may replace or supplement this.
		return super.toString();
	}

	/**
	 * Initialize the receiver with a string.
	 * 
	 * @param aString java.lang.String
	 * @category private
	 */
	protected void on_(String aString) {
		length = aString.length();
		char[] charArray = new char[length + 1];
		System.arraycopy(aString.toCharArray(), 0, charArray, 0, length);
		charArray[length] = 0;
		source = new String(charArray);
		position = 0;
	}

}
