package jp.co.sra.smalltalk;

import java.util.ArrayList;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * StMessageCatalog class
 * 
 *  @author    nisinaka
 *  @created   2004/03/18 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StMessageCatalog.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StMessageCatalog extends StObject {

	protected ResourceBundle resourceBundle;

	public static final ArrayList Catalogs = new ArrayList();

	// This message catalog always returns the default string of user messages.
	public static final StMessageCatalog DefaultMessageCatalog = new StMessageCatalog((ResourceBundle) null) {
		public String lookUp(StUserMessage aUserMessage) {
			if (aUserMessage.catalog() != null && aUserMessage.catalog() != this) {
				return null;
			}
			String aString = aUserMessage.defaultString();
			if (aString == null) {
				aString = aUserMessage.key();
			}
			return aString;
		}
	};

	/**
	 * Answer an instance of StMessageCatalog which corresponds with the specified ResourceBundle.
	 * 
	 * @param aResourceBundle java.util.ResourceBundle
	 * @return jp.co.sra.smalltalk.StMessageCatalog
	 * @category Instance creation
	 */
	public static StMessageCatalog With(ResourceBundle aResourceBundle) {
		for (int i = 0; i < Catalogs.size(); i++) {
			StMessageCatalog aMessageCatalog = (StMessageCatalog) Catalogs.get(i);
			if (aMessageCatalog.resourceBundle == aResourceBundle) {
				return aMessageCatalog;
			}
		}
		return new StMessageCatalog(aResourceBundle);
	}

	/**
	 * Answer an instance of StMessageCatalog which corresponds with the specified name of ResourceBundle.
	 * 
	 * @param aBundleName java.lang.String
	 * @return jp.co.sra.smalltalk.StMessageCatalog
	 * @throws java.util.MissingResourceException
	 * @category Instance creation
	 */
	public static StMessageCatalog With(String aBundleName) {
		return With(aBundleName, Locale.getDefault());
	}

	/**
	 * Answer an instance of StMessageCatalog which corresponds with the specified name of ResourceBundle.
	 * 
	 * @param aBundleName java.lang.String
	 * @param aLocale java.util.Locale
	 * @return jp.co.sra.smalltalk.StMessageCatalog
	 * @throws java.util.MissingResourceException
	 * @category Instance creation
	 */
	public static StMessageCatalog With(String aBundleName, Locale aLocale) {
		ResourceBundle aResourceBundle = ResourceBundle.getBundle(aBundleName, aLocale);
		if (aResourceBundle.getLocale().equals(aLocale) == false) {
			// Needs to throw a MissingResourceException even if a resource bundle for the default locale exists.
			// On Mac OS X, ResourceBundle.getBundle () method automatically throws this expception. :-)
			throw new MissingResourceException("Can't find bundle for base name " + aBundleName + ", locale " + aLocale, aBundleName + "_" + aLocale, "");
		}
		return With(aResourceBundle);
	}

	/**
	 * Look up the string for the user message.
	 * 
	 * @param aUserMessage jp.co.sra.smalltalk.StUserMessage
	 * @return java.lang.String
	 * @category Accessing
	 */
	public static String LookUp(StUserMessage aUserMessage) {
		for (int i = 0; i < Catalogs.size(); i++) {
			String aString = ((StMessageCatalog) Catalogs.get(i)).lookUp(aUserMessage);
			if (aString != null) {
				return aString;
			}
		}
		return null;
	}

	/**
	 * Give a priority to the specified message catalog.
	 * 
	 * @param aMessageCatalog jp.co.sra.smalltalk.StMessageCatalog
	 * @category Utilities
	 */
	public static void GivePriorityTo(StMessageCatalog aMessageCatalog) {
		Catalogs.remove(aMessageCatalog);
		Catalogs.add(0, aMessageCatalog);
	}

	/**
	 * Create a new instance of StMessageCatalog and initialize it.
	 * 
	 * @param aBundleName java.lang.String
	 * @deprecated since StPL7.0, use StMessageCatalog#With(String)
	 * @category Instance creation
	 */
	public StMessageCatalog(String aBundleName) {
		throw new SmalltalkException("Deprecated since StPL7.0, use StMessageCatalog#With(String)");
	}

	/**
	 * Create a new instance of StMessageCatalog and initialize it.
	 * 
	 * @param aBundleName java.lang.String
	 * @param aLocale java.util.Locale
	 * @deprecated since StPL7.0, use StMessageCatalog#With(String, Locale)
	 * @category Instance creation
	 */
	public StMessageCatalog(String aBundleName, Locale aLocale) {
		throw new SmalltalkException("Deprecated since StPL7.0, use StMessageCatalog#With(String, Locale)");
	}

	/**
	 * Create a new instance of StMessageCatalog and initialize it with ResourceBundle.
	 *
	 * @param aResourceBundle java.util.ResourceBundle
	 * @category Instance creation
	 */
	private StMessageCatalog(ResourceBundle aResourceBundle) {
		resourceBundle = aResourceBundle;
		Catalogs.add(0, this);
	}

	/**
	 * Answer my current locale.
	 * 
	 * @return java.util.Locale
	 * @category accessing
	 */
	public Locale getLocale() {
		return (resourceBundle == null) ? null : resourceBundle.getLocale();
	}

	/**
	 * Set my new locale.
	 * 
	 * @param aLocale java.util.Locale
	 * @deprecated since StPL7.0, create another message catalog for different locales.
	 * @category accessing
	 */
	public void setLocale(Locale aLocale) {
		throw new SmalltalkException("Can't change the locale dynamically. Create another one for a different locale.");
	}

	/**
	 * Look up the string for the user message.
	 * 
	 * @param aUserMessage jp.co.sra.smalltalk.StUserMessage
	 * @return java.lang.String
	 * @category accessing
	 */
	public String lookUp(StUserMessage aUserMessage) {
		if (aUserMessage.catalog() != null && aUserMessage.catalog() != this) {
			return null;
		}
		if (resourceBundle == null) {
			return null;
		}

		String aString = null;
		try {
			aString = resourceBundle.getString(aUserMessage.key());
		} catch (MissingResourceException e) {
		}
		return aString;
	}

	/**
	 * Look up the string for the key from the message catalog.
	 * 
	 * @param key java.lang.String
	 * @return java.lang.String
	 * @deprecated since StPL7.0, use StUserMessage
	 * @category accessing
	 */
	public String lookUp(String key) {
		return this.lookUp(key, null);
	}

	/**
	 * Look up the string for the key from the message catalog.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @deprecated since StPL7.0, use StUserMessage
	 * @category accessing
	 */
	public String lookUp(String key, String defaultString) {
		if (defaultString == null) {
			defaultString = key;
		}

		if (resourceBundle == null) {
			return this.lookUpFailed(key, defaultString);
		} else {
			try {
				return resourceBundle.getString(key);
			} catch (MissingResourceException e) {
				return this.lookUpFailed(key, defaultString);
			}
		}
	}

	/**
	 * Look up the string for the key defined in the resource bundle.
	 * Parameter substitutions on the string will be done.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param parameter java.lang.Object
	 * @return java.lang.String
	 * @deprecated since StPL7.0, use StUserMessage
	 * @category accessing
	 */
	public String lookUp(String key, String defaultString, Object parameter) {
		return this.lookUp(key, defaultString, new Object[] { parameter });
	}

	/**
	 * Look up the string for the key defined in the resource bundle.
	 * Parameter substitutions on the string will be done.
	 * I wish I could use String#replaceAll()...
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param parameters java.lang.Object[]
	 * @return java.lang.String
	 * @deprecated since StPL7.0, use StUserMessage
	 * @category accessing
	 */
	public String lookUp(String key, String defaultString, Object[] parameters) {
		String string = this.lookUp(key, defaultString);
		if (parameters != null) {
			for (int i = 0; i < parameters.length; i++) {
				StringBuffer stringBuffer = new StringBuffer(string.length());
				String pkey = "<" + (i + 1) + "p>";
				int from = 0, index;
				while ((index = string.indexOf(pkey, from)) >= 0) {
					stringBuffer.append(string.substring(from, index));
					stringBuffer.append(parameters[i].toString());
					from = index + pkey.length();
				}
				stringBuffer.append(string.substring(from));
				string = stringBuffer.toString();
			}
		}
		return string;
	}

	/**
	 * Called when the message for the key is missing.
	 * The subclasses may override this to handle exception in other way. 
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @deprecated since StPL7.0, use StUserMessage
	 * @category private
	 */
	protected String lookUpFailed(String key, String defaultString) {
		return defaultString;
	}

}