package jp.co.sra.smalltalk;

import java.awt.AWTEvent;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.AWTEventListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.io.Writer;

import javax.swing.SwingUtilities;

/**
 * StInputState class
 * 
 *  @author    nisinaka
 *  @created   2005/02/16 (by nisinaka)
 *  @updated   2006/10/16 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StInputState.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StInputState extends StObject implements AWTEventListener {

	protected static StInputState DefaultState = new StInputState();

	protected int modifiers;
	protected Point lastDownPoint;
	protected Point lastUpPoint;

	/**
	 * Answer the default state.
	 * 
	 * @return jp.co.sra.smalltalk.StInputState
	 * @category Accessing
	 */
	public static StInputState Default() {
		return DefaultState;
	}

	/**
	 * Create a new instance of StInputState and initialize it.
	 * 
	 * @category Instance creation
	 */
	private StInputState() {
		this.initialize();
		Toolkit.getDefaultToolkit().addAWTEventListener(this, AWTEvent.KEY_EVENT_MASK | AWTEvent.MOUSE_EVENT_MASK);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		modifiers = 0;
		lastDownPoint = null;
		lastUpPoint = null;
	}

	/**
	 * Answer whether the shift key is down.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	public boolean shiftDown() {
		return (modifiers & InputEvent.SHIFT_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the control key is down.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	public boolean ctrlDown() {
		return (modifiers & InputEvent.CTRL_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the meta key is down.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	public boolean metaDown() {
		return (modifiers & InputEvent.META_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the alt key is down.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	public boolean altDown() {
		return (modifiers & InputEvent.ALT_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the button1 is down or not.
	 * 
	 * @return boolean
	 * @category mouse
	 */
	public boolean button1Down() {
		return (modifiers & InputEvent.BUTTON1_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the button2 is down or not.
	 * 
	 * @return boolean
	 * @category mouse
	 */
	public boolean button2Down() {
		return (modifiers & InputEvent.BUTTON2_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether the button3 is down or not.
	 * 
	 * @return boolean
	 * @category mouse
	 */
	public boolean button3Down() {
		return (modifiers & InputEvent.BUTTON3_DOWN_MASK) != 0;
	}

	/**
	 * Answer whether any button is down or not.
	 * 
	 * @return boolean
	 * @category mouse
	 */
	public boolean anyButtonDown() {
		return (modifiers & (InputEvent.BUTTON1_DOWN_MASK | InputEvent.BUTTON2_DOWN_MASK | InputEvent.BUTTON3_DOWN_MASK)) != 0;
	}

	/**
	 * Dispatch an AWTEvent as AWTEventListener. 
	 * 
	 * @param e java.awt.AWTEvent
	 * @see java.awt.event.AWTEventListener#eventDispatched(java.awt.AWTEvent)
	 * @category event handling
	 */
	public void eventDispatched(AWTEvent e) {
		if (e instanceof KeyEvent) {
			this.dispatchKeyEvent((KeyEvent) e);
		} else if (e instanceof MouseEvent) {
			this.dispatchMouseEvent((MouseEvent) e);
		}
	}

	/**
	 * Dispach the key event.
	 * 
	 * @param e java.awt.event.KeyEvent
	 * @category event handling
	 */
	protected void dispatchKeyEvent(KeyEvent e) {
		this.dispatchInputEvent(e);
	}

	/**
	 * Dispach the mouse event.
	 * 
	 * @return e java.awt.event.MouseEvent
	 * @category event handling
	 */
	protected void dispatchMouseEvent(MouseEvent e) {
		this.dispatchInputEvent(e);

		Point aPoint = null;
		switch (e.getID()) {
			case MouseEvent.MOUSE_PRESSED:
				aPoint = e.getPoint();
				SwingUtilities.convertPointToScreen(aPoint, e.getComponent());
				lastDownPoint = aPoint;
				break;
			case MouseEvent.MOUSE_RELEASED:
				aPoint = e.getPoint();
				SwingUtilities.convertPointToScreen(aPoint, e.getComponent());
				lastUpPoint = aPoint;
				break;
		}
	}

	/**
	 * Dispach the input event.
	 * 
	 * @return e java.awt.event.InputEvent
	 * @category event handling
	 */
	protected void dispatchInputEvent(InputEvent e) {
		modifiers = e.getModifiersEx();
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);
		aWriter.write('(');
		aWriter.write(InputEvent.getModifiersExText(modifiers));
		aWriter.write(')');
	}

}