package jp.co.sra.jun.vrml.support;

import java.util.*;
import jp.co.sra.smalltalk.StSymbol;

/**
 * JunVrmlParser20 class
 * 
 *  @author    ASTI Beijing
 *  @created   UNKNOWN
 *  @updated   2000/03/22 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVrmlParser20.java,v 8.10 2008/02/20 06:33:17 nisinaka Exp $
 */
public class JunVrmlParser20 extends JunVrmlParser {

	/**
	 * Parse a declaration and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseDeclaration() {
		this.nextToken();

		if ((tokenType == $("identifier")) || (tokenType == $("Script")) || (tokenType == $("DEF")) || (tokenType == $("USE"))) {
			this.unNextToken();

			return this.parseNodeDeclaration();
		}

		if ((tokenType == $("PROTO")) || (tokenType == $("EXTERNPROTO"))) {
			this.unNextToken();

			return this.parseProtoDeclaration();
		}

		if (tokenType == $("ROUTE")) {
			this.unNextToken();

			return this.parseRouteDeclaration();
		}

		if (tokenType == $("NULL")) {
			return true;
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse declarations and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseDeclarations() {
		Vector tmpNode = new Vector();
		this.nextToken();

		while ((tokenType == $("identifier")) || (tokenType == $("Script")) || (tokenType == $("DEF")) || (tokenType == $("USE")) || (tokenType == $("PROTO")) || (tokenType == $("EXTERNPROTO")) || (tokenType == $("ROUTE"))) {
			this.unNextToken();

			if (this.parseDeclaration() == false) {
				failBlock.value_("Syntax Error");

				return false;
			}

			tmpNode.addElement(parseNode);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = tmpNode;

		return true;
	}

	/**
	 * Parse an external interface declaration and answer true if successfully
	 * parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseExternInterfaceDeclaration() {
		this.nextToken();

		if (tokenType == $("eventIn")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		if (tokenType == $("eventOut")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		if (tokenType == $("field")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		if (tokenType == $("exposedField")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Parse external interface declarations and answer true if successfully
	 * parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseExternInterfaceDeclarations() {
		boolean result = false;
		this.nextToken();

		while ((tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field")) || (tokenType == $("exposedField"))) {
			this.unNextToken();
			result = this.parseExternInterfaceDeclaration();
			this.nextToken();
		}

		this.unNextToken();

		return result;
	}

	/**
	 * Parse an external proto and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseExternProto() {
		this.nextToken();

		if (tokenType == $("EXTERNPROTO")) {
			this.nextToken();

			if (tokenType == $("identifier")) {
				this.nextToken();

				if (tokenType == $("leftBracket")) {
					if (this.parseExternInterfaceDeclarations()) {
						this.nextToken();

						if (tokenType == $("rightBracket")) {
							return this.parseMFStringValue();
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * Parse a field type and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseFieldType() {
		this.nextToken();

		if (tokenType == $("MFColor")) {
			return true;
		}

		if (tokenType == $("MFFloat")) {
			return true;
		}

		if (tokenType == $("MFInt32")) {
			return true;
		}

		if (tokenType == $("MFNode")) {
			return true;
		}

		if (tokenType == $("MFRotation")) {
			return true;
		}

		if (tokenType == $("MFString")) {
			return true;
		}

		if (tokenType == $("MFVec2f")) {
			return true;
		}

		if (tokenType == $("MFVec3f")) {
			return true;
		}

		if (tokenType == $("SFBool")) {
			return true;
		}

		if (tokenType == $("SFColor")) {
			return true;
		}

		if (tokenType == $("SFFloat")) {
			return true;
		}

		if (tokenType == $("SFImage")) {
			return true;
		}

		if (tokenType == $("SFInt32")) {
			return true;
		}

		if (tokenType == $("SFNode")) {
			return true;
		}

		if (tokenType == $("SFRotation")) {
			return true;
		}

		if (tokenType == $("SFString")) {
			return true;
		}

		if (tokenType == $("SFTime")) {
			return true;
		}

		if (tokenType == $("SFVec2f")) {
			return true;
		}

		if (tokenType == $("SFVec3f")) {
			return true;
		}

		return false;
	}

	/**
	 * Parse a field value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseFieldValue() {
		Vector values = new Vector();
		this.nextToken();

		if ((tokenType == $("TRUE")) || (tokenType == $("FALSE"))) {
			this.unNextToken();

			return this.parseSFBoolValue();
		}

		if (tokenType == $("number")) {
			this.nextToken();

			if (tokenType == $("number")) {
				this.unNextToken();
				this.unNextToken();

				return this.parseSFFloatValues();
			} else {
				this.unNextToken();
				parseNode = token;
				values.addElement(token);

				return true;
			}
		}

		if (tokenType == $("string")) {
			parseNode = token;
			values.addElement(token);

			return true;
		}

		if ((tokenType == $("identifier")) || (tokenType == $("Script")) || (tokenType == $("DEF")) || (tokenType == $("USE"))) {
			this.unNextToken();

			return this.parseNodeDeclaration();
		}

		if (tokenType == $("NULL")) {
			return true;
		}

		if (tokenType == $("leftBracket")) {
			this.nextToken();

			if (tokenType == $("number")) {
				this.unNextToken();
				this.unNextToken();

				return this.parseMFFloatValue();
			}

			if (tokenType == $("number")) {
				this.unNextToken();
				this.unNextToken();

				return this.parseMFInt32Value();
			}

			if (tokenType == $("string")) {
				this.unNextToken();
				this.unNextToken();

				return this.parseMFStringValue();
			}

			if ((tokenType == $("identifier")) || (tokenType == $("Script")) || (tokenType == $("DEF")) || (tokenType == $("USE"))) {
				this.unNextToken();
				this.unNextToken();

				return this.parseMFNodeValue();
			}

			if (tokenType == $("rightBracket")) {
				return true;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse an interface declaration and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseInterfaceDeclaration() {
		this.nextToken();

		if (tokenType == $("exposedField")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return this.parseFieldValue();
				} else {
					return false;
				}
			} else {
				return false;
			}
		}

		if ((tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field"))) {
			this.unNextToken();

			return this.parseRestrictedInterfaceDeclaration();
		}

		return false;
	}

	/**
	 * Parse interface declarations and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseInterfaceDeclarations() {
		boolean result = false;
		this.nextToken();

		while ((tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field")) || (tokenType == $("exposedField"))) {
			this.unNextToken();
			result = this.parseInterfaceDeclaration();
			this.nextToken();
		}

		this.unNextToken();

		return result;
	}

	/**
	 * Parse a MFFloat value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseMFFloatValue() {
		this.nextToken();

		if (tokenType == $("leftBracket")) {
			this.parseSFFloatValues();
			this.nextToken();

			if (tokenType == $("rightBracket")) {
				return true;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a MFInt32 value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseMFInt32Value() {
		this.nextToken();

		if (tokenType == $("leftBracket")) {
			this.parseSFInt32Values();
			this.nextToken();

			if (tokenType == $("rightBracket")) {
				return true;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a MFNode value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseMFNodeValue() {
		this.nextToken();

		if (tokenType == $("leftBracket")) {
			this.parseSFNodeValues();
			this.nextToken();

			if (tokenType == $("rightBracket")) {
				return true;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a MFString value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseMFStringValue() {
		this.nextToken();

		if (tokenType == $("leftBracket")) {
			this.parseSFStringValues();
			this.nextToken();

			if (tokenType == $("rightBracket")) {
				return true;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a node and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseNode() {
		this.nextToken();

		if (tokenType == $("identifier")) {
			StSymbol nodeName = (StSymbol) token;
			this.nextToken();

			if (tokenType == $("leftBrace")) {
				this.parseNodeGuts();
				parseNode = builder.newNode_with_(nodeName, parseNode);
				this.nextToken();

				if (tokenType == $("rightBrace")) {
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		}

		if (tokenType == $("Script")) {
			// nodeName = token;
			this.nextToken();

			if (tokenType == $("leftBrace")) {
				this.parseScriptGuts();
				this.nextToken();

				if (tokenType == $("rightBrace")) {
					return true;
				}
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a node declaration and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseNodeDeclaration() {
		this.nextToken();

		if ((tokenType == $("identifier")) || (tokenType == $("Script"))) {
			this.unNextToken();

			return this.parseNode();
		}

		if (tokenType == $("DEF")) {
			this.nextToken();

			if (tokenType == $("identifier")) {
				return this.parseNode();
			} else {
				return false;
			}
		}

		if (tokenType == $("USE")) {
			this.nextToken();

			if (tokenType == $("identifier")) {
				return true;
			} else {
				return false;
			}
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse a node gut and answer true if successfully parsed.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * 
	 * @return boolean
	 */
	public boolean parseNodeGut_(Hashtable aDictionary) {
		this.nextToken();

		if (tokenType == $("identifier")) {
			StSymbol fieldName = (StSymbol) token;
			this.nextToken();

			if ((tokenType == $("number")) || (tokenType == $("leftBracket")) || (tokenType == $("identifier")) || (tokenType == $("string")) || (tokenType == $("TRUE")) || (tokenType == $("FALSE"))) {
				this.unNextToken();

				if (this.parseFieldValue() == false) {
					failBlock.value_("Syntax error");

					return false;
				}

				aDictionary.put(fieldName, parseNode);

				return true;
			}

			if (tokenType == $("IS")) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		}

		if (tokenType == $("ROUTE")) {
			this.unNextToken();

			return this.parseRouteDeclaration();
		}

		if (tokenType == $("PROTO")) {
			this.unNextToken();

			return this.parseProtoDeclaration();
		}

		failBlock.value_("Syntax error");

		return false;
	}

	/**
	 * Parse node guts and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseNodeGuts() {
		Hashtable parseFields = new Hashtable();
		Vector parseNodes = new Vector();
		this.nextToken();

		while ((tokenType == $("identifier")) || (tokenType == $("ROUTE")) || (tokenType == $("PROTO"))) {
			this.unNextToken();

			if (this.parseNodeGut_(parseFields) == false) {
				failBlock.value_("Syntax error");

				return false;
			}

			if (parseFields.isEmpty()) {
				parseNodes.addElement(parseNode);
			}

			this.nextToken();
		}

		this.unNextToken();

		if (parseFields.isEmpty()) {
			parseNode = parseNodes;
		} else {
			parseNode = parseFields;
		}

		return true;
	}

	/**
	 * Parse a proto and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseProto() {
		this.nextToken();

		if (tokenType == $("PROTO")) {
			this.nextToken();

			if (tokenType == $("identifier")) {
				this.nextToken();

				if (tokenType == $("leftBracket")) {
					if (this.parseInterfaceDeclarations()) {
						this.nextToken();

						if (tokenType == $("rightBracket")) {
							this.nextToken();

							if (tokenType == $("leftBrace")) {
								if (this.parseVrmlScene()) {
									this.nextToken();

									if (tokenType == $("rightBrace")) {
										return true;
									}
								}
							}
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * Parse a proto declaration and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseProtoDeclaration() {
		this.nextToken();

		if (tokenType == $("PROTO")) {
			this.unNextToken();

			return this.parseProto();
		}

		if (tokenType == $("EXTERNPROTO")) {
			this.unNextToken();

			return this.parseExternProto();
		}

		return false;
	}

	/**
	 * Parse a restricted interface declaration and answer true if successfully
	 * parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseRestrictedInterfaceDeclaration() {
		this.nextToken();

		if (tokenType == $("eventIn")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		if (tokenType == $("eventOut")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return true;
				}
			}
		}

		if (tokenType == $("field")) {
			if (this.parseFieldType()) {
				this.nextToken();

				if (tokenType == $("identifier")) {
					return this.parseFieldValue();
				}
			}
		}

		return false;
	}

	/**
	 * Parse a route declaration and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseRouteDeclaration() {
		this.nextToken();

		if (tokenType == $("ROUTE")) {
			this.nextToken();

			if (tokenType == $("identifier")) {
				this.nextToken();

				if (tokenType == $("period")) {
					if (((Boolean) this.nextToken()).booleanValue()) { // very very strange here
						this.nextToken();

						if (tokenType == $("TO")) {
							this.nextToken();

							if (tokenType == $("identifier")) {
								this.nextToken();

								if (tokenType == $("period")) {
									this.nextToken();

									if (tokenType == $("identifier")) {
										return true;
									}
								}
							}
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * Parse a script gut and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseScriptGut() {
		this.nextToken();

		if ((tokenType == $("identifier")) || (tokenType == $("ROUTE")) || (tokenType == $("PROTO"))) {
			this.unNextToken();

			Hashtable parseFields = new Hashtable();

			return this.parseNodeGut_(parseFields);
		}

		if ((tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field"))) {
			this.unNextToken();

			if (this.parseRestrictedInterfaceDeclaration()) {
				this.nextToken();

				if (tokenType == $("IS")) {
					this.nextToken();

					if (tokenType == $("identifier")) {
						return true;
					}
				} else {
					this.unNextToken();

					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Parse script guts and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseScriptGuts() {
		Hashtable parseFields = new Hashtable();
		Vector parseNodes = new Vector();
		this.nextToken();

		while ((tokenType == $("identifier")) || (tokenType == $("ROUTE")) || (tokenType == $("PROTO")) || (tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field"))) {
			this.unNextToken();

			if (this.parseNodeGut_(parseFields) == false) {
				failBlock.value_("Syntax error");

				return false;
			}

			if (parseFields.isEmpty()) {
				parseNodes.addElement(parseNode);
			}

			this.nextToken();
		}

		this.unNextToken();

		if (parseFields.isEmpty()) {
			parseNode = parseNodes;
		} else {
			parseNode = parseFields;
		}

		return true;
	}

	/**
	 * Parse a SFBool value and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseSFBoolValue() {
		StSymbol boolValue = null;
		this.nextToken();

		if ((tokenType == $("TRUE")) || (tokenType == $("FALSE"))) {
			boolValue = (StSymbol) token;
		}

		parseNode = boolValue;

		return true;
	}

	/**
	 * Parse SFFloat values and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseSFFloatValues() {
		Vector floatValues = new Vector();
		this.nextToken();

		while (tokenType == $("number")) {
			floatValues.addElement(token);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = floatValues;

		return true;
	}

	/**
	 * Parse SFInt32 values and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseSFInt32Values() {
		Vector int32Values = new Vector();
		this.nextToken();

		while (tokenType == $("number")) {
			int32Values.addElement(token);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = int32Values;

		return true;
	}

	/**
	 * Parse SFNode values and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseSFNodeValues() {
		Vector tmpNode = new Vector();
		this.nextToken();

		while ((tokenType == $("identifier")) || (tokenType == $("Script")) || (tokenType == $("DEF")) || (tokenType == $("USE"))) {
			this.unNextToken();

			if (this.parseNodeDeclaration() == false) {
				failBlock.value_("Syntax error");

				return false;
			}

			if (parseNode instanceof Hashtable) {
				for (Enumeration e = ((Hashtable) parseNode).elements(); e.hasMoreElements();) {
					tmpNode.addElement(e.nextElement());
				}
			} else {
				tmpNode.addElement(parseNode);
			}

			this.nextToken();
		}

		this.unNextToken();
		parseNode = tmpNode;

		return true;
	}

	/**
	 * Parse SFString values and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseSFStringValues() {
		Vector stringValues = new Vector();
		this.nextToken();

		while (tokenType == $("string")) {
			stringValues.addElement(token);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = stringValues;

		return true;
	}

	/**
	 * Parse a VRML scene and answer true if successfully parsed.
	 * 
	 * @return boolean
	 */
	public boolean parseVrmlScene() {
		return parseDeclarations();
	}

	/**
	 * Answer an instance of the default syntax tree builder.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlSyntaxTreeBuilder
	 */
	protected JunVrmlSyntaxTreeBuilder defaultBuilder() {
		return new JunVrmlSyntaxTreeBuilder20();
	}

	/**
	 * Answer the default keyword table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlKeywordTable
	 */
	protected JunVrmlKeywordTable defaultKeywordTable() {
		return JunVrmlKeywordTable20.KeywordTable();
	}

	/**
	 * Answer the default scanner table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlScannerTable
	 */
	protected JunVrmlScannerTable defaultScannerTable() {
		return JunVrmlScannerTable20.ScannerTable();
	}
}
