package jp.co.sra.jun.voronoi.twoD.diagram;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;

import javax.swing.JFrame;
import javax.swing.JPanel;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StApplicationModel;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;

/**
 * JunVoronoi2dProcessor class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/01/25 (by NISHIHARA Satoshi)
 *  @updated   2006/11/06 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun668 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVoronoi2dProcessor.java,v 8.11 2008/02/20 06:33:14 nisinaka Exp $
 */
public class JunVoronoi2dProcessor extends JunVoronoi2dObject {

	protected JunVoronoi2dDot[] originalDots;
	protected JunVoronoi2dDot[] normalizedDots;
	protected JunVoronoi2dDot translationDelta;
	protected JunVoronoi2dDot scalingFactor;
	protected ArrayList voronoiDots;
	protected ArrayList voronoiPoints;
	protected ArrayList voronoiSides;
	protected double processingPrecision;
	protected boolean traceBoolean;
	private JPanel _traceCanvas;

	/**
	 * Create a new instance of JunVoronoi2dProcessor and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunVoronoi2dProcessor(Jun2dPoint[] points) {
		this.dots_(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		originalDots = null;
		normalizedDots = null;
		translationDelta = null;
		scalingFactor = null;
		voronoiDots = null;
		voronoiPoints = null;
		voronoiSides = null;
		processingPrecision = this.accuracy();
		traceBoolean = false;
	}

	/**
	 * Initialize the voronoi diagram.
	 * 
	 * @category initialize-release
	 */
	protected synchronized void initializeVoronoiDiagram() {
		JunVoronoi2dDot dot1 = new JunVoronoi2dDot(this.bigSize() * -1, this.bigSize() * -1);
		JunVoronoi2dDot dot2 = new JunVoronoi2dDot(10.0d, this.bigSize());
		JunVoronoi2dDot dot3 = new JunVoronoi2dDot(this.bigSize(), this.bigSize() * -1);
		JunVoronoi2dPoint point1 = new JunVoronoi2dPoint((dot1.x() + dot2.x()) * 0.5d, (dot1.y() + dot2.y()) * 0.5d);
		JunVoronoi2dPoint point2 = new JunVoronoi2dPoint((dot2.x() + dot3.x()) * 0.5d, (dot2.y() + dot3.y()) * 0.5d);
		JunVoronoi2dPoint point3 = new JunVoronoi2dPoint((dot3.x() + dot1.x()) * 0.5d, (dot3.y() + dot1.y()) * 0.5d);
		JunVoronoi2dLine line1 = dot1.bisector_(dot2);
		JunVoronoi2dLine line2 = dot2.bisector_(dot3);
		JunVoronoi2dPoint point4 = line1.intersectingPointWithLine_(line2);
		point1 = (new JunVoronoi2dLine(point1, point4)).pointAtX_(this.bigNumber() * -1);
		point2 = (new JunVoronoi2dLine(point2, point4)).pointAtX_(this.bigNumber());
		point3 = (new JunVoronoi2dLine(point3, point4)).pointAtY_(this.bigNumber() * -1);
		JunVoronoi2dSide side1 = new JunVoronoi2dSide(point1, point4);
		side1.dot1_(dot1);
		side1.dot2_(dot2);
		JunVoronoi2dSide side2 = new JunVoronoi2dSide(point2, point4);
		side2.dot1_(dot2);
		side2.dot2_(dot3);
		JunVoronoi2dSide side3 = new JunVoronoi2dSide(point3, point4);
		side3.dot1_(dot3);
		side3.dot2_(dot1);
		point1.side1_(side1);
		point2.side1_(side2);
		point3.side1_(side3);
		point4.side1_(side1);
		point4.side2_(side2);
		point4.side3_(side3);

		voronoiDots = new ArrayList();
		voronoiDots.add(dot1);
		voronoiDots.add(dot2);
		voronoiDots.add(dot3);

		voronoiPoints = new ArrayList();
		voronoiPoints.add(point1);
		voronoiPoints.add(point2);
		voronoiPoints.add(point3);
		voronoiPoints.add(point4);

		voronoiSides = new ArrayList();
		voronoiSides.add(side1);
		voronoiSides.add(side2);
		voronoiSides.add(side3);
	}

	/**
	 * Finalize the voronoi diagram.
	 * 
	 * @category initialize-release
	 */
	protected synchronized void finalizeVoronoiDiagram() {
		ArrayList obstacles = new ArrayList();
		for (int index = 0; index < voronoiPoints.size(); index++) {
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) voronoiPoints.get(index);
			if (point.isInvalid() && this.startIndex() <= index) {
				obstacles.add(point);
			}
		}
		voronoiPoints.removeAll(obstacles);

		obstacles = new ArrayList();
		for (int index = 0; index < voronoiSides.size(); index++) {
			JunVoronoi2dSide side = (JunVoronoi2dSide) voronoiSides.get(index);
			if (side.isInvalid() && this.startIndex() <= index) {
				obstacles.add(side);
			}
		}
		voronoiSides.removeAll(obstacles);
	}

	/**
	 * Answer my current dots.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing 
	 */
	public Jun2dPoint[] dots() {
		if (normalizedDots == null) {
			return new Jun2dPoint[0];
		}

		Jun2dPoint[] points = new Jun2dPoint[normalizedDots.length];
		for (int i = 0; i < points.length; i++) {
			points[i] = this.restoreDot_(normalizedDots[i]);
		}
		return points;
	}

	/**
	 * Set my new dots.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing 
	 */
	public void dots_(Jun2dPoint[] points) {
		ArrayList dots = new ArrayList();
		for (int i = 0; i < points.length; i++) {
			JunVoronoi2dDot it = null;
			for (int j = 0; j < dots.size(); j++) {
				JunVoronoi2dDot each = (JunVoronoi2dDot) dots.get(j);
				if (Math.abs(each.x() - points[i].x()) < this.accuracy() && Math.abs(each.y() - points[i].y()) < this.accuracy()) {
					it = each;
					break;
				}
			}
			if (it == null) {
				dots.add(new JunVoronoi2dDot(points[i].x(), points[i].y()));
			}
		}
		if (dots.isEmpty()) {
			return;
		}
		originalDots = (JunVoronoi2dDot[]) dots.toArray(new JunVoronoi2dDot[dots.size()]);
		normalizedDots = this.normalizedDots_(originalDots);
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		if (voronoiPoints == null) {
			return new Jun2dPoint[0];
		}

		ArrayList aList = new ArrayList();
		for (int index = this.startIndex(); index < voronoiPoints.size(); index++) {
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) voronoiPoints.get(index);
			aList.add(this.restorePoint_(point));
		}
		return (Jun2dPoint[]) aList.toArray(new Jun2dPoint[aList.size()]);
	}

	/**
	 * Answer my sides.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category accessing
	 */
	public Jun2dPoint[][] sides() {
		if (voronoiSides == null) {
			return new Jun2dPoint[0][2];
		}

		ArrayList aList = new ArrayList();
		for (int index = this.startIndex(); index < voronoiSides.size(); index++) {
			JunVoronoi2dSide side = (JunVoronoi2dSide) voronoiSides.get(index);
			aList.add(this.restoreSide_(side));
		}
		return (Jun2dPoint[][]) aList.toArray(new Jun2dPoint[aList.size()][2]);
	}

	/**
	 * Answer my triangles
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category accessing
	 */
	public Jun2dPoint[][] triangles() {
		if (voronoiPoints == null) {
			return new Jun2dPoint[0][3];
		}

		ArrayList aList = new ArrayList();
		for (int index = this.startIndex(); index < voronoiPoints.size(); index++) {
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) voronoiPoints.get(index);
			int[] indexes = this.triangleIndexesAtPoint_(point);
			if (indexes != null) {
				JunVoronoi2dDot dot1 = originalDots[indexes[0]];
				JunVoronoi2dDot dot2 = originalDots[indexes[1]];
				JunVoronoi2dDot dot3 = originalDots[indexes[2]];
				Jun2dPoint p1 = new Jun2dPoint(dot1.x(), dot1.y());
				Jun2dPoint p2 = new Jun2dPoint(dot2.x(), dot2.y());
				Jun2dPoint p3 = new Jun2dPoint(dot3.x(), dot3.y());
				aList.add(new Jun2dPoint[] { p1, p2, p3 });
			}
		}
		return (Jun2dPoint[][]) aList.toArray(new Jun2dPoint[aList.size()][3]);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @category accessing 
	 */
	public double area() {
		double area = 0;
		Jun2dPoint[][] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			JunVoronoi2dTriangle aTriangle = new JunVoronoi2dTriangle(triangles[i][0], triangles[i][1], triangles[i][2]);
			area += aTriangle.area();
		}
		return area;
	}

	/**
	 * Answer my current precision type.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol precision() {
		if (processingPrecision > this.accuracy() * 10.0) {
			return $("lowPrecision");
		}

		return $("highPrecision");
	}

	/**
	 * Answer my current processing precision.
	 * 
	 * @return double
	 * @category basic accessing
	 */
	protected double processingPrecision() {
		return processingPrecision;
	}

	/**
	 * Answer the start index.
	 * 
	 * @return int
	 * @category basic accessing
	 */
	protected int startIndex() {
		return 3;
	}

	/**
	 * Restore the voronoi 2D dot.
	 * 
	 * @param voronoi2dDot jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category basic accessing
	 */
	protected Jun2dPoint restoreDot_(JunVoronoi2dDot voronoi2dDot) {
		double x = voronoi2dDot.x() * scalingFactor.x() + translationDelta.x();
		double y = voronoi2dDot.y() * scalingFactor.y() + translationDelta.y();
		return new Jun2dPoint(x, y);
	}

	/**
	 * Restore the voronoi 2D point.
	 * 
	 * @param voronoi2dPoint jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category basic accessing
	 */
	protected Jun2dPoint restorePoint_(JunVoronoi2dPoint voronoi2dPoint) {
		double x = voronoi2dPoint.x() * scalingFactor.x() + translationDelta.x();
		double y = voronoi2dPoint.y() * scalingFactor.y() + translationDelta.y();
		return new Jun2dPoint(x, y);
	}

	/**
	 * Restore the voronoi 2D side.
	 * 
	 * @param voronoi2dSide jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dSide
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category basic accessing
	 */
	protected Jun2dPoint[] restoreSide_(JunVoronoi2dSide voronoi2dSide) {
		Jun2dPoint from = this.restorePoint_(voronoi2dSide.from());
		Jun2dPoint to = this.restorePoint_(voronoi2dSide.to());
		return new Jun2dPoint[] { from, to };
	}

	/**
	 * Answer the triangle indexes at the specified point.
	 * 
	 * @param voronoi2dPoint jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @return int[]
	 * @category basic accessing
	 */
	protected int[] triangleIndexesAtPoint_(JunVoronoi2dPoint voronoi2dPoint) {
		JunVoronoi2dDot[] dots = this.dotsAtPoint_(voronoi2dPoint);
		int index1 = voronoiDots.indexOf(dots[0]);
		int index3 = voronoiDots.indexOf(dots[1]);
		int index5 = voronoiDots.indexOf(dots[2]);
		if (index1 < this.startIndex()) {
			return null;
		}
		if (index3 < this.startIndex()) {
			return null;
		}
		if (index5 < this.startIndex()) {
			return null;
		}

		return new int[] { index1 - this.startIndex(), index3 - this.startIndex(), index5 - this.startIndex() };
	}

	/**
	 * Answer the dots at the specified point.
	 * 
	 * @param voronoi2dPoint jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot[]
	 * @category basic accessing
	 */
	protected JunVoronoi2dDot[] dotsAtPoint_(JunVoronoi2dPoint voronoi2dPoint) {
		JunVoronoi2dSide side1 = voronoi2dPoint.side1();
		JunVoronoi2dSide side2 = voronoi2dPoint.side2();
		JunVoronoi2dSide side3 = voronoi2dPoint.side3();

		int[] indexes = new int[6];
		indexes[0] = voronoiDots.indexOf(side1.dot1());
		indexes[1] = voronoiDots.indexOf(side1.dot2());
		indexes[2] = voronoiDots.indexOf(side2.dot1());
		indexes[3] = voronoiDots.indexOf(side2.dot2());
		indexes[4] = voronoiDots.indexOf(side3.dot1());
		indexes[5] = voronoiDots.indexOf(side3.dot2());
		Arrays.sort(indexes);

		if (indexes[0] != indexes[1]) {
			throw SmalltalkException.Error("unexpected error.");
		}
		if (indexes[2] != indexes[3]) {
			throw SmalltalkException.Error("unexpected error.");
		}
		if (indexes[4] != indexes[5]) {
			throw SmalltalkException.Error("unexpected error.");
		}

		return new JunVoronoi2dDot[] { (JunVoronoi2dDot) voronoiDots.get(indexes[0]), (JunVoronoi2dDot) voronoiDots.get(indexes[2]), (JunVoronoi2dDot) voronoiDots.get(indexes[4]) };
	}

	/**
	 * Compute the voronoi diagram.
	 * 
	 * @return boolean
	 * @category computing
	 */
	public boolean compute() {
		boolean condition = this.computeFirst();
		if (condition == false) {
			condition = this.computeSecond();
		}
		return condition;
	}

	/**
	 * Compute the voronoi diagram, the first try.
	 * 
	 * @return boolean
	 * @category computing
	 */
	protected boolean computeFirst() {
		if (normalizedDots == null) {
			return true;
		}

		this.initializeVoronoiDiagram();
		// this.debugInitialize();

		if (this.trace() && _traceCanvas == null) {
			_traceCanvas = new JPanel() {
				public void paint(Graphics aGraphics) {
					traceDisplayOn_(aGraphics);
				}
			};
			_traceCanvas.setPreferredSize(new Dimension(400, 400));

			JFrame aFrame = new JFrame();
			aFrame.setTitle("JunVoronoi2dProcessor");
			aFrame.getContentPane().setLayout(new BorderLayout());
			aFrame.addWindowListener(new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					e.getWindow().dispose();
					_traceCanvas = null;
				}
			});
			aFrame.getContentPane().add(_traceCanvas);
			aFrame.pack();
			StApplicationModel._ShowAtMousePoint(aFrame);
		}

		boolean condition = true;
		for (int index = 0; condition && index < normalizedDots.length; index++) {
			JunVoronoi2dDot dot = normalizedDots[index];
			condition = this.insertDot_(dot);

			if (_traceCanvas != null) {
				_traceCanvas.paintImmediately(_traceCanvas.getBounds());
			}
		}

		this.finalizeVoronoiDiagram();

		return condition;
	}

	/**
	 * Compute the voronoi diagram, the second try.
	 * 
	 * @return boolean
	 * @category computing
	 */
	protected boolean computeSecond() {
		processingPrecision = Math.pow(10.0, Math.round(Math.log(this.accuracy()) / Math.log(10) / 2));

		for (int index = 0; index < normalizedDots.length; index++) {
			JunVoronoi2dDot dot = normalizedDots[index];
			double trembler = Math.random() - 0.5;
			double x = Math.max(0, Math.min(dot.x() + trembler * processingPrecision, 1));
			double y = Math.max(0, Math.min(dot.y() + trembler * processingPrecision, 1));
			dot.x_(x);
			dot.y_(y);
		}

		boolean condition = this.computeFirst();
		if (condition == false) {
			throw SmalltalkException.Error("Error: infinite loop, because all points on a lattice, or no intersecintgpoint, or too many intersecting points.");
		}
		return condition;
	}

	/**
	 * Insert the dot.
	 * 
	 * @param voronoi2dDot
	 * @return boolean
	 * @category inserting
	 */
	protected boolean insertDot_(JunVoronoi2dDot voronoi2dDot) {
		ArrayList newPoints = new ArrayList();
		ArrayList newSides = new ArrayList();
		HashSet obsoletePoints = new HashSet();
		HashSet examinedSides = new HashSet();
		ArrayList rememberedEvaluations = new ArrayList();
		JunVoronoi2dDot nearestDot = this.nearestDot_(voronoi2dDot);
		JunVoronoi2dSide bisectorSide1 = this.bisectorSideBetween_and_(nearestDot, voronoi2dDot);
		JunVoronoi2dSide[] sides1 = this.searchSides_(nearestDot);
		for (int i = 0; i < sides1.length; i++) {
			JunVoronoi2dSide side1 = sides1[i];
			if (examinedSides.contains(side1)) {
				continue;
			}

			JunVoronoi2dPoint fromPoint = null;
			examinedSides.add(side1);
			JunVoronoi2dPoint newPoint1 = side1.intersectingPointWithSide_(bisectorSide1);
			if (newPoint1 == null) {
				continue;
			}

			JunVoronoi2dPoint toPoint = newPoint1;
			JunVoronoi2dPoint oldPoint1 = this.removingPoint_with_with_with_(newPoint1, bisectorSide1.from(), voronoi2dDot, side1);
			obsoletePoints.add(oldPoint1);
			JunVoronoi2dDot nearerDot = this.nearerDot_except_in_(oldPoint1, nearestDot, side1);
			JunVoronoi2dSide newSide1 = new JunVoronoi2dSide(fromPoint, toPoint);
			newSide1.dot1_(voronoi2dDot);
			newSide1.dot2_(nearestDot);
			JunVoronoi2dDot previousDot = nearerDot;
			int loopCounter = 1;
			while (nearestDot.equals(nearerDot) == false) {
				JunVoronoi2dSide bisectorSide2 = this.bisectorSideBetween_and_(nearerDot, voronoi2dDot);
				JunVoronoi2dSide[] sides2 = this.searchSides_(nearerDot);
				for (int j = 0; j < sides2.length; j++) {
					JunVoronoi2dSide side2 = sides2[j];
					if (examinedSides.contains(side2)) {
						continue;
					}

					fromPoint = toPoint;
					examinedSides.add(side2);
					JunVoronoi2dPoint newPoint2 = side2.intersectingPointWithSide_(bisectorSide2);
					if (newPoint2 == null) {
						continue;
					}

					toPoint = newPoint2;
					JunVoronoi2dPoint oldPoint2 = this.removingPoint_with_with_with_(newPoint2, bisectorSide2.from(), voronoi2dDot, side2);
					obsoletePoints.add(oldPoint2);
					nearerDot = this.nearerDot_except_in_(oldPoint2, nearerDot, side2);
					JunVoronoi2dSide newSide2 = new JunVoronoi2dSide(fromPoint, toPoint);
					newSide2.dot1_(voronoi2dDot);
					newSide2.dot2_(previousDot);
					if (side2.from().equals(oldPoint2)) {
						rememberedEvaluations.add(new Object[] { side2, $("from"), newPoint2 });
					} else {
						rememberedEvaluations.add(new Object[] { side2, $("to"), newPoint2 });
					}
					newPoint2.side1_(side2);
					newPoint2.side2_(newSide2);
					previousDot = nearerDot;
					newPoints.add(newPoint2);
					newSides.add(newSide2);
				}

				loopCounter++;
				if (loopCounter > this.loopMax()) {
					return false;
				}

				Thread.yield();
			}

			if (side1.from().equals(oldPoint1)) {
				rememberedEvaluations.add(new Object[] { side1, $("from"), newPoint1 });
			} else {
				rememberedEvaluations.add(new Object[] { side1, $("to"), newPoint1 });
			}
			newSide1.from_(toPoint);
			newPoint1.side1_(side1);
			newPoint1.side2_(newSide1);
			newPoints.add(newPoint1);
			newSides.add(newSide1);
		}

		for (int i = 0; i < rememberedEvaluations.size(); i++) {
			Object[] array = (Object[]) rememberedEvaluations.get(i);
			JunVoronoi2dSide side = (JunVoronoi2dSide) array[0];
			StSymbol selector = (StSymbol) array[1];
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) array[2];
			if (selector == $("from")) {
				side.from_(point);
			} else if (selector == $("to")) {
				side.to_(point);
			}
		}

		if (newPoints.size() > 0 && newSides.size() > 0) {
			int sideIndex = 1;
			for (int pointIndex = 0; pointIndex < newPoints.size(); pointIndex++, sideIndex++) {
				if (sideIndex >= newSides.size()) {
					sideIndex = 0;
				}
				JunVoronoi2dPoint newPoint = (JunVoronoi2dPoint) newPoints.get(pointIndex);
				JunVoronoi2dSide newSide = (JunVoronoi2dSide) newSides.get(sideIndex);
				newPoint.side3_(newSide);
			}
		}

		HashSet[] anArray = this.removePoints_andSides_inside_(obsoletePoints, examinedSides, newSides);
		HashSet removedPoints = anArray[0];
		HashSet removedSides = anArray[1];
		voronoiDots.add(voronoi2dDot);
		for (int i = 0; i < newPoints.size(); i++) {
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) newPoints.get(i);
			if (removedPoints.contains(point) == false) {
				voronoiPoints.add(point);
			}
		}
		for (int i = 0; i < newSides.size(); i++) {
			JunVoronoi2dSide side = (JunVoronoi2dSide) newSides.get(i);
			if (removedSides.contains(side) == false) {
				voronoiSides.add(side);
			}
		}
		return true;
	}

	/**
	 * Answer the nearest dot of the specified dot.
	 * 
	 * @param voronoi2dDot jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @category inserting
	 */
	private JunVoronoi2dDot nearestDot_(JunVoronoi2dDot voronoi2dDot) {
		double distance = this.bigNumber();
		JunVoronoi2dDot near = null;
		for (int i = 0; i < voronoiDots.size(); i++) {
			JunVoronoi2dDot dot = (JunVoronoi2dDot) voronoiDots.get(i);
			double d = voronoi2dDot.distance_(dot);
			if (distance > d) {
				distance = d;
				near = dot;
			}
		}
		return near;
	}

	/**
	 * Answer the bisector side between the specified two dots.
	 * 
	 * @param voronoi2dDot1 jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @param voronoi2dDot2 jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dSide
	 * @category inserting
	 */
	private JunVoronoi2dSide bisectorSideBetween_and_(JunVoronoi2dDot voronoi2dDot1, JunVoronoi2dDot voronoi2dDot2) {
		JunVoronoi2dPoint point1, point2;
		JunVoronoi2dLine line = voronoi2dDot1.bisector_(voronoi2dDot2);
		if (line.isVertical()) {
			point1 = line.pointAtY_(this.bigNumber() * -1);
			point2 = line.pointAtY_(this.bigNumber());
		} else {
			point1 = line.pointAtX_(this.bigNumber() * -1);
			point2 = line.pointAtX_(this.bigNumber());
		}
		return new JunVoronoi2dSide(point1, point2);
	}

	/**
	 * Search for the sides.
	 * 
	 * @param voronoi2dDot jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dSide[]
	 * @category inserting
	 */
	private JunVoronoi2dSide[] searchSides_(JunVoronoi2dDot voronoi2dDot) {
		ArrayList sides = new ArrayList();
		ArrayList obstacles = new ArrayList();
		for (int index = 0; index < voronoiSides.size(); index++) {
			JunVoronoi2dSide side = (JunVoronoi2dSide) voronoiSides.get(index);
			if (side.dot1().equals(voronoi2dDot) || side.dot2().equals(voronoi2dDot)) {
				if (side.isInvalid() && this.startIndex() <= index) {
					obstacles.add(side);
				} else {
					sides.add(side);
				}
			}
		}

		voronoiSides.removeAll(obstacles);

		return (JunVoronoi2dSide[]) sides.toArray(new JunVoronoi2dSide[sides.size()]);
	}

	/**
	 * Answer the removing point.
	 * 
	 * @param intersectingPoint jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @param voronoiDot1 jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @param voronoiDot2 jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @param voronoiSide jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dSide
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @category inserting
	 */
	private JunVoronoi2dPoint removingPoint_with_with_with_(JunVoronoi2dPoint intersectingPoint, JunVoronoi2dDot voronoiDot1, JunVoronoi2dDot voronoiDot2, JunVoronoi2dSide voronoiSide) {
		JunVoronoi2dTriangle triangle1 = new JunVoronoi2dTriangle(intersectingPoint, voronoiDot1, voronoiDot2);
		JunVoronoi2dTriangle triangle2 = new JunVoronoi2dTriangle(intersectingPoint, voronoiDot1, voronoiSide.from());
		return (triangle1.areaWithSign() * triangle2.areaWithSign() > 0) ? voronoiSide.from() : voronoiSide.to();
	}

	/**
	 * Answer the nearer dot.
	 * 
	 * @param voronoi2dPoint jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @param nearestDot jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @param voronoi2dSide jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dSide
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @category inserting
	 */
	private JunVoronoi2dDot nearerDot_except_in_(JunVoronoi2dPoint voronoi2dPoint, JunVoronoi2dDot nearestDot, JunVoronoi2dSide voronoi2dSide) {
		JunVoronoi2dDot[] dotsAtPoint1 = this.dotsAtPoint_(voronoi2dPoint);
		for (int i = 0; i < dotsAtPoint1.length; i++) {
			JunVoronoi2dDot dot = dotsAtPoint1[i];
			if (dot.equals(nearestDot) == false) {
				if (voronoi2dSide.dot1().equals(dot) || voronoi2dSide.dot2().equals(dot)) {
					return dot;
				}
			}
		}
		return null;
	}

	/**
	 * Remove the obsolete points and the examined sides.
	 * 
	 * @param obsoletePoints java.util.HashSet[]
	 * @param examinedSides java.util.HashSet[]
	 * @param newSides java.util.ArrayList
	 * @return java.util.HashSet[]
	 * @category inserting
	 */
	private HashSet[] removePoints_andSides_inside_(HashSet obsoletePoints, HashSet examinedSides, ArrayList newSides) {
		HashSet removedPoints = new HashSet();
		HashSet removedSides = new HashSet();

		for (Iterator iterator = obsoletePoints.iterator(); iterator.hasNext();) {
			JunVoronoi2dPoint obsoletePoint = (JunVoronoi2dPoint) iterator.next();
			obsoletePoint.side1_(null);
			obsoletePoint.side2_(null);
			obsoletePoint.side3_(null);
			if (voronoiPoints.remove(obsoletePoint)) {
				removedPoints.add(obsoletePoint);
			}
		}

		for (Iterator iterator = examinedSides.iterator(); iterator.hasNext();) {
			JunVoronoi2dSide examinedSide = (JunVoronoi2dSide) iterator.next();
			if (obsoletePoints.contains(examinedSide.from()) || obsoletePoints.contains(examinedSide.to())) {
				JunVoronoi2dSide targetSide = null;
				for (int i = 0; i < newSides.size(); i++) {
					JunVoronoi2dSide newSide = (JunVoronoi2dSide) newSides.get(i);
					if (examinedSide.intersectingPointWithSide_(newSide) != null) {
						targetSide = newSide;
						break;
					}
				}
				if (targetSide == null) {
					if (voronoiPoints.remove(examinedSide.from())) {
						removedPoints.add(examinedSide.from());
					}
					if (voronoiPoints.remove(examinedSide.to())) {
						removedPoints.add(examinedSide.to());
					}
					examinedSide.from_(null);
					examinedSide.to_(null);
					examinedSide.dot1_(null);
					examinedSide.dot2_(null);
					if (voronoiSides.remove(examinedSide)) {
						removedSides.add(examinedSide);
					}
				} else {
					if (examinedSide.isInvalid() && this.startIndex() <= voronoiSides.indexOf(examinedSide)) {
						if (voronoiSides.remove(examinedSide)) {
							removedSides.add(examinedSide);
						}
					}
				}
			} else {
				if (examinedSide.isInvalid() && this.startIndex() <= voronoiSides.indexOf(examinedSide)) {
					if (voronoiSides.remove(examinedSide)) {
						removedSides.add(examinedSide);
					}
				}
			}
		}

		return new HashSet[] { removedPoints, removedSides };
	}

	/**
	 * Answer my current tracing status.
	 * 
	 * @return boolean
	 * @category tracing
	 */
	public boolean trace() {
		return traceBoolean;
	}

	/**
	 * Set my current tracing status.
	 * 
	 * @param b boolean
	 * @category tracing 
	 */
	public void trace_(boolean b) {
		traceBoolean = b;
	}

	/**
	 * Display the receiver on the graphics for tracing.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category tracing
	 */
	protected void traceDisplayOn_(Graphics aGraphics) {
		if (_traceCanvas == null || _traceCanvas.isShowing() == false) {
			return;
		}

		Rectangle bounds = new Rectangle(_traceCanvas.getSize());
		Point origin = new Point(bounds.width / 2, bounds.height / 2);
		int factor = 200;
		origin.x -= factor * 0.5;
		origin.y -= factor * 0.5;

		Graphics2D graphics2d = (Graphics2D) aGraphics;
		graphics2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		graphics2d.setColor(Color.white);
		graphics2d.fillRect(bounds.x, bounds.y, bounds.width, bounds.height);

		graphics2d.setColor(Color.red);
		graphics2d.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		for (int i = 0; i < voronoiDots.size(); i++) {
			JunVoronoi2dDot dot = (JunVoronoi2dDot) voronoiDots.get(i);
			int x = (int) Math.round(dot.x() * factor + origin.x);
			int y = (int) Math.round(dot.y() * factor + origin.y);
			StRectangle box = StRectangle.Origin_extent_(new Point(x, y), new Point(1, 1)).expandedBy_(2);
			this.displayBoxOn_box_clip_(graphics2d, box, bounds);
		}

		for (int i = 0; i < voronoiPoints.size(); i++) {
			JunVoronoi2dPoint point = (JunVoronoi2dPoint) voronoiPoints.get(i);
			int x = (int) Math.round(point.x() * factor + origin.x);
			int y = (int) Math.round(point.y() * factor + origin.y);
			StRectangle box = StRectangle.Origin_extent_(new Point(x, y), new Point(1, 1)).expandedBy_(3);
			graphics2d.setColor(Color.green);
			this.displayBoxOn_box_clip_(graphics2d, box, bounds);

			ArrayList sides = new ArrayList(3);
			if (point.side1() != null) {
				sides.add(point.side1());
			}
			if (point.side2() != null) {
				sides.add(point.side2());
			}
			if (point.side3() != null) {
				sides.add(point.side3());
			}

			graphics2d.setColor(Color.blue);
			graphics2d.setStroke(new BasicStroke(3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			for (int j = 0; j < sides.size(); j++) {
				JunVoronoi2dSide side = (JunVoronoi2dSide) sides.get(j);
				if (side.isValid()) {
					int x1 = (int) Math.round(side.from().x() * factor + origin.x);
					int y1 = (int) Math.round(side.from().y() * factor + origin.y);
					int x2 = (int) Math.round(side.to().x() * factor + origin.x);
					int y2 = (int) Math.round(side.to().y() * factor + origin.y);
					this.displayLineOn_from_to_clip_(graphics2d, new Point(x1, y1), new Point(x2, y2), bounds);
				}
			}
		}

		graphics2d.setColor(Color.cyan);
		graphics2d.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		for (int i = 0; i < voronoiSides.size(); i++) {
			JunVoronoi2dSide side = (JunVoronoi2dSide) voronoiSides.get(i);
			int x1 = (int) Math.round(side.from().x() * factor + origin.x);
			int y1 = (int) Math.round(side.from().y() * factor + origin.y);
			int x2 = (int) Math.round(side.to().x() * factor + origin.x);
			int y2 = (int) Math.round(side.to().y() * factor + origin.y);
			this.displayLineOn_from_to_clip_(graphics2d, new Point(x1, y1), new Point(x2, y2), bounds);
		}

		graphics2d.setColor(Color.black);
		graphics2d.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));

		StRectangle box = StRectangle.Origin_extent_(new Point(0, 0), new Point(1, 1)).scaledBy_(factor, factor).translatedBy_(origin.x, origin.y);
		graphics2d.drawRect(box.x(), box.y(), box.width(), box.height());
	}

	/**
	 * Display the box on the graphics.
	 * 
	 * @param graphics java.awt.Grahpics2D
	 * @param box jp.co.sra.smalltalk.StRectangle
	 * @param bounds java.awt.Rectangle
	 * @category displaying
	 */
	protected void displayBoxOn_box_clip_(Graphics2D graphics, StRectangle box, Rectangle bounds) {
		if (box.intersects_(new StRectangle(bounds))) {
			graphics.fillRect(box.x(), box.y(), box.width(), box.height());
		}

	}

	/**
	 * Display a line between the two points.
	 * 
	 * @param graphics java.awt.Graphics2D
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @param clippingBox java.awt.Rectangle
	 * @category displaying
	 */
	protected void displayLineOn_from_to_clip_(Graphics2D graphics, Point fromPoint, Point toPoint, Rectangle clippingBox) {
		double slope = Math.abs(toPoint.y - fromPoint.y) / Math.max(Math.abs(toPoint.x - fromPoint.x), this.accuracy());
		StRectangle bounds = new StRectangle(clippingBox);
		bounds.expandedBy_(new StRectangle(0, 0, 1, 1));
		boolean boolean1 = bounds.containsPoint_(fromPoint);
		boolean boolean2 = bounds.containsPoint_(toPoint);
		bounds.insetBy_(new StRectangle(0, 0, 1, 1));

		if (boolean1 && boolean2) {
			graphics.drawLine(fromPoint.x, fromPoint.y, toPoint.x, toPoint.y);
		} else if (!boolean1 && !boolean2) {
			boolean boolean3 = false;
			JunVoronoi2dSide side = new JunVoronoi2dSide(fromPoint, toPoint);
			if (side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topLeft(), bounds.bottomLeft())) != null) {
				boolean3 = true;
			}
			if (side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topRight(), bounds.bottomRight())) != null) {
				boolean3 = true;
			}
			if (side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topLeft(), bounds.topRight())) != null) {
				boolean3 = true;
			}
			if (side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.bottomLeft(), bounds.bottomRight())) != null) {
				boolean3 = true;
			}
			if (boolean3) {
				JunVoronoi2dLine line = new JunVoronoi2dLine(fromPoint, toPoint);
				JunVoronoi2dPoint point1, point2;
				if (slope > 1.0) {
					point1 = line.pointAtY_(bounds.top());
					point2 = line.pointAtY_(bounds.bottom());
				} else {
					point1 = line.pointAtX_(bounds.top());
					point2 = line.pointAtX_(bounds.bottom());
				}
				graphics.drawLine((int) Math.round(point1.x()), (int) Math.round(point1.y()), (int) Math.round(point2.x()), (int) Math.round(point2.y()));
			}
		} else {
			JunVoronoi2dPoint point1, point2;
			if (boolean1) {
				point1 = new JunVoronoi2dPoint(fromPoint);
				point2 = new JunVoronoi2dPoint(toPoint);
			} else {
				point1 = new JunVoronoi2dPoint(toPoint);
				point2 = new JunVoronoi2dPoint(fromPoint);
			}
			JunVoronoi2dSide side = new JunVoronoi2dSide(fromPoint, toPoint);
			JunVoronoi2dPoint point = side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topLeft(), bounds.bottomLeft()));
			if (point != null) {
				point2 = point;
			}
			point = side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topRight(), bounds.bottomRight()));
			if (point != null) {
				point2 = point;
			}
			point = side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.topLeft(), bounds.topRight()));
			if (point != null) {
				point2 = point;
			}
			point = side.intersectingPointWithSide_(new JunVoronoi2dSide(bounds.bottomLeft(), bounds.bottomRight()));
			if (point != null) {
				point2 = point;
			}
			graphics.drawLine((int) Math.round(point1.x()), (int) Math.round(point1.y()), (int) Math.round(point2.x()), (int) Math.round(point2.y()));
		}
	}

	/**
	 * Display the loop on the graphics.
	 * 
	 * @param graphics java.awt.Graphics2D
	 * @param points java.awt.Point[]
	 * @param clippingBox java.awt.Rectangle
	 * @category displaying
	 */
	protected void displayLoopOn_points_clip_(Graphics2D graphics, Point[] points, Rectangle clippingBox) {
		if (points == null || points.length < 3) {
			return;
		}

		for (int i = 1; i < points.length; i++) {
			Point fromPoint = points[i - 1];
			Point toPoint = points[i];
			this.displayLineOn_from_to_clip_(graphics, fromPoint, toPoint, clippingBox);
		}
		if (points[0].equals(points[points.length - 1]) == false) {
			this.displayLineOn_from_to_clip_(graphics, points[points.length - 1], points[0], clippingBox);
		}
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		PrintWriter pw = (aWriter instanceof PrintWriter) ? (PrintWriter) aWriter : new PrintWriter(aWriter);
		pw.println("diagram ( ");
		Jun2dPoint[] dots = this.dots();
		for (int i = 0; i < dots.length; i++) {
			pw.print('\t');
			dots[i].printOn_(aWriter);
			pw.println();
		}
		pw.println(")");
	}

	/**
	 * Answer the big size.
	 * 
	 * @return double
	 * @category private
	 */
	protected double bigSize() {
		return 100.0;
	}

	/**
	 * Answer the big factor.
	 * 
	 * @return double
	 * @category private
	 */
	protected double bigFactor() {
		return 100.0;
	}

	/**
	 * Answer the big number.
	 * 
	 * @return double
	 * @category private
	 */
	protected double bigNumber() {
		return this.bigSize() * this.bigFactor();
	}

	/**
	 * Answer the max number for the loop.
	 * 
	 * @return int
	 * @category private 
	 */
	protected int loopMax() {
		return 5000;
	}

	/**
	 * Normalize the specified dots.
	 * 
	 * @param arrayOfDots jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot[]
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot[]
	 * @category private
	 */
	protected JunVoronoi2dDot[] normalizedDots_(JunVoronoi2dDot[] arrayOfDots) {
		JunVoronoi2dDot[] dots = new JunVoronoi2dDot[arrayOfDots.length];
		System.arraycopy(arrayOfDots, 0, dots, 0, dots.length);

		double minX = Double.MAX_VALUE;
		double minY = Double.MAX_VALUE;
		double maxX = Double.MIN_VALUE;
		double maxY = Double.MIN_VALUE;
		for (int i = 0; i < dots.length; i++) {
			minX = Math.min(minX, dots[i].x());
			minY = Math.min(minY, dots[i].y());
			maxX = Math.max(maxX, dots[i].x());
			maxY = Math.max(maxY, dots[i].y());
		}
		if (0 <= minX && maxX <= 1 && 0 <= minY && maxY <= 1) {
			translationDelta = new JunVoronoi2dDot(0, 0);
			scalingFactor = new JunVoronoi2dDot(1, 1);
			return dots;
		}

		JunVoronoi2dDot delta = new JunVoronoi2dDot(minX, minY);
		for (int i = 0; i < dots.length; i++) {
			dots[i] = new JunVoronoi2dDot(dots[i].x() - delta.x(), dots[i].y() - delta.y());
		}
		double width = Math.max(maxX - minX, this.accuracy());
		double height = Math.max(maxY - minY, this.accuracy());
		double factor = Math.max(width, height);
		for (int i = 0; i < dots.length; i++) {
			dots[i] = new JunVoronoi2dDot(dots[i].x() / factor, dots[i].y() / factor);
		}
		translationDelta = delta;
		scalingFactor = new JunVoronoi2dDot(factor, factor);
		return dots;
	}

}
