package jp.co.sra.jun.topology.grapher;

import java.awt.BasicStroke;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;
import jp.co.sra.jun.topology.graph.JunElementalNode;

/**
 * JunGrapherViewSwing class
 * 
 *  @author    nisinaka
 *  @created   2006/04/14 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun602 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGrapherViewSwing.java,v 8.11 2008/02/20 06:33:13 nisinaka Exp $
 */
public class JunGrapherViewSwing extends JunAbstractViewJPanel implements JunGrapherView {

	protected Point scrollAmount;
	protected Jun2dPoint _scaleFactor;

	/**
	 * Create a new instance of JunGrapherViewSwing and initialize it.
	 *
	 * @param aGrapher jp.co.sra.jun.topology.graph.JunElementalGrapher
	 * @category Instance creation
	 */
	public JunGrapherViewSwing(JunGrapher aGrapher) {
		super(aGrapher);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		scrollAmount = new Point(0, 0);
		this._flushScaleFactor();
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setPreferredSize(DefaultBounds.getSize());
		this.setBackground(DefaultBackgroundColor);
	}

	/**
	 * Answer my model as JunGrapher.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGrapher
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#getGrapher()
	 * @category model accessing
	 */
	public JunGrapher getGrapher() {
		return (JunGrapher) this.model();
	}

	/**
	 * Create an instance of my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunGrapherController();
	}

	/**
	 * Answer my scale factor for the bird view.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#_scaleFactor()
	 * @category accessing
	 */
	public Jun2dPoint _scaleFactor() {
		if (_scaleFactor == null) {
			Rectangle viewBounds = this.getBounds();
			StRectangle modelBoundingBox = this.getGrapher().boundingBox();
			double scaleFactorX = (double) viewBounds.width / modelBoundingBox.width();
			double scaleFactorY = (double) viewBounds.height / modelBoundingBox.height();
			_scaleFactor = new Jun2dPoint(scaleFactorX, scaleFactorY);
		}
		return _scaleFactor;
	}

	/**
	 * Flush my scale factor for the bird view.
	 * 
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#_flushScaleFactor()
	 * @category flushing
	 */
	public void _flushScaleFactor() {
		_scaleFactor = null;
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		if (_scaleFactor == null) {
			this.getGrapher().displayOn_at_(aGraphics, this.convertModelPointToViewPoint_(aPoint));
		} else {
			Graphics2D aGraphics2D = (Graphics2D) aGraphics;

			StRectangle boundingBox = this.getGrapher().boundingBox();
			int x = boundingBox.originX();
			int y = boundingBox.originY();
			this.getGrapher().displayOn_at_scaledBy_(aGraphics2D, new Point(-x, -y), _scaleFactor);

			Point scrollAmount = this.scrollAmount();
			int dx = 0 - x - scrollAmount.x;
			int dy = 0 - y - scrollAmount.y;
			StRectangle aBox = new StRectangle(this.getBounds());
			aBox = aBox.translatedBy_(dx, dy).scaledBy_(_scaleFactor.x(), _scaleFactor.y());
			aGraphics2D.setColor(this.getGrapher().selectionBorderColor());
			aGraphics2D.setStroke(new BasicStroke(this.getGrapher().selectionBorderWidth()));
			aGraphics2D.drawRect(aBox.x(), aBox.y(), aBox.width(), aBox.height());
		}
	}

	/**
	 * Redislay the view.
	 * 
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#redisplay()
	 * @category displaying
	 */
	public void redisplay() {
		if (this.isVisible()) {
			this.paintImmediately(0, 0, this.getWidth(), this.getHeight());
		}
	}

	/**
	 * Answer the scroll amount.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#scrollAmount()
	 * @category scrolling
	 */
	public Point scrollAmount() {
		return scrollAmount;
	}

	/**
	 * Set the scroll amount.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#scrollAmount_(java.awt.Point)
	 * @category scrolling
	 */
	public void scrollAmount_(Point aPoint) {
		scrollAmount = aPoint;
	}

	/**
	 * Scroll to the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#scrollTo_(java.awt.Point)
	 * @category scrolling
	 */
	public void scrollTo_(Point aPoint) {
		this.scrollAmount_(aPoint);
		this.redisplay();
	}

	/**
	 * Scroll by the specified amount.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#scrollBy_(java.awt.Point)
	 * @category scrolling 
	 */
	public void scrollBy_(Point aPoint) {
		scrollAmount.translate(aPoint.x, aPoint.y);
		this.redisplay();
	}

	/**
	 * Scroll for the specified node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#scrollFor_(jp.co.sra.jun.topology.graph.JunElementalNode)
	 * @category scrolling
	 */
	public void scrollFor_(JunElementalNode aNode) {
		Point viewCenter = new StRectangle(this.getBounds()).center();
		Point nodeCenter = aNode.center();
		this.scrollAmount_(new Point(viewCenter.x - nodeCenter.x, viewCenter.y - nodeCenter.y));
		this.redisplay();
	}

	/**
	 * Convert a model point to a view point, and answer it.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#convertModelPointToViewPoint_(java.awt.Point)
	 * @category converting
	 */
	public Point convertModelPointToViewPoint_(Point aPoint) {
		int x = aPoint.x + this.scrollAmount().x;
		int y = aPoint.y + this.scrollAmount().y;
		return new Point(x, y);
	}

	/**
	 * Convert a view point to a model point, and answer it.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.topology.grapher.JunGrapherView#convertViewPointToModelPoint_(java.awt.Point)
	 * @category converting
	 */
	public Point convertViewPointToModelPoint_(Point aPoint) {
		int x = aPoint.x - this.scrollAmount().x;
		int y = aPoint.y - this.scrollAmount().y;
		return new Point(x, y);
	}

	/**
	 * Show the popup menu at the specified point on the view.
	 *
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
	 * @category popup menu
	 */
	public void _showPopupMenu(int x, int y) {
		Point aPoint = this.convertViewPointToModelPoint_(new Point(x, y));
		this.popupMenu_(this.getGrapher()._popupMenuAt(aPoint));
		super._showPopupMenu(x, y);
	}

}
