package jp.co.sra.jun.topology.grapher;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collection;

import javax.swing.SwingUtilities;

import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;
import jp.co.sra.jun.topology.graph.JunElementalNode;

/**
 * JunGrapherController class
 * 
 *  @author    nisinaka
 *  @created   2006/04/14 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun614 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGrapherController.java,v 8.10 2008/02/20 06:33:13 nisinaka Exp $
 */
public class JunGrapherController extends JunAbstractController {

	protected boolean freehandState;
	protected Thread _waitThread;
	protected StSymbol _activity;
	protected Point _previousPoint;
	protected JunElementalNode[] _movingNodes;
	protected ArrayList _freehandPoints;
	protected boolean _altDown;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StController#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		super.initialize();
		freehandState = false;
		_waitThread = null;
		_activity = null;
		_previousPoint = null;
		_movingNodes = null;
		_freehandPoints = null;
		_altDown = false;
	}

	/**
	 * Release the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.breakDependents();

		this.view().toComponent().removeKeyListener(this);
		this.view().toComponent().removeMouseListener(this);
		this.view().toComponent().removeMouseMotionListener(this);
	}

	/**
	 * Add the myself as a listener of the view.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView aView) {
		this.view().toComponent().addKeyListener(this);
		this.view().toComponent().addMouseListener(this);
		this.view().toComponent().addMouseMotionListener(this);
	}

	/**
	 * Answer my current freehand state.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean freehandState() {
		return freehandState;
	}

	/**
	 * Set my new current freehand state.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void freehandState_(boolean aBoolean) {
		freehandState = aBoolean;
	}

	/**
	 * Answer my model as JunGrapher.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunGrapher
	 * @category model accessing
	 */
	public JunGrapher getGrapher() {
		return this.getGrapherView().getGrapher();
	}

	/**
	 * Answer my view as JunGrapherView.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunGrapherView
	 * @category view accessing
	 */
	public JunGrapherView getGrapherView() {
		return (JunGrapherView) this.view();
	}

	/**
	 * Invoked when a key is pressed on the view.
	 * 
	 * @param e java.awt.event.KeyEvent
	 * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
	 * @category key events
	 */
	public void keyPressed(KeyEvent e) {
		if (_activity == $("grab")) {
			if (_altDown == false && e.isAltDown() == true) {
				this._setBirdViewVisible(true);
			}
		}

		_altDown = e.isAltDown();
	}

	/**
	 * Invoked when a key is released on the view.
	 * 
	 * @param e java.awt.event.KeyEvent
	 * @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
	 * @category key events
	 */
	public void keyReleased(KeyEvent e) {
		if (_activity == $("grab")) {
			if (_altDown == true && e.isAltDown() == false) {
				this._setBirdViewVisible(false);
			}
		}

		_altDown = e.isAltDown();
	}

	/**
	 * Invoked when a mouse button has been pressed on the view.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent e) {
		this._activity(null);
		this.view().toComponent().requestFocus();

		if (SwingUtilities.isLeftMouseButton(e)) {
			_previousPoint = e.getPoint();
			_waitThread = new Thread() {
				public void run() {
					synchronized (JunGrapherController.this) {
						try {
							JunGrapherController.this.wait(250);
						} catch (InterruptedException e) {
							e.printStackTrace();
						}

						if (_waitThread == null) {
							// The single or double click activity has been done.
						} else if (_activity == null) {
							// Do the grab activity
							Point aPoint = getGrapherView().convertViewPointToModelPoint_(_previousPoint);
							JunElementalNode aNode = getGrapher().whichAt_(aPoint);
							if (aNode == null) {
								_activity($("grab"));

								if (_altDown) {
									_setBirdViewVisible(true);
								}
							} else {
								_activity($("moveNodes"));

								Collection selections = getGrapher().selections();
								if (selections.contains(aNode)) {
									_movingNodes = (JunElementalNode[]) selections.toArray(new JunElementalNode[selections.size()]);
								} else {
									_movingNodes = new JunElementalNode[] { aNode };
								}
							}

							_waitThread = null;
						}
					}
				}
			};
			_waitThread.start();
		} else if (SwingUtilities.isRightMouseButton(e) && e.isAltDown()) {
			this._activity($("freehand"));
			this._freehandDragged(e);
		} else {
			super.mousePressed(e);
		}
	}

	/**
	 * Invoked when a mouse button has been released on the view.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent e) {
		if (SwingUtilities.isLeftMouseButton(e)) {
			synchronized (this) {
				if (_waitThread != null && _waitThread.isAlive()) {
					_waitThread = null;
					this.notify();

					switch (e.getClickCount()) {
						case 1:
							this._activity($("singleClick"));
							break;
						case 2:
							this._activity($("doubleClick"));
							break;
					}
				}
			}

			if (_activity == $("singleClick")) {
				Point aPoint = this.getGrapherView().convertViewPointToModelPoint_(e.getPoint());
				JunElementalNode aNode = this.getGrapher().whichAt_(aPoint);
				if (aNode == null) {
					this.getGrapher().selections_(null);
				} else {
					if (e.isShiftDown()) {
						ArrayList aList = new ArrayList(this.getGrapher().selections());
						if (aList.contains(aNode)) {
							aList.remove(aNode);
						} else {
							aList.add(aNode);
						}
						this.getGrapher().selections_(aList);
					} else {
						this.getGrapher().selection_(aNode);
					}
				}
			} else if (_activity == $("doubleClick")) {
				Point aPoint = this.getGrapherView().convertViewPointToModelPoint_(e.getPoint());
				JunElementalNode aNode = this.getGrapher().whichAt_(aPoint);
				if (aNode != null) {
					this.getGrapher().selectAndArrangeAndScrollFor_(aNode);
				}
			} else if (_activity == $("grab")) {
				this._grabReleased(e);
			} else if (_activity == $("moveNodes")) {
				this.getGrapher().flushBoundingBox();
			}
		} else if (_activity == $("freehand")) {
			this._freehandReleased(e);
		} else {
			super.mouseReleased(e);
		}

		this._activity(null);
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param e java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent e) {
		if (_activity == null) {
			this._activity($("none"));
		}

		if (_activity == $("grab")) {
			this._grabDragged(e);
		} else if (_activity == $("moveNodes")) {
			this._moveNodesDragged(e);
		} else if (_activity == $("freehand")) {
			this._freehandDragged(e);
		}
	}

	/**
	 * Mouse released for the grab activity.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @category control activities
	 */
	protected void _grabReleased(MouseEvent e) {
		if (_altDown) {
			this._setBirdViewVisible(false);
		}
	}

	/**
	 * Mouse dragged for the grab activity.
	 * 
	 * @param e java.awt.event.MouseEvent 
	 * @category control activities
	 */
	protected void _grabDragged(MouseEvent e) {
		Point currentPoint = e.getPoint();
		if (currentPoint.equals(_previousPoint)) {
			return;
		}

		int dx = currentPoint.x - _previousPoint.x;
		int dy = currentPoint.y - _previousPoint.y;

		if (e.isAltDown()) {
			this._birdViewDragged(e);
		} else {
			this.getGrapherView()._flushScaleFactor();
			this.getGrapherView().scrollBy_(new Point(dx, dy));
		}

		_previousPoint = currentPoint;
	}

	/**
	 * Mouse dragged for the move nodes activity.
	 * 
	 * @param e java.awt.event.MouseEvent 
	 * @category control activities
	 */
	protected void _moveNodesDragged(MouseEvent e) {
		Point currentPoint = e.getPoint();
		if (currentPoint.equals(_previousPoint)) {
			return;
		}

		int dx = currentPoint.x - _previousPoint.x;
		int dy = currentPoint.y - _previousPoint.y;

		for (int i = 0; i < _movingNodes.length; i++) {
			Point aPoint = _movingNodes[i].locationPoint();
			_movingNodes[i].locationPoint_(new Point(aPoint.x + dx, aPoint.y + dy));
			if (this.getGrapher().nodeSettings().stuff() == _movingNodes[i]) {
				this.getGrapher().nodeSettingsFor_(_movingNodes[i]);
			}
		}
		this.getGrapher().changed_($("graph"));

		_previousPoint = currentPoint;
	}

	/**
	 * Mouse released for the freehand activity.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @category control activities
	 */
	protected void _freehandReleased(MouseEvent e) {
		if (_freehandPoints == null || _freehandPoints.size() < 2 || e.isAltDown() == false) {
			this.getGrapherView().redisplay();
			return;
		}

		Point[] points = new Point[_freehandPoints.size()];
		for (int i = 0; i < points.length; i++) {
			points[i] = this.getGrapherView().convertViewPointToModelPoint_((Point) _freehandPoints.get(i));
		}
		JunElementalNode startNode = this.getGrapher().whichAt_(points[0]);
		JunElementalNode endNode = this.getGrapher().whichAt_(points[points.length - 1]);
		if (startNode != null && endNode != null) {
			this.getGrapher().freehandConnect_with_(startNode, endNode);
		} else {
			this.getGrapher().freehandPoints_(points, e);
		}

		_freehandPoints = null;
	}

	/**
	 * Mouse dragged for the freehand activity.
	 * 
	 * @param e java.awt.event.MouseEvent 
	 * @category control activities
	 */
	protected void _freehandDragged(MouseEvent e) {
		Point currentPoint = e.getPoint();
		if (currentPoint.equals(_previousPoint)) {
			return;
		}

		if (_freehandPoints == null) {
			_freehandPoints = new ArrayList();
		}
		_freehandPoints.add(currentPoint);

		if (_freehandPoints.size() > 1) {
			Graphics2D aGraphics = null;
			try {
				aGraphics = (Graphics2D) this.getGrapherView().toComponent().getGraphics();

				aGraphics.setColor(Color.red);
				aGraphics.setStroke(new BasicStroke(1));

				for (int i = 1; i < _freehandPoints.size(); i++) {
					Point p1 = (Point) _freehandPoints.get(i - 1);
					Point p2 = (Point) _freehandPoints.get(i);
					aGraphics.drawLine(p1.x, p1.y, p2.x, p2.y);
				}

			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
				}
			}
		}

		_previousPoint = currentPoint;
	}

	/**
	 * Mouse dragged for the bird view activity.
	 * 
	 * @param e java.awt.event.MouseEvent 
	 * @category control activities
	 */
	protected synchronized void _birdViewDragged(MouseEvent e) {
		Point currentPoint = e.getPoint();
		if (currentPoint.equals(_previousPoint)) {
			return;
		}

		Jun2dPoint scaleFactor = this.getGrapherView()._scaleFactor();
		Point scrollAmount = this.getGrapherView().scrollAmount();
		int scrollAmountX = scrollAmount.x - (int) Math.round((currentPoint.x - _previousPoint.x) / scaleFactor.x());
		int scrollAmountY = scrollAmount.y - (int) Math.round((currentPoint.y - _previousPoint.y) / scaleFactor.y());
		this.getGrapherView().scrollAmount_(new Point(scrollAmountX, scrollAmountY));
		this.getGrapherView().redisplay();

		_previousPoint = currentPoint;
	}

	/**
	 * Set either the bird view visible or not.
	 * 
	 * @param beVisible boolean
	 * @category private
	 */
	protected synchronized void _setBirdViewVisible(boolean beVisible) {
		Point scrollAmount = this.getGrapherView().scrollAmount();
		StRectangle modelBoundingBox = this.getGrapher().boundingBox();
		Jun2dPoint scaleFactor = this.getGrapherView()._scaleFactor();

		if (beVisible) {
			Point aPoint = this.cursorPoint();
			int x = (int) Math.round((aPoint.x - scrollAmount.x - modelBoundingBox.originX()) * scaleFactor.x());
			int y = (int) Math.round((aPoint.y - scrollAmount.y - modelBoundingBox.originY()) * scaleFactor.y());
			aPoint = new Point(x, y);
			this.cursorPoint_(aPoint);
			_previousPoint = aPoint;
		} else {
			Point aPoint = this.cursorPoint();
			int x = (int) (Math.round(aPoint.x / scaleFactor.x())) + scrollAmount.x + modelBoundingBox.originX();
			int y = (int) (Math.round(aPoint.y / scaleFactor.y())) + scrollAmount.y + modelBoundingBox.originY();
			aPoint = new Point(x, y);
			this.cursorPoint_(aPoint);
			_previousPoint = aPoint;

			this.getGrapherView()._flushScaleFactor();
		}

		this.getGrapherView().redisplay();
	}

	/**
	 * Set the current activity.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category private
	 */
	protected void _activity(StSymbol aSymbol) {
		if (_activity == aSymbol) {
			return;
		}

		_activity = aSymbol;

		if (_activity == $("singleClick")) {
			this.getGrapherView().toComponent().setCursor(JunCursors.CrossCursor());
		} else if (_activity == $("grab")) {
			this.getGrapherView().toComponent().setCursor(JunCursors.Quarters2Cursor());
		} else if (_activity == $("moveNodes")) {
			this.getGrapherView().toComponent().setCursor(JunCursors.QuartersCursor());
		} else if (_activity == $("freehand")) {
			this.getGrapherView().toComponent().setCursor(JunCursors.PencilCursor());
		} else {
			this.getGrapherView().toComponent().setCursor(null);
		}
	}

}
