package jp.co.sra.jun.topology.graph;

import java.awt.Color;
import java.awt.Point;
import java.util.HashMap;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.tables.JunAttributeTable;

/**
 * JunElementalNodeSettings class
 * 
 *  @author    nisinaka
 *  @created   2006/04/17 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun593 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalNodeSettings.java,v 8.10 2008/02/20 06:33:02 nisinaka Exp $
 */
public class JunElementalNodeSettings extends JunElementalStuffSettings {

	protected StValueHolder labelStringHolder;
	protected StValueHolder xLocationStringHolder;
	protected StValueHolder yLocationStringHolder;
	protected StValueHolder foregroundColorHolder;
	protected StValueHolder backgroundColorHolder;
	protected StValueHolder borderColorHolder;
	protected StValueHolder borderWidthHolder;

	/**
	 * Create a new instance of JunElementalNodeSettings and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalNodeSettings() {
		super();
	}

	/**
	 * Create a new instance of JunElementalNodeSettings and initialize it.
	 *
	 * @param anElementalNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category Instance creation
	 */
	public JunElementalNodeSettings(JunElementalNode anElementalNode) {
		this.stuff_(anElementalNode);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		StBlockClosure updateSettingsBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				updateSettings();
				return null;
			}
		};

		labelStringHolder = new StValueHolder(JunElementalNode.DefaultLabelString());
		labelStringHolder.compute_(updateSettingsBlock);

		xLocationStringHolder = new StValueHolder(String.valueOf(JunElementalNode.DefaultLocationPoint().x));
		xLocationStringHolder.compute_(updateSettingsBlock);

		yLocationStringHolder = new StValueHolder(String.valueOf(JunElementalNode.DefaultLocationPoint().y));
		yLocationStringHolder.compute_(updateSettingsBlock);

		foregroundColorHolder = new StValueHolder(JunElementalNode.DefaultForegroundColor());
		foregroundColorHolder.compute_(updateSettingsBlock);

		backgroundColorHolder = new StValueHolder(JunElementalNode.DefaultBackgroundColor());
		backgroundColorHolder.compute_(updateSettingsBlock);

		borderColorHolder = new StValueHolder(JunElementalNode.DefaultBorderColor());
		borderColorHolder.compute_(updateSettingsBlock);

		borderWidthHolder = new StValueHolder(JunElementalNode.DefaultBorderWidth());
		borderWidthHolder.compute_(updateSettingsBlock);
	}

	/**
	 * Answer the current settings.
	 * 
	 * @return java.util.Map
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#currentSettings()
	 * @category accessing
	 */
	public Map currentSettings() {
		Point locationPoint = null;
		try {
			locationPoint = new Point(Integer.parseInt((String) this.xLocationStringHolder().value()), Integer.parseInt((String) this.yLocationStringHolder().value()));
		} catch (NumberFormatException e) {
			System.out.println(e);
		}

		HashMap aMap = new HashMap(6);
		aMap.put($("labelString"), this.labelStringHolder().value());
		aMap.put($("locationPoint"), locationPoint);
		aMap.put($("foregroundColor"), this.foregroundColorHolder().value());
		aMap.put($("backgroundColor"), this.backgroundColorHolder().value());
		aMap.put($("borderColor"), this.borderColorHolder().value());
		aMap.put($("borderWidth"), this.borderWidthHolder().value());
		return aMap;
	}

	/**
	 * Answer the stuff settings.
	 * 
	 * @return java.util.Map
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#currentSettings()
	 * @category accessing
	 */
	public Map stuffSettings() {
		HashMap aMap = new HashMap(7);
		if (this.stuff() == null) {
			JunAttributeTable attributes = this.defaultAttributes();
			aMap.put($("labelStringHolder"), attributes.at_($("labelString")));
			aMap.put($("xLocationStringHolder"), String.valueOf(((Point) attributes.at_($("locationPoint"))).x));
			aMap.put($("yLocationStringHolder"), String.valueOf(((Point) attributes.at_($("locationPoint"))).y));
			aMap.put($("foregroundColorHolder"), attributes.at_($("foregroundColor")));
			aMap.put($("backgroundColorHolder"), attributes.at_($("backgroundColor")));
			aMap.put($("borderColorHolder"), attributes.at_($("borderColor")));
			aMap.put($("borderWidthHolder"), attributes.at_($("borderWidth")));
		} else if (this.stuff() instanceof JunElementalNode) {
			JunElementalNode aNode = (JunElementalNode) this.stuff();
			aMap.put($("labelStringHolder"), aNode.labelString());
			aMap.put($("xLocationStringHolder"), String.valueOf(aNode.locationPoint().x));
			aMap.put($("yLocationStringHolder"), String.valueOf(aNode.locationPoint().y));
			aMap.put($("foregroundColorHolder"), aNode.foregroundColor());
			aMap.put($("backgroundColorHolder"), aNode.backgroundColor());
			aMap.put($("borderColorHolder"), aNode.borderColor());
			aMap.put($("borderWidthHolder"), new Integer(aNode.borderWidth()));
		}
		return aMap;
	}

	/**
	 * Answer the value holder for the label string.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder labelStringHolder() {
		return labelStringHolder;
	}

	/**
	 * Answer the value holder for the x-coordinate of the location.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder xLocationStringHolder() {
		return xLocationStringHolder;
	}

	/**
	 * Answer the value holder for the x-coordinate of the location.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder yLocationStringHolder() {
		return yLocationStringHolder;
	}

	/**
	 * Answer the value holder for the foreground color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder foregroundColorHolder() {
		return foregroundColorHolder;
	}

	/**
	 * Answer the value holder for the background color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder backgroundColorHolder() {
		return backgroundColorHolder;
	}

	/**
	 * Answer the value holder for the border color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder borderColorHolder() {
		return borderColorHolder;
	}

	/**
	 * Answer the value holder for the border width.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder borderWidthHolder() {
		return borderWidthHolder;
	}

	/**
	 * Create my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunElementalNodeSettingsViewAwt(this);
		} else {
			return new JunElementalNodeSettingsViewSwing(this);
		}
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#defaultAttributes()
	 * @category defaults
	 */
	protected JunAttributeTable defaultAttributes() {
		return JunElementalNode.DefaultAttributes;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		String aString = $String("Node Settings");
		if (this.stuff() == null) {
			aString += " (==" + $String("Global") + "==)";
		} else {
			aString += " " + this.stuff().displayString();
		}
		return aString;
	}

	/**
	 * Action for the foreground color button.
	 * 
	 * @category actions
	 */
	protected void foregroundColorButtonPressed() {
		this._chooseColorFor(this.foregroundColorHolder());
	}

	/**
	 * Action for the background color button.
	 * 
	 * @category actions
	 */
	protected void backgroundColorButtonPressed() {
		this._chooseColorFor(this.backgroundColorHolder());
	}

	/**
	 * Action for the border color button.
	 * 
	 * @category actions
	 */
	protected void borderColorButtonPressed() {
		this._chooseColorFor(this.borderColorHolder());
	}

	/**
	 * Copy settings from current to stuff.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#settingsFromCurrentToStuff()
	 * @category private
	 */
	protected void settingsFromCurrentToStuff() {
		Map aMap = this.currentSettings();
		if (this.stuff() == null) {
			JunAttributeTable attributes = this.defaultAttributes();
			attributes.at_put_($("labelString"), aMap.get($("labelString")));
			if (aMap.get($("locationPoint")) != null) {
				attributes.at_put_($("locationPoint"), aMap.get($("locationPoint")));
			}
			attributes.at_put_($("foregroundColor"), aMap.get($("foregroundColor")));
			attributes.at_put_($("backgroundColor"), aMap.get($("backgroundColor")));
			attributes.at_put_($("borderColor"), aMap.get($("borderColor")));
			attributes.at_put_($("borderWidth"), aMap.get($("borderWidth")));
		} else if (this.stuff() instanceof JunElementalNode) {
			JunElementalNode aNode = (JunElementalNode) this.stuff();
			aNode.labelString_((String) aMap.get($("labelString")));
			if (aMap.get($("locationPoint")) != null) {
				aNode.locationPoint_((Point) aMap.get($("locationPoint")));
			}
			aNode.foregroundColor_((Color) aMap.get($("foregroundColor")));
			aNode.backgroundColor_((Color) aMap.get($("backgroundColor")));
			aNode.borderColor_((Color) aMap.get($("borderColor")));
			aNode.borderWidth_(((Number) aMap.get($("borderWidth"))).intValue());
		}
	}

	/**
	 * Copy settings from stuff to current.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#settingsFromStuffToCurrent()
	 * @category private
	 */
	protected void settingsFromStuffToCurrent() {
		Map aMap = this.stuffSettings();
		this.labelStringHolder().setValue_(aMap.get($("labelStringHolder")));
		this.xLocationStringHolder().setValue_(aMap.get($("xLocationStringHolder")));
		this.yLocationStringHolder().setValue_(aMap.get($("yLocationStringHolder")));
		this.foregroundColorHolder().setValue_(aMap.get($("foregroundColorHolder")));
		this.backgroundColorHolder().setValue_(aMap.get($("backgroundColorHolder")));
		this.borderColorHolder().setValue_(aMap.get($("borderColorHolder")));
		this.borderWidthHolder().setValue_(aMap.get($("borderWidthHolder")));

		settingsHolder.value_(this.stuffSettings());
		this.updateWidgets();
	}

	/**
	 * Reset the default attributes.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#resetDefaultAttributes()
	 * @category private
	 */
	protected void resetDefaultAttributes() {
		JunAttributeTable attributes = this.defaultAttributes();
		attributes.at_put_($("labelString"), JunElementalNode.DefaultLabelString());
		attributes.at_put_($("locationPoint"), JunElementalNode.DefaultLocationPoint());
		attributes.at_put_($("foregroundColor"), JunElementalNode.DefaultForegroundColor());
		attributes.at_put_($("backgroundColor"), JunElementalNode.DefaultBackgroundColor());
		attributes.at_put_($("borderColor"), JunElementalNode.DefaultBorderColor());
		attributes.at_put_($("borderWidth"), new Integer(JunElementalNode.DefaultBorderWidth()));
	}

}
