package jp.co.sra.jun.topology.graph;

import java.awt.Color;
import java.util.HashMap;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.tables.JunAttributeTable;

/**
 * JunElementalArcSettings class
 * 
 *  @author    nisinaka
 *  @created   2006/04/20 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun593 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalArcSettings.java,v 8.10 2008/02/20 06:33:02 nisinaka Exp $
 */
public class JunElementalArcSettings extends JunElementalStuffSettings {

	protected StValueHolder lineColorHolder;
	protected StValueHolder lineWidthHolder;
	protected StValueHolder firstBoxColorHolder;
	protected StValueHolder lastBoxColorHolder;
	protected StValueHolder firstPointAlignmentHolder;
	protected StValueHolder lastPointAlignmentHolder;

	/**
	 * Create a new instance of JunElementalArcSettings and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalArcSettings() {
		super();
	}

	/**
	 * Create a new instance of JunElementalArcSettings and initialize it.
	 *
	 * @param anElementalArc jp.co.sra.jun.topology.graph.JunElementalArc
	 * @category Instance creation
	 */
	public JunElementalArcSettings(JunElementalArc anElementalArc) {
		this.stuff_(anElementalArc);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		StBlockClosure updateSettingsBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				updateSettings();
				return null;
			}
		};

		lineColorHolder = new StValueHolder(JunElementalArc.DefaultLineColor());
		lineColorHolder.compute_(updateSettingsBlock);

		lineWidthHolder = new StValueHolder(JunElementalArc.DefaultLineWidth());
		lineWidthHolder.compute_(updateSettingsBlock);

		firstBoxColorHolder = new StValueHolder(JunElementalArc.DefaultFirstBoxColor());
		firstBoxColorHolder.compute_(updateSettingsBlock);

		lastBoxColorHolder = new StValueHolder(JunElementalArc.DefaultLastBoxColor());
		lastBoxColorHolder.compute_(updateSettingsBlock);

		firstPointAlignmentHolder = new StValueHolder(JunElementalArc.DefaultFirstPointAlignment());
		firstPointAlignmentHolder.compute_(updateSettingsBlock);

		lastPointAlignmentHolder = new StValueHolder(JunElementalArc.DefaultLastPointAlignment());
		lastPointAlignmentHolder.compute_(updateSettingsBlock);
	}

	/**
	 * Answer the current settings.
	 * 
	 * @return java.util.Map
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#currentSettings()
	 * @category accessing
	 */
	public Map currentSettings() {
		HashMap aMap = new HashMap(6);
		aMap.put($("lineColor"), this.lineColorHolder().value());
		aMap.put($("lineWidth"), this.lineWidthHolder().value());
		aMap.put($("firstBoxColor"), this.firstBoxColorHolder().value());
		aMap.put($("lastBoxColor"), this.lastBoxColorHolder().value());
		aMap.put($("firstPointAlignment"), this.firstPointAlignmentHolder().value());
		aMap.put($("lastPointAlignment"), this.lastPointAlignmentHolder().value());
		return aMap;
	}

	/**
	 * Answer the stuff settings.
	 * 
	 * @return java.util.Map
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#currentSettings()
	 * @category accessing
	 */
	public Map stuffSettings() {
		HashMap aMap = new HashMap(6);
		if (this.stuff() == null) {
			JunAttributeTable attributes = this.defaultAttributes();
			aMap.put($("lineColorHolder"), attributes.at_($("lineColor")));
			aMap.put($("lineWidthHolder"), attributes.at_($("lineWidth")));
			aMap.put($("firstBoxColorHolder"), attributes.at_($("firstBoxColor")));
			aMap.put($("lastBoxColorHolder"), attributes.at_($("lastBoxColor")));
			aMap.put($("firstPointAlignmentHolder"), attributes.at_($("firstPointAlignment")));
			aMap.put($("lastPointAlignmentHolder"), attributes.at_($("lastPointAlignment")));
		} else if (this.stuff() instanceof JunElementalArc) {
			JunElementalArc anArc = (JunElementalArc) this.stuff();
			aMap.put($("lineColorHolder"), anArc.lineColor());
			aMap.put($("lineWidthHolder"), new Integer(anArc.lineWidth()));
			aMap.put($("firstBoxColorHolder"), anArc.firstBoxColor());
			aMap.put($("lastBoxColorHolder"), anArc.lastBoxColor());
			aMap.put($("firstPointAlignmentHolder"), anArc.firstPointAlignment());
			aMap.put($("lastPointAlignmentHolder"), anArc.lastPointAlignment());
		}
		return aMap;
	}

	/**
	 * Answer the value holder for the line color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder lineColorHolder() {
		return lineColorHolder;
	}

	/**
	 * Answer the value holder for the line width.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder lineWidthHolder() {
		return lineWidthHolder;
	}

	/**
	 * Answer the value holder for the first box color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder firstBoxColorHolder() {
		return firstBoxColorHolder;
	}

	/**
	 * Answer the value holder for the last box color.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder lastBoxColorHolder() {
		return lastBoxColorHolder;
	}

	/**
	 * Answer the value holder for the first point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder firstPointAlignmentHolder() {
		return firstPointAlignmentHolder;
	}

	/**
	 * Answer the value holder for the last point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder lastPointAlignmentHolder() {
		return lastPointAlignmentHolder;
	}

	/**
	 * Create my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunElementalArcSettingsViewAwt(this);
		} else {
			return new JunElementalArcSettingsViewSwing(this);
		}
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#defaultAttributes()
	 * @category defaults
	 */
	protected JunAttributeTable defaultAttributes() {
		return JunElementalArc.DefaultAttributes;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		String aString = $String("Arc Settings");
		if (this.stuff() == null) {
			aString += " (==" + $String("Global") + "==)";
		} else {
			aString += " " + this.stuff().displayString();
		}
		return aString;
	}

	/**
	 * Action for the line color button.
	 * 
	 * @category actions
	 */
	protected void lineColorButtonPressed() {
		this._chooseColorFor(this.lineColorHolder());
	}

	/**
	 * Action for the first box color button.
	 * 
	 * @category actions
	 */
	protected void firstBoxColorButtonPressed() {
		this._chooseColorFor(this.firstBoxColorHolder());
	}

	/**
	 * Action for the last box color button.
	 * 
	 * @category actions
	 */
	protected void lastBoxColorButtonPressed() {
		this._chooseColorFor(this.lastBoxColorHolder());
	}

	/**
	 * Copy settings from current to stuff.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#settingsFromCurrentToStuff()
	 * @category private
	 */
	protected void settingsFromCurrentToStuff() {
		Map aMap = this.currentSettings();
		if (this.stuff() == null) {
			JunAttributeTable attributes = this.defaultAttributes();
			attributes.at_put_($("lineColor"), aMap.get($("lineColor")));
			attributes.at_put_($("lineWidth"), aMap.get($("lineWidth")));
			attributes.at_put_($("firstBoxColor"), aMap.get($("firstBoxColor")));
			attributes.at_put_($("lastBoxColor"), aMap.get($("lastBoxColor")));
			attributes.at_put_($("firstPointAlignment"), aMap.get($("firstPointAlignment")));
			attributes.at_put_($("lastPointAlignment"), aMap.get($("lastPointAlignment")));
		} else if (this.stuff() instanceof JunElementalArc) {
			JunElementalArc anArc = (JunElementalArc) this.stuff();
			anArc.lineColor_((Color) aMap.get($("lineColor")));
			anArc.lineWidth_(((Number) aMap.get($("lineWidth"))).intValue());
			anArc.firstBoxColor_((Color) aMap.get($("firstBoxColor")));
			anArc.lastBoxColor_((Color) aMap.get($("lastBoxColor")));
			anArc.firstPointAlignment_((StSymbol) aMap.get($("firstPointAlignment")));
			anArc.lastPointAlignment_((StSymbol) aMap.get($("lastPointAlignment")));
		}
	}

	/**
	 * Copy settings from stuff to current.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#settingsFromStuffToCurrent()
	 * @category private
	 */
	protected void settingsFromStuffToCurrent() {
		Map aMap = this.stuffSettings();
		this.lineColorHolder().setValue_(aMap.get($("lineColorHolder")));
		this.lineWidthHolder().setValue_(aMap.get($("lineWidthHolder")));
		this.firstBoxColorHolder().setValue_(aMap.get($("firstBoxColorHolder")));
		this.lastBoxColorHolder().setValue_(aMap.get($("lastBoxColorHolder")));
		this.firstPointAlignmentHolder().setValue_(aMap.get($("firstPointAlignmentHolder")));
		this.lastPointAlignmentHolder().setValue_(aMap.get($("lastPointAlignmentHolder")));

		settingsHolder.value_(this.stuffSettings());
		this.updateWidgets();
	}

	/**
	 * Reset the default attributes.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuffSettings#resetDefaultAttributes()
	 * @category private
	 */
	protected void resetDefaultAttributes() {
		JunAttributeTable attributes = this.defaultAttributes();
		attributes.at_put_($("lineColor"), JunElementalArc.DefaultLineColor());
		attributes.at_put_($("lineWidth"), new Integer(JunElementalArc.DefaultLineWidth()));
		attributes.at_put_($("firstBoxColor"), JunElementalArc.DefaultFirstBoxColor());
		attributes.at_put_($("lastBoxColor"), JunElementalArc.DefaultLastBoxColor());
		attributes.at_put_($("firstPointAlignment"), JunElementalArc.DefaultFirstPointAlignment());
		attributes.at_put_($("lastPointAlignment"), JunElementalArc.DefaultLastPointAlignment());
	}

}
