package jp.co.sra.jun.topology.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.calendar.JunCalendarModel;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;

/**
 * JunElementalArc class
 * 
 *  @author    nisinaka
 *  @created   2006/04/06 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun598 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalArc.java,v 8.10 2008/02/20 06:33:02 nisinaka Exp $
 */
public class JunElementalArc extends JunElementalStuff {

	protected static JunAttributeTable DefaultAttributes = new JunAttributeTable();
	protected static StSymbol[] AttributeSymbolsToReset = new StSymbol[] { $("lineColor"), $("lineWidth"), $("firstBoxColor"), $("lastBoxColor"), $("firstPointAlignment"), $("lastPointAlignment") };

	protected JunElementalNode firstNode;
	protected JunElementalNode lastNode;

	/**
	 * Answer the default line width.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultLineWidth() {
		return 1;
	}

	/**
	 * Answer the default line color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultLineColor() {
		return StColorValue.Gray;
	}

	/**
	 * Answer the default first point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	protected static StSymbol DefaultFirstPointAlignment() {
		return $("center");
	}

	/**
	 * Answer the default first box color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultFirstBoxColor() {
		return StColorValue.VeryDarkGray;
	}

	/**
	 * Answer the default last box color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultLastBoxColor() {
		return StColorValue.VeryDarkGray;
	}

	/**
	 * Answer the default last point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	protected static StSymbol DefaultLastPointAlignment() {
		return $("center");
	}

	/**
	 * Create a new instance of JunElementalArc and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalArc() {
		this(null, null, null);
	}

	/**
	 * Create a new instance of JunElementalArc and initialize it.
	 *
	 * @param firstNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param lastNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category Instance creation
	 */
	public JunElementalArc(JunElementalNode firstNode, JunElementalNode lastNode) {
		this(firstNode, lastNode, null);
	}

	/**
	 * Create a new instance of JunElementalArc and initialize it.
	 *
	 * @param firstNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param lastNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param arcAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category Instance creation
	 */
	public JunElementalArc(JunElementalNode firstNode, JunElementalNode lastNode, JunAttributeTable arcAttributes) {
		this.initialize();
		this.from_to_attributes_(firstNode, lastNode, arcAttributes);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		firstNode = null;
		lastNode = null;
		myAttributes = new JunAttributeTable();
		String aString = JunCalendarModel.StringFromDateAndTime();
		myAttributes.at_put_($("creationDate"), aString);
		myAttributes.at_put_($("modificationDate"), aString);
	}

	/**
	 * Answer my first node.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category accessing
	 */
	public JunElementalNode firstNode() {
		return firstNode;
	}

	/**
	 * Answer the point of the first node.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point firstPoint() {
		if (this.firstNode() == null) {
			return new Point(0, 0);
		}
		StRectangle aBox = this.firstNode().boundingBox();
		try {
			return (Point) aBox.perform_(this.firstPointAlignment().toString());
		} catch (Exception e) {
			return new Point(0, 0);
		}
	}

	/**
	 * Answer the area of the first node.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle firstBox() {
		StRectangle aBox = StRectangle.Origin_extent_(this.firstPoint(), new Point(0, 0));
		return aBox.expandedBy_(this.lineWidth());
	}

	/**
	 * Answer my last node.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category accessing
	 */
	public JunElementalNode lastNode() {
		return lastNode;
	}

	/**
	 * Answer the point of the last node.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point lastPoint() {
		if (this.lastNode() == null) {
			return new Point(0, 0);
		}
		StRectangle aBox = this.lastNode().boundingBox();
		try {
			return (Point) aBox.perform_(this.lastPointAlignment().toString());
		} catch (Exception e) {
			return new Point(0, 0);
		}
	}

	/**
	 * Answer the area of the last node.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle lastBox() {
		StRectangle aBox = StRectangle.Origin_extent_(this.lastPoint(), new Point(0, 0));
		return aBox.expandedBy_(this.lineWidth());
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle boundingBox() {
		return this.firstBox().merge_(this.lastBox());
	}

	/**
	 * Answer my attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category accessing
	 */
	protected JunAttributeTable arcAttributes() {
		return myAttributes;
	}

	/**
	 * Add the attribute.
	 * 
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#addAttribute_(java.lang.Object, java.lang.Object)
	 * @category attributes
	 */
	public void addAttribute_(Object key, Object value) {
		myAttributes.at_put_(key, value);
		this.updateModificationDate();
	}

	/**
	 * Answer my line width.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int lineWidth() {
		Number aNumber = (Number) myAttributes.at_($("lineWidth"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultLineWidth());
			myAttributes.at_put_($("lineWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my line width.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void lineWidth_(int anInteger) {
		myAttributes.at_put_($("lineWidth"), new Integer(anInteger));
		this.updateModificationDate();
	}

	/**
	 * Answer my line color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color lineColor() {
		Color aColor = (Color) myAttributes.at_($("lineColor"));
		if (aColor == null) {
			aColor = this.defaultLineColor();
			myAttributes.at_put_($("lineColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my line color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void lineColor_(Color aColor) {
		myAttributes.at_put_($("lineColor"), aColor);
		this.updateModificationDate();
	}

	/**
	 * Answer my first point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category attributes
	 */
	public StSymbol firstPointAlignment() {
		StSymbol aSymbol = (StSymbol) myAttributes.at_($("firstPointAlignment"));
		if (aSymbol == null) {
			aSymbol = this.defaultFirstPointAlignment();
			myAttributes.at_put_($("firstPointAlignment"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Set my first point alignment.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category attributes
	 */
	public void firstPointAlignment_(StSymbol aSymbol) {
		myAttributes.at_put_($("firstPointAlignment"), aSymbol);
		this.updateModificationDate();
	}

	/**
	 * Answer my first box color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color firstBoxColor() {
		Color aColor = (Color) myAttributes.at_($("firstBoxColor"));
		if (aColor == null) {
			aColor = this.defaultFirstBoxColor();
			myAttributes.at_put_($("firstBoxColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my first box color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void firstBoxColor_(Color aColor) {
		myAttributes.at_put_($("firstBoxColor"), aColor);
		this.updateModificationDate();
	}

	/**
	 * Answer my last point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category attributes
	 */
	public StSymbol lastPointAlignment() {
		StSymbol aSymbol = (StSymbol) myAttributes.at_($("lastPointAlignment"));
		if (aSymbol == null) {
			aSymbol = this.defaultLastPointAlignment();
			myAttributes.at_put_($("lastPointAlignment"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Set my last point alignment.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category attributes
	 */
	public void lastPointAlignment_(StSymbol aSymbol) {
		myAttributes.at_put_($("lastPointAlignment"), aSymbol);
		this.updateModificationDate();
	}

	/**
	 * Answer my last box color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color lastBoxColor() {
		Color aColor = (Color) myAttributes.at_($("lastBoxColor"));
		if (aColor == null) {
			aColor = this.defaultLastBoxColor();
			myAttributes.at_put_($("lastBoxColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my last box color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void lastBoxColor_(Color aColor) {
		myAttributes.at_put_($("lastBoxColor"), aColor);
		this.updateModificationDate();
	}

	/**
	 * Answer the attribute symbols to reset.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol[]
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#attributeSymbolsToReset()
	 * @category attributes
	 */
	protected StSymbol[] attributeSymbolsToReset() {
		return AttributeSymbolsToReset;
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#defaultAttributes()
	 * @category defaults
	 */
	protected JunAttributeTable defaultAttributes() {
		return DefaultAttributes;
	}

	/**
	 * Answer the default line width.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultLineWidth() {
		Number aNumber = (Number) this.defaultAttributes().at_($("lineWidth"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultLineWidth());
			this.defaultAttributes().at_put_($("lineWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Answer the default line color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLineColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("lineColor"));
		if (aColor == null) {
			aColor = DefaultLineColor();
			this.defaultAttributes().at_put_($("lineColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default first point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultFirstPointAlignment() {
		StSymbol aSymbol = (StSymbol) this.defaultAttributes().at_($("firstPointAlignment"));
		if (aSymbol == null) {
			aSymbol = DefaultFirstPointAlignment();
			this.defaultAttributes().at_put_($("firstPointAlignment"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Answer the default first box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultFirstBoxColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("firstBoxColor"));
		if (aColor == null) {
			aColor = DefaultFirstBoxColor();
			this.defaultAttributes().at_put_($("firstBoxColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default last point alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultLastPointAlignment() {
		StSymbol aSymbol = (StSymbol) this.defaultAttributes().at_($("lastPointAlignment"));
		if (aSymbol == null) {
			aSymbol = DefaultLastPointAlignment();
			this.defaultAttributes().at_put_($("lastPointAlignment"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Answer the default last box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLastBoxColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("lastBoxColor"));
		if (aColor == null) {
			aColor = DefaultLastBoxColor();
			this.defaultAttributes().at_put_($("lastBoxColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("arc(");
		if (firstNode != null) {
			firstNode.printOn_(aWriter);
		}
		aWriter.write("=>");
		if (lastNode != null) {
			lastNode.printOn_(aWriter);
		}
		aWriter.write(')');
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		if (this.firstNode() == null || this.lastNode() == null) {
			return;
		}

		Graphics2D aGraphics2D = (Graphics2D) aGraphics;
		StRectangle displayBox = this.boundingBox().translatedBy_(aPoint.x, aPoint.y);
		Rectangle clippingBox = aGraphics.getClipBounds();
		if (clippingBox == null || displayBox.toRectangle().intersects(clippingBox)) {
			Point firstPoint = this.firstPoint();
			firstPoint.translate(aPoint.x, aPoint.y);
			Point lastPoint = this.lastPoint();
			lastPoint.translate(aPoint.x, aPoint.y);

			int lineWidth = this.lineWidth();
			if (lineWidth > 0) {
				aGraphics2D.setStroke(new BasicStroke(lineWidth));
				aGraphics2D.setColor(this.lineColor());
				aGraphics2D.drawLine(firstPoint.x, firstPoint.y, lastPoint.x, lastPoint.y);
			}

			StRectangle firstBox = this.firstBox().translatedBy_(aPoint.x, aPoint.y);
			if (firstBox.area() > 0) {
				aGraphics2D.setColor(this.firstBoxColor());
				aGraphics2D.fillRect(firstBox.x(), firstBox.y(), firstBox.width(), firstBox.height());
			}

			StRectangle lastBox = this.lastBox().translatedBy_(aPoint.x, aPoint.y);
			if (lastBox.area() > 0) {
				aGraphics2D.setColor(this.lastBoxColor());
				aGraphics2D.fillRect(lastBox.x(), lastBox.y(), lastBox.width(), lastBox.height());
			}
		}
	}

	/**
	 * Display the receiver on the graphics at the specified point with the specified scale factor.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param scaleFactor jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category displaying
	 */
	public void displayOn_at_scaledBy_(Graphics aGraphics, Point aPoint, Jun2dPoint scaleFactor) {
		if (this.firstNode() == null || this.lastNode() == null) {
			return;
		}

		Graphics2D aGraphics2D = (Graphics2D) aGraphics;
		StRectangle displayBox = this.boundingBox().translatedBy_(aPoint.x, aPoint.y);
		displayBox = displayBox.scaledBy_(scaleFactor.x(), scaleFactor.y());
		displayBox = displayBox.expandedBy_(this.lineWidth());
		Rectangle clippingBox = aGraphics.getClipBounds();
		if (clippingBox == null || displayBox.toRectangle().intersects(clippingBox)) {
			Point firstPoint = this.firstPoint();
			int firstPointX = (int) Math.round((firstPoint.x + aPoint.x) * scaleFactor.x());
			int firstPointY = (int) Math.round((firstPoint.y + aPoint.y) * scaleFactor.y());
			Point lastPoint = this.lastPoint();
			int lastPointX = (int) Math.round((lastPoint.x + aPoint.x) * scaleFactor.x());
			int lastPointY = (int) Math.round((lastPoint.y + aPoint.y) * scaleFactor.y());

			int lineWidth = this.lineWidth();
			if (lineWidth > 0) {
				aGraphics2D.setStroke(new BasicStroke(lineWidth));
				aGraphics2D.setColor(this.lineColor());
				aGraphics2D.drawLine(firstPointX, firstPointY, lastPointX, lastPointY);
			}

			StRectangle firstBox = this.firstBox().translatedBy_(aPoint.x, aPoint.y).scaledBy_(scaleFactor.x(), scaleFactor.y());
			if (firstBox.area() > 0) {
				aGraphics2D.setColor(this.firstBoxColor());
				aGraphics2D.fillRect(firstBox.x(), firstBox.y(), firstBox.width(), firstBox.height());
			}

			StRectangle lastBox = this.lastBox().translatedBy_(aPoint.x, aPoint.y).scaledBy_(scaleFactor.x(), scaleFactor.y());
			if (lastBox.area() > 0) {
				aGraphics2D.setColor(this.lastBoxColor());
				aGraphics2D.fillRect(lastBox.x(), lastBox.y(), lastBox.width(), lastBox.height());
			}
		}
	}

	/**
	 * Set the attributes.
	 * 
	 * @param from jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param to jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param attributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category private
	 */
	private void from_to_attributes_(JunElementalNode from, JunElementalNode to, JunAttributeTable attributes) {
		firstNode = from;
		lastNode = to;

		if (attributes != null) {
			Set entities = myAttributes.entries();
			Iterator i = entities.iterator();
			while (i.hasNext()) {
				Map.Entry entry = (Map.Entry) i.next();
				if (attributes.at_(entry.getKey()) == null) {
					attributes.at_put_(entry.getKey(), entry.getValue());
				}
			}

			this.setMyAttributes_(attributes);
		}
	}

}
