package jp.co.sra.jun.opengl.texture;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.font.JunFontModel;
import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunOpenGLTexture class
 * 
 *  @author    nisinaka
 *  @created   1998/10/13 (by nisinaka)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2000/03/17 (by nisinaka)
 *  @updated   2004/06/17 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun660 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLTexture.java,v 8.14 2008/02/20 06:32:50 nisinaka Exp $
 */
public class JunOpenGLTexture extends JunAbstractObject {

	protected StImage textureImage;
	protected JunOpenGL3dObject textureObject;
	protected Jun2dPoint[] textureCoordinates;
	protected boolean clampBoolean = true;
	protected boolean nearestBoolean = true;
	protected boolean modulateBoolean = true;
	protected boolean mipmapBoolean = false;
	protected double[] sParameters;
	protected double[] tParameters;
	protected byte[] cachedPixels;

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunOpenGLTexture() {
		super();
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> with the image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category Instance creation
	 */
	public JunOpenGLTexture(StImage anImage) {
		this();
		this.image_(anImage);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> with the image.
	 * 
	 * @param anImage java.awt.Image
	 * @category Instance creation
	 */
	public JunOpenGLTexture(Image anImage) {
		this();
		this.image_(anImage);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGLTexture(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> with the image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	public static JunOpenGLTexture Image_(StImage anImage) {
		JunOpenGLTexture texture = new JunOpenGLTexture();
		texture.image_(anImage);
		return texture;
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> with the image.
	 * 
	 * @param anImage java.awt.Image
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	public static JunOpenGLTexture Image_(Image anImage) {
		JunOpenGLTexture texture = new JunOpenGLTexture();
		texture.image_(anImage);
		return texture;
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @deprecated since Jun454, use the constructor.
	 * @category Lisp support
	 */
	public static JunOpenGLTexture FromLispList_(JunLispList aList) {
		return new JunOpenGLTexture(aList);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the string.
	 * 
	 * @param aString java.lang.String
	 * @param foregroundColor java.awt.Color 
	 * @param backgroundColor java.awt.Color 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category Utilities
	 */
	public static JunOpenGLTexture TextureForString_foreColor_backColor_(String aString, Color foregroundColor, Color backgroundColor) {
		return Texture1ForString_foreColor_backColor_(aString, foregroundColor, backgroundColor);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the string.
	 * 
	 * @param aString java.lang.String
	 * @param foregroundColor java.awt.Color 
	 * @param backgroundColor java.awt.Color
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category Utilities
	 */
	public static JunOpenGLTexture TextureForString_foreColor_backColor_alignmentSymbol_(String aString, Color foregroundColor, Color backgroundColor, StSymbol alignmentSymbol) {
		return Texture1ForString_foreColor_backColor_alignmentSymbol_(aString, foregroundColor, backgroundColor, alignmentSymbol);
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the string.
	 * 
	 * @param aString java.lang.String
	 * @param foregroundColor java.awt.Color 
	 * @param backgroundColor java.awt.Color 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category Utilities
	 */
	protected static JunOpenGLTexture Texture1ForString_foreColor_backColor_(String aString, Color foregroundColor, Color backgroundColor) {
		return Texture1ForString_foreColor_backColor_alignmentSymbol_(aString, foregroundColor, backgroundColor, $("center"));
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the string.
	 * 
	 * @param aString java.lang.String
	 * @param foregroundColor java.awt.Color 
	 * @param backgroundColor java.awt.Color 
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category Utilities
	 */
	protected static JunOpenGLTexture Texture1ForString_foreColor_backColor_alignmentSymbol_(String aString, Color foregroundColor, Color backgroundColor, StSymbol alignmentSymbol) {
		StComposedText text = new StComposedText(aString, JunFontModel.TextStyle(100));
		Dimension extent = text.extent();
		extent = NthPowerExtent_(new Point(extent.width + 2, extent.height + 2));
		Image image = new StImage(extent.width, extent.height).image();

		Graphics graphics = null;
		try {
			graphics = image.getGraphics();
			graphics.setColor(backgroundColor);
			graphics.fillRect(0, 0, extent.width, extent.height);
			graphics.setColor(foregroundColor);

			StRectangle textBounds = new StRectangle(text.bounds());
			StRectangle imageBounds = new StRectangle(extent);
			Point point = textBounds.align_with_(textBounds._pointAt(alignmentSymbol), imageBounds._pointAt(alignmentSymbol)).origin();
			text.displayOn_at_(graphics, point);
		} finally {
			if (graphics != null) {
				graphics.dispose();
				graphics = null;
			}
		}

		JunOpenGLTexture aTexture = new JunOpenGLTexture(image);
		Jun2dPoint[] coordinates = new Jun2dPoint[24];
		int index = 0;
		Jun2dPoint[] points = new Jun2dPoint[] { new Jun2dPoint(0.0, 0.0), new Jun2dPoint(1.0, 0.0), new Jun2dPoint(1.0, 1.0), new Jun2dPoint(0.0, 1.0) };
		for (int i = 0; i < 4; i++) {
			coordinates[index++] = points[i];
		}
		for (int i = 3; i >= 0; i--) {
			coordinates[index++] = points[i];
		}
		for (int i = 0; i < 16; i++) {
			coordinates[index++] = new Jun2dPoint(0.0, 0.0);
		}
		aTexture.coordinates_(coordinates);
		aTexture.pointerOfPixels();
		return aTexture;
	}

	/**
	 * Create a new instance of <code>JunOpenGLTexture</code> from the string.
	 * 
	 * @param aString java.lang.String
	 * @param foregroundColor java.awt.Color 
	 * @param backgroundColor java.awt.Color 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category Utilities
	 */
	protected static JunOpenGLTexture Texture2ForString_foreColor_backColor_(String aString, Color foregroundColor, Color backgroundColor) {
		StComposedText text = new StComposedText(aString, JunFontModel.TextStyle(100));
		Dimension extent = text.extent();
		Image image = new StImage(extent.width + 2, extent.height + 2).image();

		Graphics graphics = null;
		try {
			graphics = image.getGraphics();
			graphics.setColor(backgroundColor);
			graphics.fillRect(0, 0, extent.width, extent.height);
			graphics.setColor(foregroundColor);
			text.displayOn_at_(graphics, new Point(1, 1));
		} finally {
			if (graphics != null) {
				graphics.dispose();
				graphics = null;
			}
		}

		JunOpenGLTexture aTexture = Image_(image);
		Jun2dPoint[] coordinates = new Jun2dPoint[24];
		int index = 0;
		Jun2dPoint[] points = new Jun2dPoint[] { new Jun2dPoint(0.0, 0.0), new Jun2dPoint(1.0, 0.0), new Jun2dPoint(1.0, 1.0), new Jun2dPoint(0.0, 1.0) };
		for (int i = 0; i < 4; i++) {
			coordinates[index++] = points[i];
		}
		for (int i = 3; i >= 0; i--) {
			coordinates[index++] = points[i];
		}
		for (int i = 0; i < 16; i++) {
			coordinates[index++] = new Jun2dPoint(0.0, 0.0);
		}
		aTexture.coordinates_(coordinates);
		aTexture.pointerOfPixels();
		return aTexture;
	}

	/**
	 * Calculate the Nth power extent.
	 * 
	 * @param anExtent java.awt.Point
	 * @return java.awt.Dimension
	 * @category Utilities
	 */
	protected static Dimension NthPowerExtent_(Point anExtent) {
		int width = anExtent.x;
		int height = anExtent.y;
		int size = 11;
		int[] collection = new int[size];
		for (int n = 0; n < size; n++) {
			collection[n] = (int) Math.pow(2, n);
		}
		for (int index = 1; index < size; index++) {
			int start = collection[index - 1];
			int end = collection[index];
			if ((width > start) & (width <= end)) {
				width = end;
			}
			if ((height > start) & (height <= end)) {
				height = end;
			}
		}
		return new Dimension(width, height);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		textureImage = null;
		textureObject = null;
		textureCoordinates = null;
		clampBoolean = true;
		nearestBoolean = true;
		modulateBoolean = true;
		mipmapBoolean = false;
		sParameters = null;
		tParameters = null;
		cachedPixels = null;
	}

	/**
	 * Answer my current texture image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage image() {
		if (textureImage == null) {
			StImage newImage = new StImage(1, 1);
			newImage.valueAtPoint_put_(new Point(0, 0), Color.black);
			textureImage = newImage;
		}
		return textureImage;
	}

	/**
	 * Set my new texture image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public void image_(StImage anImage) {
		if (anImage.width() == 0 || anImage.height() == 0) {
			return;
		}

		textureImage = this.convertToTextureImage_(anImage);
		this.flushPointerOfPixels();
	}

	/**
	 * Set my new texture image.
	 * 
	 * @param anImage java.awt.Image
	 * @category accessing
	 */
	public void image_(Image anImage) {
		this.image_(new StImage(anImage));
	}

	/**
	 * Answer my current 3D object on which the texture is mapped.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject object() {
		return textureObject;
	}

	/**
	 * Set my new 3D object on which the texture is mapped.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public void object_(JunOpenGL3dObject a3dObject) {
		textureObject = a3dObject;
	}

	/**
	 * Answer my current texture coordinates.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] coordinates() {
		return textureCoordinates;
	}

	/**
	 * Set my new texture coordinates.
	 * 
	 * @param collectionOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public void coordinates_(Jun2dPoint[] collectionOfPoint) {
		if (collectionOfPoint == null) {
			textureCoordinates = null;
		} else {
			Jun2dPoint[] newCoordinates = new Jun2dPoint[collectionOfPoint.length];
			System.arraycopy(collectionOfPoint, 0, newCoordinates, 0, collectionOfPoint.length);
			textureCoordinates = newCoordinates;
		}
	}

	/**
	 * Answer the width of the texture image.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return this.image().width();
	}

	/**
	 * Answer the height of the texture image.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return this.image().height();
	}

	/**
	 * Answer true if the texture coordinates is specified, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasCoordinates() {
		return (textureCoordinates != null && textureCoordinates.length > 0);
	}

	/**
	 * Answer true if the parameter S is specified, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasParametersS() {
		return (sParameters != null && sParameters.length > 0);
	}

	/**
	 * Answer true if the parameter T is specified, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasParametersT() {
		return (tParameters != null && tParameters.length > 0);
	}

	/**
	 * Answer true if the clamp attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean clamp() {
		return clampBoolean;
	}

	/**
	 * Set the clamp attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void clamp_(boolean aBoolean) {
		clampBoolean = aBoolean;
	}

	/**
	 * Answer true if the linear attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean linear() {
		return !this.nearest();
	}

	/**
	 * Set the linear attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void linear_(boolean aBoolean) {
		this.nearest_(!aBoolean);
	}

	/**
	 * Answer true if the mipmap attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean mipmap() {
		return mipmapBoolean;
	}

	/**
	 * Set the mipmap attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void mipmap_(boolean aBoolean) {
		mipmapBoolean = aBoolean;
	}

	/**
	 * Answer true if the modulate attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean modulate() {
		return modulateBoolean;
	}

	/**
	 * Set the modulate attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void modulate_(boolean aBoolean) {
		modulateBoolean = aBoolean;
	}

	/**
	 * Answer true if the nearest attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean nearest() {
		return nearestBoolean;
	}

	/**
	 * Set the nearest attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void nearest_(boolean aBoolean) {
		nearestBoolean = aBoolean;
	}

	/**
	 * Answer true if the repeat attribute is set, otherwise false.
	 * 
	 * @return boolean
	 * @category setting
	 */
	public boolean repeat() {
		return !this.clamp();
	}

	/**
	 * Set the repeat attribute.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void repeat_(boolean aBoolean) {
		this.clamp_(!aBoolean);
	}

	/**
	 * Answer my current parameters for S-axis.
	 * 
	 * @return double[]
	 * @category parameters
	 */
	public double[] parametersForS() {
		if (sParameters == null) {
			if (this.object() == null) {
				this.defaultParameters();
			} else {
				this.xPositiveParameters();
			}
		}
		return sParameters;
	}

	/**
	 * Set my new parameters for S-axis.
	 * 
	 * @param newParameters double[]
	 * @category parameters
	 */
	public void parametersForS_(double[] newParameters) {
		sParameters = newParameters;
	}

	/**
	 * Answer my current parameters for T-axis.
	 * 
	 * @return double[]
	 * @category parameters
	 */
	public double[] parametersForT() {
		if (tParameters == null) {
			if (this.object() == null) {
				this.defaultParameters();
			} else {
				this.xPositiveParameters();
			}
		}
		return tParameters;
	}

	/**
	 * Set my new parameters for T-axis.
	 * 
	 * @param newParameters double[]
	 * @category parameters
	 */
	public void parametersForT_(double[] newParameters) {
		tParameters = newParameters;
	}

	/**
	 * Set my parameters for X-positive direction.
	 * 
	 * @category parameters
	 */
	public void xPositiveParameters() {
		try {
			this.parametersForS_(new double[] { 0.0, 1.0 / this.object().boundingBox().height(), 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 0.0, 1.0 / this.object().boundingBox().depth(), 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my parameters for X-negative direction.
	 * 
	 * @category parameters
	 */
	public void xNegativeParameters() {
		try {
			this.parametersForS_(new double[] { 0.0, -1.0 / this.object().boundingBox().height(), 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 0.0, 1.0 / this.object().boundingBox().depth(), 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my parameters for Y-positive direction.
	 * 
	 * @category parameters
	 */
	public void yPositiveParameters() {
		try {
			this.parametersForS_(new double[] { -1.0 / this.object().boundingBox().width(), 0.0, 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 0.0, 1.0 / this.object().boundingBox().depth(), 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my parameters for Y-negative direction.
	 * 
	 * @category parameters
	 */
	public void yNegativeParameters() {
		try {
			this.parametersForS_(new double[] { 1.0 / this.object().boundingBox().width(), 0.0, 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 0.0, 1.0 / this.object().boundingBox().depth(), 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my parameters for Z-positive direction.
	 * 
	 * @category parameters
	 */
	public void zPositiveParameters() {
		try {
			this.parametersForS_(new double[] { 1.0 / this.object().boundingBox().width(), 0.0, 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 1.0 / this.object().boundingBox().height(), 0.0, 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my parameters for Z-negative direction.
	 * 
	 * @category parameters
	 */
	public void zNegativeParameters() {
		try {
			this.parametersForS_(new double[] { -1.0 / this.object().boundingBox().width(), 0.0, 0.0, 0.5 });
			this.parametersForT_(new double[] { 0.0, 1.0 / this.object().boundingBox().height(), 0.0, 0.5 });
		} catch (ArithmeticException e) {
			this.defaultParameters();
		}
	}

	/**
	 * Set my default parameters.
	 * 
	 * @category parameters
	 */
	protected void defaultParameters() {
		this.parametersForS_(new double[] { 0.0, 1.0, 0.0, 0.0 });
		this.parametersForT_(new double[] { 0.0, 0.0, 1.0, 0.0 });
	}

	/**
	 * Answer the pixels of the texture image.
	 * 
	 * @return byte[]
	 * @category externals
	 */
	public byte[] pointerOfPixels() {
		if (cachedPixels == null) {
			int imageWidth = this.image().width();
			int imageHeight = this.image().height();
			byte[] byteArray = new byte[imageWidth * imageHeight * 3];
			int index = 0;
			for (int y = imageHeight - 1; y > -1; y = y - 1) {
				for (int x = 0; x < imageWidth; x++) {
					Color color = this.image().valueAtPoint_(new Point(x, y));
					int red = Math.min(255, Math.max(0, color.getRed()));
					int green = Math.min(255, Math.max(0, color.getGreen()));
					int blue = Math.min(255, Math.max(0, color.getBlue()));
					byteArray[index++] = (byte) red;
					byteArray[index++] = (byte) green;
					byteArray[index++] = (byte) blue;
				}
			}
			cachedPixels = byteArray;
		}
		return cachedPixels;
	}

	/**
	 * Answer the parameters for S-axis as an array of float.
	 * 
	 * @return float[]
	 * @category externals
	 */
	public float[] pointerOfParametersForS() {
		double[] parameters = this.parametersForS();
		int size = parameters.length;
		float[] floatArray = new float[size];
		for (int i = 0; i < parameters.length; i++) {
			floatArray[i] = (float) parameters[i];
		}
		return floatArray;
	}

	/**
	 * Answer the parameters for T-axis as an array of float.
	 * 
	 * @return float[]
	 * @category externals
	 */
	public float[] pointerOfParametersForT() {
		double[] parameters = this.parametersForT();
		int size = parameters.length;
		float[] floatArray = new float[size];
		for (int i = 0; i < parameters.length; i++) {
			floatArray[i] = (float) parameters[i];
		}
		return floatArray;
	}

	/**
	 * Flush the cached pixels of the texture image.
	 * 
	 * @category externals
	 */
	protected void flushPointerOfPixels() {
		cachedPixels = null;
	}

	/**
	 * Enable the texture on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category texture
	 */
	public void enableTextureOn_(JunOpenGLRenderingContext aRenderingContext) {
		aRenderingContext.texture_(this);
		aRenderingContext.enableTexture2d();
	}

	/**
	 * Disable the texture on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category texture
	 */
	public void disableTextureOn_(JunOpenGLRenderingContext aRenderingContext) {
		aRenderingContext.texture_(null);
		aRenderingContext.disableTexture2d();
	}

	/**
	 * Convert the image to a texture image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return StImage jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	protected StImage convertToTextureImage_(StImage anImage) {
		StImage image = this.convertToNthPowerExtentImage_(anImage);
		image = this.convertToPaletteImage_(image);
		return image;
	}

	/**
	 * Convert the image to an Nth power extent image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return StImage jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	protected StImage convertToNthPowerExtentImage_(StImage anImage) {
		StImage image = anImage;
		if (image.width() <= 0 || image.height() <= 0) {
			return image;
		}

		Dimension extent = NthPowerExtent_(anImage.extent());
		image = JunImageAdjuster.Adjust_extent_(image, extent);
		return image;
	}

	/**
	 * This is only stubbing method. an StImage always use standard RGB
	 * palette. So, this palette converting method is not needed.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return StImage jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	private StImage convertToPaletteImage_(StImage anImage) {
		return anImage;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return this._className();
	}

	/**
	 * Create an instance of JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons lispCons() {
		return (JunLispCons) JunLispCons.Cell();
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		list.add_(this.imageToLispList());
		if (this.hasCoordinates()) {
			list.add_(this.coordinatesToLispList());
		}
		list.add_(this.clampToLispList());
		list.add_(this.nearestToLispList());
		list.add_(this.modulateToLispList());
		list.add_(this.mipmapToLispList());
		if (this.hasParametersS()) {
			list.add_(this.sParametersToLispList());
		}
		if (this.hasParametersT()) {
			list.add_(this.tParametersToLispList());
		}

		return list;
	}

	/**
	 * Convert the receiver's texture image as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList imageToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("image"));
		list.tail_(this.image());
		return list;
	}

	/**
	 * Convert the receiver's texture coordinates as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList coordinatesToLispList() {
		Jun2dPoint[] coordinates = this.coordinates();
		Object[] collection = new Object[coordinates.length + 1];
		collection[0] = ($("coordinates"));
		for (int index = 0; index < coordinates.length; index++) {
			collection[index + 1] = coordinates[index];
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's clamp attribute as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList clampToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("clamp"));
		list.tail_(new Boolean(this.clamp()));
		return list;
	}

	/**
	 * Convert the receiver's nearest attribute as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList nearestToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("nearest"));
		list.tail_(new Boolean(this.nearest()));
		return list;
	}

	/**
	 * Convert the receiver's modulate attribute as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList modulateToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("modulate"));
		list.tail_(new Boolean(this.modulate()));
		return list;
	}

	/**
	 * Convert the receiver's mipmap attribute as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList mipmapToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("mipmap"));
		list.tail_(new Boolean(this.mipmap()));
		return list;
	}

	/**
	 * Convert the receiver's S-parameters as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList sParametersToLispList() {
		double[] parameters = this.parametersForS();
		ArrayList elements = new ArrayList();
		elements.add($("sParameters"));
		for (int i = 0; i < parameters.length; i++) {
			elements.add(new Double(parameters[i]));
		}
		return JunLispCons.List_(elements);
	}

	/**
	 * Convert the receiver's S-parameters as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList tParametersToLispList() {
		double[] parameters = this.parametersForT();
		ArrayList elements = new ArrayList();
		elements.add($("sParameters"));
		for (int i = 0; i < parameters.length; i++) {
			elements.add(new Double(parameters[i]));
		}
		return JunLispCons.List_(elements);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.imageFromLispList(aList);
		this.coordinatesFromLispList(aList);
		this.clampFromLispList(aList);
		this.nearestFromLispList(aList);
		this.modulateFromLispList(aList);
		this.mipmapFromLispList(aList);
		this.sParametersFromLispList(aList);
		this.tParametersFromLispList(aList);
	}

	/**
	 * Get my texture image from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void imageFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("image")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.image_((StImage) list.tail());
	}

	/**
	 * Get my texture coordinates from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void coordinatesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("coordinates")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Jun2dPoint[] coordinates = new Jun2dPoint[anArray.length];
		System.arraycopy(anArray, 0, coordinates, 0, anArray.length);
		this.coordinates_(coordinates);
	}

	/**
	 * Get my clamp attribute from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void clampFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("clamp")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.clamp_(((Boolean) list.tail()).booleanValue());
	}

	/**
	 * Get my nearest attribute from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void nearestFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("nearest")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.nearest_(((Boolean) list.tail()).booleanValue());
	}

	/**
	 * Get my modulate attribute from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void modulateFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("modulate")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.modulate_(((Boolean) list.tail()).booleanValue());
	}

	/**
	 * Get my mipmap attribute from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void mipmapFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("mipmap")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.mipmap_(((Boolean) list.tail()).booleanValue());
	}

	/**
	 * Get my S-parameters from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void sParametersFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("sParameters")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		double[] parameters = new double[anArray.length];
		for (int i = 0; i < parameters.length; i++) {
			parameters[i] = ((Number) anArray[i]).doubleValue();
		}
		this.parametersForS_(parameters);
	}

	/**
	 * Get my T-parameters from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void tParametersFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("tParameters")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		double[] parameters = new double[anArray.length];
		for (int i = 0; i < parameters.length; i++) {
			parameters[i] = ((Number) anArray[i]).doubleValue();
		}
		this.parametersForT_(parameters);
	}

	/**
	 * Write the texture as SFImage of VRML on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @category vrml support
	 */
	public void vrmlSFImageOn_(Writer aWriter) {
		try {
			StImage image = this.image();
			int width = image.width();
			int height = image.height();
			aWriter.write(String.valueOf(width));
			aWriter.write(" ");
			aWriter.write(String.valueOf(height));
			aWriter.write(" 3");
			for (int y = height - 1; y >= 0; y--) {
				for (int x = 0; x < width; x++) {
					aWriter.write(" 0x");
					aWriter.write(Integer.toHexString(image.getPixel(x, y)).toUpperCase());
				}
			}
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

}
