package jp.co.sra.jun.opengl.support;

import java.awt.Color;
import java.awt.Dimension;
import java.util.HashMap;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.JunNurbsCurve;
import jp.co.sra.jun.geometry.surfaces.JunNurbsSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.texture.JunOpenGLStipple;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;

/**
 * JunOpenGLRenderingContext class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/10/09 (by MATSUDA Ryouichi)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2003/09/09 (by nisinaka)
 *  @updated   2007/08/20 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun608 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLRenderingContext.java,v 8.14 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunOpenGLRenderingContext extends JunOpenGLResource implements Cloneable {

	protected JunOpenGLDrawable drawable;
	protected Color color;
	protected float pointSize;
	protected float lineWidth;
	protected float nurbsSamplingTolerance;
	protected JunOpenGLProjection projection;
	protected Jun3dTransformation transformation;
	protected StSymbol presentation;
	protected StSymbol shading;
	protected int lightIndex;
	protected HashMap lightStates;
	protected JunOpenGLTexture texture;
	protected int textureCoordinateIndex;
	protected StBlockClosure textureCoordinateBlock;

	/**
	 * Create a new instance of JunOpenGLRenderingContext and initialize it.
	 *
	 * @param anOpenGLDrawable jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @category Instance creation
	 */
	public JunOpenGLRenderingContext(JunOpenGLDrawable anOpenGLDrawable) {
		this.setDrawable(anOpenGLDrawable);
		this.initializeContext();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLResource#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		drawable = null;
		color = null;
		pointSize = 0;
		lineWidth = Float.NaN;
		nurbsSamplingTolerance = 0;
		projection = null;
		transformation = null;
		presentation = null;
		shading = null;
		lightIndex = -1;
		lightStates = null;
		texture = null;
		textureCoordinateIndex = -1;
		textureCoordinateBlock = null;
	}

	/**
	 * Initialize the receiver's context.
	 * 
	 * @category initialize-release
	 */
	protected void initializeContext() {
		lightStates = new HashMap();
		this.projection_(this.defaultProjection());
		transformation = Jun3dTransformation.Unity();
		this.clearDepthBuffer();
		this.enableDepthTest();
		this.enableColorMaterial();
		this.setColor(this.defaultColor());
		this.resetLightIndex();
		//this.disableLighting();
		this.disableCullFace();
		this.enableNormalize();
		this.disableBlending();
		this.disableTexture2d();
		this.nurbsSamplingTolerance_(this.defaultNurbsSamplingTolerance());

		this.initializeLighting();
	}

	/**
	 * Initialize the receiver's lighting.
	 * 
	 * @category initialize-release
	 */
	protected void initializeLighting() {
		this.lightAmbientColor_(Color.black);
		for (int index = 0; index <= this.maxLightIndex(); index++) {
			this.lightIndex_position_color_(index, Jun3dPoint.Zero(), Color.black);
			this.disableLightIndex_(index);
		}
		this.disableLighting();
	}

	/**
	 * Answer my current OpenGL interface.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLInterface
	 * @category accessing
	 */
	protected JunOpenGLInterface getOpenGLInterface() {
		return this.getDrawable().getOpenGLInterface();
	}

	/**
	 * Answer my current drawable.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @category accessing
	 */
	protected JunOpenGLDrawable getDrawable() {
		return drawable;
	}

	/**
	 * Set my new drawable.
	 * 
	 * @param anOpenGLDrawable jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @category accessing
	 */
	protected void setDrawable(JunOpenGLDrawable anOpenGLDrawable) {
		drawable = anOpenGLDrawable;
	}

	/**
	 * Answer my current color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	protected Color getColor() {
		if (color == null) {
			color = this.defaultColor();
		}
		return color;
	}

	/**
	 * Set my new color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	protected void setColor(Color aColor) {
		if (aColor == null) {
			aColor = Color.white;
		}

		color = aColor;

		float[] rgba = color.getRGBComponents(null);
		this.getOpenGLInterface().glColor4f(rgba[0], rgba[1], rgba[2], rgba[3]);
	}

	/**
	 * Answer the paint color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color paint() {
		return new Color(this.getColor().getRGB());
	}

	/**
	 * Set the paint color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	public void paint_(Color aColor) {
		this.setColor(new Color(aColor.getRed(), aColor.getGreen(), aColor.getBlue(), this.getColor().getAlpha()));
	}

	/**
	 * Answer the alpha of the color.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float alpha() {
		return this.getColor().getAlpha() / 255f;
	}

	/**
	 * Set the alpha of the color.
	 * 
	 * @param alphaNumber float
	 * @category accessing
	 */
	public void alpha_(float alphaNumber) {
		int alpha = (int) (alphaNumber * 255 + 0.5);
		this.setColor(new Color(this.getColor().getRed(), this.getColor().getGreen(), this.getColor().getBlue(), alpha));
	}

	/**
	 * Answer the current point size.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float pointSize() {
		return pointSize;
	}

	/**
	 * Set the new point size.
	 * 
	 * @param size float
	 * @category accessing
	 */
	public void pointSize_(float size) {
		pointSize = size;
		this.getOpenGLInterface().glPointSize(pointSize);
	}

	/**
	 * Answer the current line width.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float lineWidth() {
		return lineWidth;
	}

	/**
	 * Set the new line width.
	 * 
	 * @param width float
	 * @category accessing
	 */
	public void lineWidth_(float width) {
		lineWidth = width;
		this.getOpenGLInterface().glLineWidth(lineWidth);
	}

	/**
	 * Answer the current nurbs sampling tolerance.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float nurbsSamplingTolerance() {
		return nurbsSamplingTolerance;
	}

	/**
	 * Set the new nurbs sampling tolerance.
	 * 
	 * @param aNumber float
	 * @category accessing
	 */
	public void nurbsSamplingTolerance_(float aNumber) {
		nurbsSamplingTolerance = aNumber;
	}

	/**
	 * Answer my background color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	protected Color background() {
		return this.getDrawable().getBackgroundColor();
	}

	/**
	 * Answer the default color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultColor() {
		return Color.black;
	}

	/**
	 * Answer the default projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category defaults
	 */
	protected JunOpenGLProjection defaultProjection() {
		return JunOpenGLProjection.Default();
	}

	/**
	 * Answer the default nurbs sampling tolerance.
	 * 
	 * @return float
	 * @category defaults
	 */
	protected float defaultNurbsSamplingTolerance() {
		return 10.0f;
	}

	/**
	 * Enable the blending.
	 * 
	 * @category blending
	 */
	public void enableBlending() {
		this.getOpenGLInterface().glBlendFunc(JunOpenGLInterface.GL_SRC_ALPHA, JunOpenGLInterface.GL_ONE_MINUS_SRC_ALPHA);
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_BLEND);
	}

	/**
	 * Disable the blending.
	 * 
	 * @category blending
	 */
	public void disableBlending() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_BLEND);
	}

	/**
	 * Creates and returns a copy of this object.
	 * 
	 * @return java.lang.Object
	 * @category copying
	 */
	public Object clone() {
		try {
			JunOpenGLRenderingContext clone = (JunOpenGLRenderingContext) super.clone();
			clone.color = new Color(this.color.getRGB(), true);
			return clone;
		} catch (CloneNotSupportedException e) {
			return null;
		}
	}

	/**
	 * Set the cull face to back.
	 * 
	 * @category culling
	 */
	public void cullFaceBack() {
		this.getOpenGLInterface().glCullFace(JunOpenGLInterface.GL_BACK);
	}

	/**
	 * Set the cull face to font.
	 * 
	 * @category culling
	 */
	public void cullFaceFront() {
		this.getOpenGLInterface().glCullFace(JunOpenGLInterface.GL_FRONT);
	}

	/**
	 * Enable the cull face.
	 * 
	 * @category culling
	 */
	public void enableCullFace() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_CULL_FACE);
	}

	/**
	 * Disable the cull face.
	 * 
	 * @category culling
	 */
	public void disableCullFace() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_CULL_FACE);
	}

	/**
	 * Call the display list.
	 * 
	 * @param listId int
	 * @category display list
	 */
	public void callDisplayList_(int listId) {
		this.getOpenGLInterface().glCallList(listId);
	}

	/**
	 * Create a display list and execute if specified.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param doExecute boolean
	 * @return int
	 * @category display list
	 */
	public int createDisplayList_displayImmediately_(StBlockClosure aBlock, boolean doExecute) {
		int listId = this.getOpenGLInterface().glGenLists(1);
		int mode = doExecute ? JunOpenGLInterface.GL_COMPILE_AND_EXECUTE : JunOpenGLInterface.GL_COMPILE;
		this.getOpenGLInterface().glNewList(listId, mode);
		try {
			aBlock.value();
		} finally {
			this.getOpenGLInterface().glEndList();
		}
		return listId;
	}

	/**
	 * Clear the display.
	 * 
	 * @category displaying
	 */
	public void clear() {
		this.clearDepthBuffer();
		this.clearColorBuffer();
	}

	/**
	 * Clear the depth buffer.
	 * 
	 * @category displaying
	 */
	protected void clearDepthBuffer() {
		this.getOpenGLInterface().glClear(JunOpenGLInterface.GL_DEPTH_BUFFER_BIT);
	}

	/**
	 * Clear the color buffer.
	 * 
	 * @category displaying
	 */
	protected void clearColorBuffer() {
		float[] rgba = this.background().getRGBComponents(null);
		this.getOpenGLInterface().glClearColor(rgba[0], rgba[1], rgba[2], 1.0f);
		this.getOpenGLInterface().glClear(JunOpenGLInterface.GL_COLOR_BUFFER_BIT);
	}

	/**
	 * Display the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category displaying
	 */
	public void displayPoint_(Jun3dPoint aPoint) {
		this.displayPoints_(new Jun3dPoint[] { aPoint });
	}

	/**
	 * Display the points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayPoints_(Jun3dPoint[] points) {
		this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_POINTS);
		for (int i = 0; i < points.length; i++) {
			this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
		}
		this.getOpenGLInterface().glEnd();
	}

	/**
	 * Display the lines.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayLines_(Jun3dPoint[] points) {
		this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_LINE_STRIP);
		for (int i = 0; i < points.length; i++) {
			this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
		}
		this.getOpenGLInterface().glEnd();
	}

	/**
	 * Display the line loop.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayLineLoop_(Jun3dPoint[] points) {
		this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_LINE_LOOP);
		for (int i = 0; i < points.length; i++) {
			this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
		}
		this.getOpenGLInterface().glEnd();
	}

	/**
	 * Display the polygon.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayPolygon_(Jun3dPoint[] points) {
		if (this.presentation() == $("wireframe")) {
			this.displaySeeThroughWireframe_(points);
		} else if (this.presentation() == $("hiddenline")) {
			this.displayHiddenLineRendering_(points);
		} else {
			Jun3dPoint normalVector = this.normalVectorOfPoints_(points);
			if (normalVector == null) {
				return;
			}
			this.displayPolygon_normalVector_(points, normalVector);
		}
	}

	/**
	 * Display the polygon with the normal vector.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category displaying
	 */
	public void displayPolygon_normalVector_(Jun3dPoint[] points, Jun3dPoint normalVector) {
		this.displayPolygon_normalVectors_(points, new Jun3dPoint[] { normalVector });
	}

	/**
	 * Display the polygon with the normal vectors.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param normalVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayPolygon_normalVectors_(Jun3dPoint[] points, Jun3dPoint[] normalVectors) {
		if (this.presentation() == $("wireframe")) {
			this.displaySeeThroughWireframe_(points);
		} else if (this.presentation() == $("hiddenline")) {
			this.displayHiddenLineRendering_(points);
		} else {
			this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_POLYGON);
			for (int i = 0; i < points.length; i++) {
				if (textureCoordinateBlock != null) {
					Jun2dPoint p = (Jun2dPoint) textureCoordinateBlock.value_(points[i]);
					if (p != null) {
						this.getOpenGLInterface().glTexCoord2d(p.x(), p.y());
					}
				}
				if (normalVectors != null && normalVectors.length > i) {
					this.getOpenGLInterface().glNormal3d(normalVectors[i].x(), normalVectors[i].y(), normalVectors[i].z());
				}
				this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
			}
			this.getOpenGLInterface().glEnd();
		}
	}

	/**
	 * Display the polygon with the colors and the normal vectors.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param colors java.awt.Color[]
	 * @param normalVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	public void displayPolygon_colors_normalVectors_(Jun3dPoint[] points, Color[] colors, Jun3dPoint[] normalVectors) {
		if (this.presentation() == $("wireframe")) {
			this.displaySeeThroughWireframe_(points);
		} else if (this.presentation() == $("hiddenline")) {
			this.displayHiddenLineRendering_(points);
		} else {
			this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_POLYGON);
			float alpha = this.alpha();
			for (int i = 0; i < points.length; i++) {
				if (textureCoordinateBlock != null) {
					Jun2dPoint p = (Jun2dPoint) textureCoordinateBlock.value_(points[i]);
					if (p != null) {
						this.getOpenGLInterface().glTexCoord2d(p.x(), p.y());
					}
				}
				if (colors != null && colors.length > i) {
					float[] rgb = colors[i].getRGBColorComponents(null);
					this.getOpenGLInterface().glColor4f(rgb[0], rgb[1], rgb[2], alpha);
				}
				if (normalVectors != null && normalVectors.length > i) {
					this.getOpenGLInterface().glNormal3d(normalVectors[i].x(), normalVectors[i].y(), normalVectors[i].z());
				}
				this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
			}
			this.getOpenGLInterface().glEnd();
		}
	}

	/**
	 * Display the nurbs curve. 
	 * 
	 * @param aNurbsCurve jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category displaying
	 */
	public void displayNurbsCurve_(JunNurbsCurve aNurbsCurve) {
		float[] knotVector = new float[aNurbsCurve.knotVector().length];
		for (int i = 0; i < aNurbsCurve.knotVector().length; i++) {
			knotVector[i] = (float) aNurbsCurve.knotVector()[i];
		}
		Jun3dPoint[] controlPoints = aNurbsCurve.controlPoints();
		double[] weights = aNurbsCurve.weights();
		float[] floatArray = new float[aNurbsCurve.controlPointSize() * 4];
		for (int i = 0; i < aNurbsCurve.controlPointSize(); i++) {
			Jun3dPoint point = controlPoints[i];
			double weight = weights[i];
			floatArray[(i * 4) + 0] = (float) (point.x() * weight);
			floatArray[(i * 4) + 1] = (float) (point.y() * weight);
			floatArray[(i * 4) + 2] = (float) (point.z() * weight);
			floatArray[(i * 4) + 3] = (float) weight;
		}
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_AUTO_NORMAL);
		int aNurbs = this.getOpenGLInterface().gluNewNurbsRenderer();
		this.getOpenGLInterface().gluNurbsProperty(aNurbs, JunOpenGLInterface.GLU_SAMPLING_TOLERANCE, nurbsSamplingTolerance);
		this.getOpenGLInterface().glPushMatrix();
		this.getOpenGLInterface().gluBeginCurve(aNurbs);
		this.getOpenGLInterface().gluNurbsCurve(aNurbs, knotVector.length, knotVector, 4, floatArray, aNurbsCurve.order(), JunOpenGLInterface.GL_MAP1_VERTEX_4);
		this.getOpenGLInterface().gluEndCurve(aNurbs);
		this.getOpenGLInterface().glPopMatrix();
		this.getOpenGLInterface().gluDeleteNurbsRenderer(aNurbs);
	}

	/**
	 * Display the nurbs surface.
	 * 
	 * @param aNurbsSurface jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @category displaying
	 */
	public void displayNurbsSurface_(JunNurbsSurface aNurbsSurface) {
		if (this.presentation() == $("wireframe")) {
			this.displayNurbsSurface_displayMode_(aNurbsSurface, JunOpenGLInterface.GLU_OUTLINE_POLYGON);
		} else if (this.presentation() == $("hiddenline")) {
			this.disableLighting();
			Color savedPaint = null;
			if (this.paint().equals(Color.white)) {
				savedPaint = this.paint();
				this.paint_(StColorValue.Brightness_(0.95));
			}
			this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_DEPTH_TEST);
			this.getOpenGLInterface().glPolygonMode(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_LINE);
			this.displayNurbsSurface_displayMode_(aNurbsSurface, JunOpenGLInterface.GLU_OUTLINE_POLYGON);
			if (savedPaint != null) {
				this.paint_(savedPaint);
			}
			savedPaint = this.paint();
			this.paint_(Color.white);
			this.getOpenGLInterface().glPolygonMode(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_FILL);
			this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_POLYGON_OFFSET_FILL);
			this.getOpenGLInterface().glPolygonOffset(1.0f, 1.0f);
			this.displayNurbsSurface_displayMode_(aNurbsSurface, JunOpenGLInterface.GLU_FILL);
			this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_POLYGON_OFFSET_FILL);
			this.paint_(savedPaint);
			this.enableLighting();
		} else {
			this.displayNurbsSurface_displayMode_(aNurbsSurface, JunOpenGLInterface.GLU_FILL);
		}
	}

	/**
	 * Display the see through wireframe.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	protected void displaySeeThroughWireframe_(Jun3dPoint[] points) {
		this.displayLineLoop_(points);
	}

	/**
	 * Display the see through wireframe.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category displaying
	 */
	protected void displayHiddenLineRendering_(Jun3dPoint[] points) {
		this.getOpenGLInterface().glPolygonMode(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_LINE);
		this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_LINE_LOOP);
		for (int i = 0; i < points.length; i++) {
			this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
		}
		this.getOpenGLInterface().glEnd();

		this.disableLighting();
		Color savedPaint = this.paint();
		this.paint_(this.background());

		this.getOpenGLInterface().glPolygonMode(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_FILL);
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_POLYGON_OFFSET_FILL);
		this.getOpenGLInterface().glPolygonOffset(1.0f, 1.0f);
		this.getOpenGLInterface().glBegin(JunOpenGLInterface.GL_POLYGON);
		for (int i = 0; i < points.length; i++) {
			this.getOpenGLInterface().glVertex3d(points[i].x(), points[i].y(), points[i].z());
		}
		this.getOpenGLInterface().glEnd();
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_POLYGON_OFFSET_FILL);

		this.paint_(savedPaint);
		this.enableLighting();
	}

	/**
	 * Display the nurbs surface with the specified display mode.
	 * 
	 * @param aNurbsSurface jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @param mode int
	 * @category displaying
	 */
	protected void displayNurbsSurface_displayMode_(JunNurbsSurface aNurbsSurface, int mode) {
		float[] uKnotVector = new float[aNurbsSurface.uKnotVector().length];
		for (int i = 0; i < aNurbsSurface.uKnotVector().length; i++) {
			uKnotVector[i] = (float) aNurbsSurface.uKnotVector()[i];
		}
		float[] vKnotVector = new float[aNurbsSurface.vKnotVector().length];
		for (int i = 0; i < aNurbsSurface.vKnotVector().length; i++) {
			vKnotVector[i] = (float) aNurbsSurface.vKnotVector()[i];
		}
		int uSize = aNurbsSurface.uSize();
		int vSize = aNurbsSurface.vSize();
		float[] floatArray = new float[uSize * vSize * 4];
		int _arrayIndex = 0;
		for (int uIndex = 0; uIndex < uSize; uIndex++) {
			for (int vIndex = 0; vIndex < vSize; vIndex++) {
				Jun3dPoint point;
				double weight;
				point = aNurbsSurface.controlPointAtU_v_(uIndex, vIndex);
				weight = aNurbsSurface.weightAtU_v_(uIndex, vIndex);
				floatArray[_arrayIndex++] = (float) (point.x() * weight);
				floatArray[_arrayIndex++] = (float) (point.y() * weight);
				floatArray[_arrayIndex++] = (float) (point.z() * weight);
				floatArray[_arrayIndex++] = (float) weight;
			}
		}

		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_AUTO_NORMAL);
		int aNurbs = this.getOpenGLInterface().gluNewNurbsRenderer();
		this.getOpenGLInterface().gluNurbsProperty(aNurbs, JunOpenGLInterface.GLU_SAMPLING_TOLERANCE, nurbsSamplingTolerance);
		this.getOpenGLInterface().gluNurbsProperty(aNurbs, JunOpenGLInterface.GLU_DISPLAY_MODE, mode);
		this.getOpenGLInterface().glPushMatrix();
		this.getOpenGLInterface().gluBeginSurface(aNurbs);
		this.getOpenGLInterface().gluNurbsSurface(aNurbs, uKnotVector.length, uKnotVector, vKnotVector.length, vKnotVector, 4 * vSize, 4, floatArray, aNurbsSurface.uOrder(), aNurbsSurface.vOrder(), JunOpenGLInterface.GL_MAP2_VERTEX_4);
		/*
		aNurbsSurface.trimCurvesDo_(new StBlockClosure() {
			public Object value_(Object o) {
				JunOpenGL2dTrimCurve trimCurve = (JunOpenGL2dTrimCurve) o;
				trimCurve.renderOn_with_for_(this, aNurbsSurface, aNurbs);
				return null;
			}
		});
		*/
		this.getOpenGLInterface().gluEndSurface(aNurbs);
		this.getOpenGLInterface().glPopMatrix();
		this.getOpenGLInterface().gluDeleteNurbsRenderer(aNurbs);
	}

	/**
	 * Enable the dithering.
	 * 
	 * @category dithering
	 */
	public void enableDithering() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_DITHER);
	}

	/**
	 * Disable the dithering.
	 * 
	 * @category dithering
	 */
	public void disableDithering() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_DITHER);
	}

	/**
	 * Set the fog color.
	 * 
	 * @param aColor java.awt.Color
	 * @category fogging
	 */
	public void fogColor_(Color aColor) {
		this.getOpenGLInterface().glFogfv(JunOpenGLInterface.GL_FOG_COLOR, GetRGBA(aColor, 1.0f));
	}

	/**
	 * Set the exp2 fog density.
	 * 
	 * @param aNumber float
	 * @category fogging
	 */
	public void fogExp2Density_(float aNumber) {
		this.getOpenGLInterface().glFogi(JunOpenGLInterface.GL_FOG_MODE, JunOpenGLInterface.GL_EXP2);
		this.getOpenGLInterface().glFogf(JunOpenGLInterface.GL_FOG_DENSITY, aNumber);
	}

	/**
	 * Set the exp fog density.
	 * 
	 * @param aNumber float
	 * @category fogging
	 */
	public void fogExpDensity(float aNumber) {
		this.getOpenGLInterface().glFogi(JunOpenGLInterface.GL_FOG_MODE, JunOpenGLInterface.GL_EXP);
		this.getOpenGLInterface().glFogf(JunOpenGLInterface.GL_FOG_DENSITY, aNumber);
	}

	/**
	 * Set the linear fog density.
	 * 
	 * @param startNumber float
	 * @param endNumber float
	 * @category fogging
	 */
	public void fogLinearStartDensity_endDensity_(float startNumber, float endNumber) {

	}

	/**
	 * Enable the fogging.
	 * 
	 * @category fogging
	 */
	public void enableFog() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_FOG);
	}

	/**
	 * Disable the fogging.
	 * 
	 * @category fogging
	 */
	public void disableFog() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_FOG);
	}

	/**
	 * Answer the current light index.
	 * 
	 * @return int
	 * @category lighting
	 */
	public int getLightIndex() {
		lightIndex = (lightIndex % this.maxLightIndex()) + 1;
		return lightIndex;
	}

	/**
	 * Set the color as an ambient light.
	 * 
	 * @param aColor java.awt.Color
	 * @category lighting
	 */
	public void lightAmbientColor_(Color aColor) {
		this.lightAmbientColor_alpha_twoSide_(aColor, 1.0f, false);
	}

	/**
	 * Set the color as an ambient light.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category lighting
	 */
	public void lightAmbientColor_alpha_(Color aColor, float aNumber) {
		this.lightAmbientColor_alpha_twoSide_(aColor, aNumber, false);
	}

	/**
	 * Set the color as an ambient light.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @param aBoolean boolean
	 * @category lighting
	 */
	public void lightAmbientColor_alpha_twoSide_(Color aColor, float aNumber, boolean aBoolean) {
		this.getOpenGLInterface().glLightModelfv(JunOpenGLInterface.GL_LIGHT_MODEL_AMBIENT, GetRGBA(aColor, aNumber));
		this.getOpenGLInterface().glLightModelf(JunOpenGLInterface.GL_LIGHT_MODEL_LOCAL_VIEWER, 0.0f);
		this.getOpenGLInterface().glLightModelf(JunOpenGLInterface.GL_LIGHT_MODEL_TWO_SIDE, aBoolean ? 1.0f : 0.0f);
	}

	/**
	 * Setup the specified light. 
	 * 
	 * @param index0to7 int
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aColor java.awt.Color
	 * @category lighting
	 */
	public void lightIndex_position_color_(int index0to7, Jun3dPoint a3dPoint, Color aColor) {
		this.lightIndex_position_direction_cutoffAngle_color_alpha_(index0to7, a3dPoint, null, null, aColor, 1.0f);
	}

	/**
	 * Setup the specified light. 
	 * 
	 * @param index0to7 int
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aColorValue java.awt.Color
	 * @param aNumber float
	 * @category lighting
	 */
	public void lightIndex_position_direction_cutoffAngle_color_alpha_(int index0to7, Jun3dPoint a3dPoint, Jun3dPoint a3dPoint2, JunAngle aJunAngle, Color aColorValue, float aNumber) {
		this.lightIndex_position_direction_cutoffAngle_color_alpha_linearAttenuation_quadraticAttenuation_constantAttenuation_(index0to7, a3dPoint, a3dPoint2, aJunAngle, aColorValue, aNumber, 0.0f, 0.0f, 1.0f);
	}

	/**
	 * Setup the specified light.
	 * 
	 * @param index0to7 int
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aColorValue java.awt.Color
	 * @param aNumber float
	 * @param aNumber2 float
	 * @param aNumber3 float
	 * @param aNumber4 float
	 * @category lighting
	 */
	public void lightIndex_position_direction_cutoffAngle_color_alpha_linearAttenuation_quadraticAttenuation_constantAttenuation_(int index0to7, Jun3dPoint a3dPoint, Jun3dPoint a3dPoint2, JunAngle aJunAngle, Color aColorValue, float aNumber,
			float aNumber2, float aNumber3, float aNumber4) {
		StBlockClosure copyPointToArray3Block = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				float[] array = (float[]) o1;
				Jun3dPoint point = (Jun3dPoint) o2;
				array[0] = (float) point.x();
				array[1] = (float) point.y();
				array[2] = (float) point.z();
				return this;
			}
		};
		StBlockClosure copyPointToArray4Block = new StBlockClosure() {
			public Object value_value_value_(Object o1, Object o2, Object o3) {
				float[] array = (float[]) o1;
				Jun3dPoint point = (Jun3dPoint) o2;
				Number w = (Number) o3;
				array[0] = (float) point.x();
				array[1] = (float) point.y();
				array[2] = (float) point.z();
				array[3] = w.floatValue();
				return this;
			}
		};
		StBlockClosure copyColorToArray4Block = new StBlockClosure() {
			public Object value_value_value_(Object o1, Object o2, Object o3) {
				float[] array = (float[]) o1;
				Color colorValue = (Color) o2;
				Number a = (Number) o3;
				array[0] = (float) colorValue.getRed() / 255.0f;
				array[1] = (float) colorValue.getGreen() / 255.0f;
				array[2] = (float) colorValue.getBlue() / 255.0f;
				array[3] = a.floatValue();
				return this;
			}
		};

		float[] positionArray;
		float[] directionArray;
		Float cutoff;

		Jun3dPoint position = a3dPoint;
		Jun3dPoint direction = a3dPoint2;
		if (position == null) {
			if (direction == null) {
				return;
			} else {
				positionArray = new float[4];
				copyPointToArray4Block.value_value_value_(positionArray, direction.negated(), new Float(0.0f));
				directionArray = new float[3];
				copyPointToArray3Block.value_value_(directionArray, direction);
				cutoff = new Float(90.0f);
			}
		} else {
			if (direction == null) {
				positionArray = new float[4];
				copyPointToArray4Block.value_value_value_(positionArray, position, new Float(1.0f));
				directionArray = null;
				cutoff = null;
			} else {
				positionArray = new float[4];
				copyPointToArray4Block.value_value_value_(positionArray, position, new Float(1.0f));
				directionArray = new float[3];
				copyPointToArray3Block.value_value_(directionArray, direction);
				if (aJunAngle == null) {
					cutoff = new Float(1.0f);
				} else {
					cutoff = new Float(aJunAngle.deg());
				}
			}
		}
		float[] colorArray = new float[4];
		copyColorToArray4Block.value_value_value_(colorArray, (aColorValue == null) ? Color.white : aColorValue, new Float(aNumber));
		Object[] lightInfo = new Object[7];
		lightInfo[0] = positionArray;
		lightInfo[1] = directionArray;
		lightInfo[2] = cutoff;
		lightInfo[3] = colorArray;
		lightInfo[4] = new Float(aNumber2);
		lightInfo[5] = new Float(aNumber3);
		lightInfo[6] = new Float(aNumber4);
		this.lightIndex_put_(index0to7, lightInfo);
	}

	/**
	 * Setup the specified light.
	 * 
	 * @param index0to7 int
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBoolean boolean
	 * @param aColorValue java.awt.Color
	 * @param aNumber float
	 * @category lighting
	 */
	public void lightIndex_position_parallel_color_alpha_(int index0to7, Jun3dPoint a3dPoint, boolean aBoolean, Color aColorValue, float aNumber) {
		this.lightIndex_position_direction_cutoffAngle_color_alpha_(index0to7, null, (Jun3dPoint) a3dPoint.negated(), null, aColorValue, aNumber);
	}

	/**
	 * Setup the specified light.
	 * 
	 * @param index0to7 int
	 * @param anArray java.lang.Object[]
	 * @category lighting
	 */
	protected void lightIndex_put_(int index0to7, Object[] anArray) {
		float[] positionArray = (float[]) anArray[0];
		float[] directionArray = (float[]) anArray[1];
		Float cutoff = (Float) anArray[2];
		float[] colorArray = (float[]) anArray[3];
		Float linear = (Float) anArray[4];
		Float quadratic = (Float) anArray[5];
		Float constant = (Float) anArray[6];
		this.enableLightIndex_(index0to7);

		// lights.put(new Integer(index0to7), anArray)
		int index = JunOpenGLInterface.GL_LIGHT0 + index0to7;
		if (positionArray != null) {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_POSITION, positionArray);
		} else {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_POSITION, new float[] { 0.0f, 0.0f, 1.0f, 0.0f });
		}
		if (directionArray != null) {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_SPOT_DIRECTION, directionArray);
		} else {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_SPOT_DIRECTION, new float[] { 0.0f, 0.0f, -1.0f });
		}
		if (cutoff != null) {
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_SPOT_CUTOFF, cutoff.floatValue());
		} else {
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_SPOT_CUTOFF, 180.0f);
		}
		if (colorArray != null) {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_DIFFUSE, colorArray);
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_SPECULAR, colorArray);
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_LINEAR_ATTENUATION, linear.floatValue());
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_QUADRATIC_ATTENUATION, quadratic.floatValue());
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_CONSTANT_ATTENUATION, constant.floatValue());
		} else {
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_DIFFUSE, new float[] { 1.0f, 1.0f, 1.0f, 1.0f });
			this.getOpenGLInterface().glLightfv(index, JunOpenGLInterface.GL_SPECULAR, new float[] { 1.0f, 1.0f, 1.0f, 1.0f });
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_LINEAR_ATTENUATION, 0.0f);
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_QUADRATIC_ATTENUATION, 0.0f);
			this.getOpenGLInterface().glLightf(index, JunOpenGLInterface.GL_CONSTANT_ATTENUATION, 0.0f);
		}
	}

	/**
	 * Enable the lighting.
	 * 
	 * @category lighting
	 */
	public void enableLighting() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_LIGHTING);
	}

	/**
	 * Disable the lighting.
	 * 
	 * @category lighting
	 */
	public void disableLighting() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_LIGHTING);
	}

	/**
	 * Enable the specified light.
	 * 
	 * @param index0to7 int
	 * @category lighting
	 */
	public void enableLightIndex_(int index0to7) {
		int index = JunOpenGLInterface.GL_LIGHT0 + index0to7;
		lightStates.put(new Integer(index0to7), Boolean.TRUE);
		this.getOpenGLInterface().glEnable(index);
	}

	/**
	 * Disable the specified light.
	 * 
	 * @param index0to7 int
	 * @category lighting
	 */
	public void disableLightIndex_(int index0to7) {
		int index = JunOpenGLInterface.GL_LIGHT0 + index0to7;
		lightStates.put(new Integer(index0to7), Boolean.FALSE);
		this.getOpenGLInterface().glDisable(index);
	}

	/**
	 * Answer the maximum number of the light index.
	 * 
	 * @return int
	 * @category lighting
	 */
	protected int maxLightIndex() {
		return 7;
	}

	/**
	 * Reset the light index.
	 * 
	 * @category lighting
	 */
	protected void resetLightIndex() {
		lightIndex = 0;
	}

	/**
	 * Set the material ambient.
	 * 
	 * @param aColor java.awt.Color
	 * @category material
	 */
	public void materialAmbient_(Color aColor) {
		this.materialAmbient_alpha_(aColor, 1.0f);
	}

	/**
	 * Set the material ambient.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category material
	 */
	public void materialAmbient_alpha_(Color aColor, float aNumber) {
		this.getOpenGLInterface().glMaterialfv(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_AMBIENT, GetRGBA(aColor, aNumber));
	}

	/**
	 * Set the material diffuse.
	 * 
	 * @param aColor java.awt.Color
	 * @category material
	 */
	public void materialDiffuse_(Color aColor) {
		this.materialDiffuse_alpha_(aColor, 1.0f);
	}

	/**
	 * Set the material diffuse.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category material
	 */
	public void materialDiffuse_alpha_(Color aColor, float aNumber) {
		this.getOpenGLInterface().glMaterialfv(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_DIFFUSE, GetRGBA(aColor, aNumber));
	}

	/**
	 * Set the material emission.
	 * 
	 * @param aColor java.awt.Color
	 * @category material
	 */
	public void materialEmission_(Color aColor) {
		this.materialEmission_alpha_(aColor, 1.0f);
	}

	/**
	 * Set the material emission.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category material
	 */
	public void materialEmission_alpha_(Color aColor, float aNumber) {
		this.getOpenGLInterface().glMaterialfv(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_EMISSION, GetRGBA(aColor, aNumber));
	}

	/**
	 * Set the material shininess.
	 * 
	 * @param aNumber float
	 * @category material
	 */
	public void materialShininess_(float aNumber) {
		this.getOpenGLInterface().glMaterialf(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_SHININESS, aNumber);
	}

	/**
	 * Set the material specular.
	 * 
	 * @param aColor java.awt.Color
	 * @category material
	 */
	public void materialSpecular_(Color aColor) {
		this.materialSpecular_alpha_(aColor, 1.0f);
	}

	/**
	 * Set the material specular.
	 * 
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category material
	 */
	public void materialSpecular_alpha_(Color aColor, float aNumber) {
		this.getOpenGLInterface().glMaterialfv(JunOpenGLInterface.GL_FRONT_AND_BACK, JunOpenGLInterface.GL_SPECULAR, GetRGBA(aColor, aNumber));
	}

	/**
	 * Enable the color material.
	 * 
	 * @category material
	 */
	public void enableColorMaterial() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_COLOR_MATERIAL);
	}

	/**
	 * Disable the color material.
	 * 
	 * @category material
	 */
	public void disableColorMaterial() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_COLOR_MATERIAL);
	}

	/**
	 * Enable the normalize.
	 * 
	 * @category normalizing
	 */
	public void enableNormalize() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_NORMALIZE);
	}

	/**
	 * Disable the normalize.
	 * 
	 * @category normalizing
	 */
	public void disableNormalize() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_NORMALIZE);
	}

	/**
	 * Set my presentation to solid.
	 * 
	 * @category presentation
	 */
	public void solidPresentation() {
		this.presentation_($("solid"));
	}

	/**
	 * Set my presentation to wireframe.
	 * 
	 * @category presentation
	 */
	public void wireframePresentation() {
		this.presentation_($("wireframe"));
	}

	/**
	 * Set my presentation to hiddenline.
	 * 
	 * @category presentation
	 */
	public void hiddenlinePresentation() {
		this.presentation_($("hiddenline"));
	}

	/**
	 * Answer my current presentation.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category presentation
	 */
	protected StSymbol presentation() {
		if (presentation == null) {
			presentation = $("solid");
		}
		return presentation;
	}

	/**
	 * Set my new presentation.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category presentation
	 */
	protected void presentation_(StSymbol aSymbol) {
		if (aSymbol == $("solid") || aSymbol == $("wireframe") || aSymbol == $("hiddenline")) {
			presentation = aSymbol;
		}
	}

	/**
	 * Set the specified parallel projection.
	 * 
	 * @param anOpenGLParallelProjection jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @category projecting
	 */
	public void parallelProjection_(JunOpenGLParallelProjection anOpenGLParallelProjection) {
		JunOpenGLParallelProjection projection = (JunOpenGLParallelProjection) anOpenGLParallelProjection.copy();
		this.setProjection(projection);

		Dimension extent = this.getDrawable().getSize();
		double aspect = extent.getWidth() / extent.getHeight();
		double halfHeight = projection.height() / 2;
		double halfWidth = halfHeight * aspect;
		Jun3dPoint eyePoint = projection.eyePoint();
		Jun3dPoint sightPoint = projection.sightPoint();
		Jun3dPoint upVector = projection.upVector();

		this.getOpenGLInterface().glMatrixMode(JunOpenGLInterface.GL_PROJECTION);
		this.getOpenGLInterface().glLoadIdentity();
		this.getOpenGLInterface().glOrtho(-halfWidth, halfWidth, -halfHeight, halfHeight, projection.near(), projection.far());
		this.getOpenGLInterface().glMatrixMode(JunOpenGLInterface.GL_MODELVIEW);
		this.getOpenGLInterface().glLoadIdentity();
		this.getOpenGLInterface().gluLookAt(eyePoint.x(), eyePoint.y(), eyePoint.z(), sightPoint.x(), sightPoint.y(), sightPoint.z(), upVector.x(), upVector.y(), upVector.z());
	}

	/**
	 * Set the specified perspective projection.
	 * 
	 * @param anOpenGLPerspectiveProjection jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @category projecting
	 */
	public void perspective_(JunOpenGLPerspectiveProjection anOpenGLPerspectiveProjection) {
		JunOpenGLPerspectiveProjection projection = (JunOpenGLPerspectiveProjection) anOpenGLPerspectiveProjection.copy();
		this.setProjection(projection);

		Dimension extent = this.getDrawable().getSize();
		double aspect = extent.getWidth() / extent.getHeight();
		Jun3dPoint eyePoint = projection.eyePoint();
		Jun3dPoint sightPoint = projection.sightPoint();
		Jun3dPoint upVector = projection.upVector();

		this.getOpenGLInterface().glMatrixMode(JunOpenGLInterface.GL_PROJECTION);
		this.getOpenGLInterface().glLoadIdentity();
		this.getOpenGLInterface().gluPerspective(projection.fovy().deg(), aspect, projection.near(), projection.far());
		this.getOpenGLInterface().glMatrixMode(JunOpenGLInterface.GL_MODELVIEW);
		this.getOpenGLInterface().glLoadIdentity();
		this.getOpenGLInterface().gluLookAt(eyePoint.x(), eyePoint.y(), eyePoint.z(), sightPoint.x(), sightPoint.y(), sightPoint.z(), upVector.x(), upVector.y(), upVector.z());
	}

	/**
	 * Answer the copy of the current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category projecting
	 */
	public JunOpenGLProjection projection() {
		return (JunOpenGLProjection) projection.copy();
	}

	/**
	 * Set my new projection.
	 * 
	 * @param anOpenGLProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category projecting
	 */
	public void projection_(JunOpenGLProjection anOpenGLProjection) {
		anOpenGLProjection.projectOn_(this);
	}

	/**
	 * Set my shading to flat.
	 * 
	 * @category shading
	 */
	public void flatShading() {
		shading = $("flat");
		this.getOpenGLInterface().glShadeModel(JunOpenGLInterface.GL_FLAT);
	}

	/**
	 * Set my shading to smooth.
	 * 
	 * @category shading
	 */
	public void smoothShading() {
		shading = $("smooth");
		this.getOpenGLInterface().glShadeModel(JunOpenGLInterface.GL_SMOOTH);
	}

	/**
	 * Answer my current shading.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category shading
	 */
	public StSymbol shading() {
		return shading;
	}

	/**
	 * Set the line smooth hint to nicest.
	 * 
	 * @category smoothing
	 */
	public void lineSmoothNicest() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_LINE_SMOOTH_HINT, JunOpenGLInterface.GL_NICEST);
	}

	/**
	 * Set the line smooth hint to fastest.
	 * 
	 * @category smoothing
	 */
	public void lineSmoothFastest() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_LINE_SMOOTH_HINT, JunOpenGLInterface.GL_FASTEST);
	}

	/**
	 * Set the line smooth hint to don't care.
	 * 
	 * @category smoothing
	 */
	public void lineSmoothDontCare() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_LINE_SMOOTH_HINT, JunOpenGLInterface.GL_DONT_CARE);
	}

	/**
	 * Enable the line smooth.
	 * 
	 * @category smoothing
	 */
	public void enableLineSmooth() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_LINE_SMOOTH);
		this.lineSmoothNicest();
	}

	/**
	 * Disable the line smooth.
	 * 
	 * @category smoothing
	 */
	public void disableLineSmooth() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_LINE_SMOOTH);
	}

	/**
	 * Set the polygon smooth hint to nicest.
	 * 
	 * @category smoothing
	 */
	public void polygonSmoothNicest() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_POLYGON_SMOOTH_HINT, JunOpenGLInterface.GL_NICEST);
	}

	/**
	 * Set the polygon smooth hint to fastest.
	 * 
	 * @category smoothing
	 */
	public void polygonSmoothFastest() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_POLYGON_SMOOTH_HINT, JunOpenGLInterface.GL_FASTEST);
	}

	/**
	 * Set the polygon smooth hint to don't care.
	 * 
	 * @category smoothing
	 */
	public void polygonSmoothDontCare() {
		this.getOpenGLInterface().glHint(JunOpenGLInterface.GL_POLYGON_SMOOTH_HINT, JunOpenGLInterface.GL_DONT_CARE);
	}

	/**
	 * Enable the polygon smooth.
	 * 
	 * @category smoothing
	 */
	public void enablePolygonSmooth() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_POLYGON_SMOOTH);
		this.polygonSmoothNicest();
	}

	/**
	 * Disable the polygon smooth.
	 * 
	 * @category smoothing
	 */
	public void disablePolygonSmooth() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_POLYGON_SMOOTH);
	}

	/**
	 * Set the line stipple factor and pattern.
	 * 
	 * @param factor int
	 * @param pattern short
	 * @category stippling
	 */
	public void lineStippleFactor_pattern_(int factor, short pattern) {
		this.getOpenGLInterface().glLineStipple(factor, pattern);
	}

	/**
	 * Enable the line stipple.
	 * 
	 * @category stippling
	 */
	public void enableLineStipple() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_LINE_STIPPLE);
	}

	/**
	 * Disable the line stipple.
	 * 
	 * @category stippling
	 */
	public void disableLineStipple() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_LINE_STIPPLE);
	}

	/**
	 * Set the polygon stipple.
	 * 
	 * @param anOpenGLStipple jp.co.sra.jun.opengl.texture.JunOpenGLStipple
	 * @category stippling
	 */
	public void polygonStipple_(JunOpenGLStipple anOpenGLStipple) {
		this.getOpenGLInterface().glPolygonStipple(anOpenGLStipple.pointerOfPixels());
	}

	/**
	 * Enable the polygon stipple.
	 * 
	 * @category stippling
	 */
	public void enablePolygonStipple() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_POLYGON_STIPPLE);
	}

	/**
	 * Disable the polygon stipple.
	 * 
	 * @category stippling
	 */
	public void disablePolygonStipple() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_POLYGON_STIPPLE);
	}

	/**
	 * Answer the current texture.
	 * 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category texture mapping
	 */
	public JunOpenGLTexture texture() {
		return texture;
	}

	/**
	 * Set the new texture.
	 * 
	 * @param anOpenGLTexture jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category texture mapping
	 */
	public void texture_(JunOpenGLTexture anOpenGLTexture) {
		if (anOpenGLTexture != null) {
			this.getOpenGLInterface().glPixelStorei(JunOpenGLInterface.GL_UNPACK_ALIGNMENT, 1);
			if (anOpenGLTexture.mipmap()) {
				this.getOpenGLInterface().gluBuild2DMipmaps(JunOpenGLInterface.GL_TEXTURE_2D, 3, anOpenGLTexture.width(), anOpenGLTexture.height(), JunOpenGLInterface.GL_RGB, JunOpenGLInterface.GL_UNSIGNED_BYTE, anOpenGLTexture.pointerOfPixels());
			} else {
				this.getOpenGLInterface().glTexImage2D(JunOpenGLInterface.GL_TEXTURE_2D, 0, 3, anOpenGLTexture.width(), anOpenGLTexture.height(), 0, JunOpenGLInterface.GL_RGB, JunOpenGLInterface.GL_UNSIGNED_BYTE, anOpenGLTexture.pointerOfPixels());
			}
			this.getOpenGLInterface().glTexParameteri(JunOpenGLInterface.GL_TEXTURE_2D, JunOpenGLInterface.GL_TEXTURE_WRAP_S, (anOpenGLTexture.clamp()) ? JunOpenGLInterface.GL_CLAMP : JunOpenGLInterface.GL_REPEAT);
			this.getOpenGLInterface().glTexParameteri(JunOpenGLInterface.GL_TEXTURE_2D, JunOpenGLInterface.GL_TEXTURE_WRAP_T, (anOpenGLTexture.clamp()) ? JunOpenGLInterface.GL_CLAMP : JunOpenGLInterface.GL_REPEAT);
			this.getOpenGLInterface().glTexParameteri(JunOpenGLInterface.GL_TEXTURE_2D, JunOpenGLInterface.GL_TEXTURE_MAG_FILTER, (anOpenGLTexture.nearest()) ? JunOpenGLInterface.GL_NEAREST : JunOpenGLInterface.GL_LINEAR);
			if (anOpenGLTexture.mipmap()) {
				this.getOpenGLInterface().glTexParameteri(JunOpenGLInterface.GL_TEXTURE_2D, JunOpenGLInterface.GL_TEXTURE_MIN_FILTER, (anOpenGLTexture.nearest()) ? JunOpenGLInterface.GL_NEAREST_MIPMAP_NEAREST : JunOpenGLInterface.GL_LINEAR_MIPMAP_LINEAR);
			} else {
				this.getOpenGLInterface().glTexParameteri(JunOpenGLInterface.GL_TEXTURE_2D, JunOpenGLInterface.GL_TEXTURE_MIN_FILTER, (anOpenGLTexture.nearest()) ? JunOpenGLInterface.GL_NEAREST : JunOpenGLInterface.GL_LINEAR);
			}
			if (anOpenGLTexture.modulate()) {
				this.getOpenGLInterface().glTexEnvi(JunOpenGLInterface.GL_TEXTURE_ENV, JunOpenGLInterface.GL_TEXTURE_ENV_MODE, JunOpenGLInterface.GL_MODULATE);
			} else {
				this.getOpenGLInterface().glTexEnvi(JunOpenGLInterface.GL_TEXTURE_ENV, JunOpenGLInterface.GL_TEXTURE_ENV_MODE, JunOpenGLInterface.GL_DECAL);
			}
			if (anOpenGLTexture.hasCoordinates()) {
				textureCoordinateBlock = new StBlockClosure() {
					public Object value_(Object o) {
						if (textureCoordinateIndex() > (texture.coordinates().length - 1)) {
							textureCoordinateIndex_(0);
						}
						Jun2dPoint point = (Jun2dPoint) texture.coordinates()[textureCoordinateIndex()];
						textureCoordinateIndex_(textureCoordinateIndex() + 1);
						return point;
					}
				};
			} else {
				this.getOpenGLInterface().glTexGeni(JunOpenGLInterface.GL_S, JunOpenGLInterface.GL_TEXTURE_GEN_MODE, JunOpenGLInterface.GL_OBJECT_LINEAR);
				this.getOpenGLInterface().glTexGenfv(JunOpenGLInterface.GL_S, JunOpenGLInterface.GL_OBJECT_PLANE, anOpenGLTexture.pointerOfParametersForS());
				this.getOpenGLInterface().glTexGeni(JunOpenGLInterface.GL_T, JunOpenGLInterface.GL_TEXTURE_GEN_MODE, JunOpenGLInterface.GL_OBJECT_LINEAR);
				this.getOpenGLInterface().glTexGenfv(JunOpenGLInterface.GL_T, JunOpenGLInterface.GL_OBJECT_PLANE, anOpenGLTexture.pointerOfParametersForT());
				this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_TEXTURE_GEN_S);
				this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_TEXTURE_GEN_T);
				this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_TEXTURE_2D);
			}
		} else {
			textureCoordinateBlock = null;
		}

		texture = anOpenGLTexture;
	}

	/**
	 * Answer the current texture coordinate index.
	 * 
	 * @return int
	 * @category texture mapping
	 */
	public int textureCoordinateIndex() {
		if (textureCoordinateIndex < 0) {
			textureCoordinateIndex = 0;
		}
		return textureCoordinateIndex;
	}

	/**
	 * Set the new texture coordinate index.
	 * 
	 * @param index int
	 * @category texture mapping
	 */
	public void textureCoordinateIndex_(int index) {
		textureCoordinateIndex = index;
	}

	/**
	 * Enable the texture mapping. 
	 * 
	 * @category texture mapping
	 */
	public void enableTexture2d() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_TEXTURE_2D);
	}

	/**
	 * Disable the texture mapping. 
	 * 
	 * @category texture mapping
	 */
	public void disableTexture2d() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_TEXTURE_2D);
	}

	/**
	 * Answer my current transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transformation
	 */
	public Jun3dTransformation transformation() {
		return transformation;
	}

	/**
	 * Product the transformation while evaluating the block.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure 
	 * @category transformation
	 */
	public void productTransformation_while_(Jun3dTransformation aTransformation, StBlockClosure aBlock) {
		Jun3dTransformation originalTransformation = transformation;
		transformation = transformation.product_(aTransformation);
		double[] matrix = aTransformation.asArray();
		try {
			this.getOpenGLInterface().glPushMatrix();
			this.getOpenGLInterface().glMultMatrixd(matrix);

			aBlock.value();

		} finally {
			this.getOpenGLInterface().glPopMatrix();
			transformation = originalTransformation;
		}
	}

	/**
	 * Flush the rendering context.
	 * 
	 * @category user interaction
	 */
	public void flush() {
		this.getOpenGLInterface().glFlush();
	}

	/**
	 * Enable the depth test.
	 * 
	 * @category z testing
	 */
	public void enableDepthTest() {
		this.getOpenGLInterface().glEnable(JunOpenGLInterface.GL_DEPTH_TEST);
	}

	/**
	 * Disable the depth test.
	 * 
	 * @category z testing
	 */
	public void disableDepthTest() {
		this.getOpenGLInterface().glDisable(JunOpenGLInterface.GL_DEPTH_TEST);
	}

	/**
	 * Enable the depth mask.
	 * 
	 * @category z testing
	 */
	public void enableDepthMask() {
		this.getOpenGLInterface().glDepthMask(true);
	}

	/**
	 * Disable the depth mask.
	 * 
	 * @category z testing
	 */
	public void disableDepthMask() {
		this.getOpenGLInterface().glDepthMask(false);
	}

	/**
	 * Answer the normal vector of the points.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint normalVectorOfPoints_(Jun3dPoint[] anArrayOfPoints) {
		return NormalVectorOfPoints_(anArrayOfPoints);
	}

	/**
	 * Answer the normal unit vector of the points.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint normalUnitVectorOfPoints_(Jun3dPoint[] anArrayOfPoints) {
		return NormalUnitVectorOfPoints_(anArrayOfPoints);
	}

	/**
	 * Set the projection.
	 * 
	 * @param anOpenGLProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category private
	 */
	private void setProjection(JunOpenGLProjection anOpenGLProjection) {
		projection = anOpenGLProjection;
	}

	/**
	 * Answer the normal vector of the points.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint NormalVectorOfPoints_(Jun3dPoint[] anArrayOfPoints) {
		int size = anArrayOfPoints.length;
		if (size < 3) {
			return null;
		}

		int oneThirds = size / 3;
		Jun3dPoint point1 = (Jun3dPoint) anArrayOfPoints[0];
		Jun3dPoint point2 = (Jun3dPoint) ((Jun3dPoint) anArrayOfPoints[1]).plus_(oneThirds);
		Jun3dPoint point3 = (Jun3dPoint) ((Jun3dPoint) anArrayOfPoints[2]).plus_(oneThirds * 2);
		Jun3dPoint vector1 = new Jun3dPoint(point2.x() - point1.x(), point2.y() - point1.y(), point2.z() - point1.z());
		Jun3dPoint vector2 = new Jun3dPoint(point3.x() - point2.x(), point3.y() - point2.y(), point3.z() - point2.z());
		return vector1.product_(vector2);
	}

	/**
	 * Answer the normal unit vector of the points.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint NormalUnitVectorOfPoints_(Jun3dPoint[] anArrayOfPoints) {
		Jun3dPoint normalVector = NormalVectorOfPoints_(anArrayOfPoints);
		if (normalVector == null) {
			return null;
		}

		return normalVector.unitVector();
	}

	/**
	 * Answer the float array from the specified color and the alpha number.
	 * 
	 * @param aColor java.awt.Color
	 * @param alpha float
	 * @return float[]
	 * @category Utilities
	 */
	public static float[] GetRGBA(Color aColor, float alpha) {
		float[] rgba = aColor.getRGBComponents(null);
		rgba[3] = alpha;
		return rgba;
	}
}
