package jp.co.sra.jun.opengl.objects;

import java.awt.Color;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.Collection;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.curves.Jun3dPolyline;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunOpenGL3dVertexesObject class
 * 
 *  @author    nisinaka
 *  @created   1998/10/13 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2003/07/01 (by nisinaka)
 *  @updated   2003/09/09 (by nisinaka)
 *  @updated   2004/05/14 (by nisinaka)
 *  @updated   2007/08/29 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun682 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dVertexesObject.java,v 8.15 2008/02/20 06:32:35 nisinaka Exp $
 */
public abstract class JunOpenGL3dVertexesObject extends JunOpenGL3dPrimitiveObject {

	/** An array of Jun3dPoint which compose a JunOpenGL3dVertexesObject. */
	protected Jun3dPoint[] vertexes;

	/** An array of Color which is for Gouraud Shading. */
	protected Color[] colors;

	/** An array of Jun3dPoint which holds normal vectors of each vertexes. */
	protected Jun3dPoint[] normalVectors;

	/** A face normal vector. */
	protected Jun3dPoint preferredNormalVector;

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject() {
		super();
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints) {
		this();
		this.vertexes_(anArrayOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points and a Color.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Color aColor) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.paint_(aColor);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points and a Color.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Color aColor, float aNumber) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.paint_(aColor);
		this.alpha_(aNumber);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points and an array of colors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Color[] anArrayOfColors) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.colors_(anArrayOfColors);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points, an array of colors and alpha value.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Color[] anArrayOfColors, float aNumber) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.colors_(anArrayOfColors);
		this.alpha_(aNumber);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points, and an array of 3d vectors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.normalVectors_(anArrayOf3dVectors);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points, an array of 3d vectors and an array of colors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors, Color[] anArrayOfColors) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.normalVectors_(anArrayOf3dVectors);
		this.colors_(anArrayOfColors);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with an array of 3d points, an array of 3d vectors, an array of colors and alpha value.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors, Color[] anArrayOfColors, float aNumber) {
		this();
		this.vertexes_(anArrayOf3dPoints);
		this.normalVectors_(anArrayOf3dVectors);
		this.colors_(anArrayOfColors);
		this.alpha_(aNumber);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Collection aCollectionOf3dPoints) {
		this();
		this.vertexes_(aCollectionOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @param aColor java.awt.Color
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Collection aCollectionOf3dPoints, Color aColor) {
		this();
		this.vertexes_(aCollectionOf3dPoints);
		this.paint_(aColor);
	}

	/**
	 * Create a new JunOpenGL3dVertexesObject and initialize it with a collection of 3d points, a color and alpha value.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(Collection aCollectionOf3dPoints, Color aColor, float aNumber) {
		this();
		this.vertexes_(aCollectionOf3dPoints);
		this.paint_(aColor);
		this.alpha_(aNumber);
	}

	/**
	 * Create a new instance of JunOpenGL3dVertexesObject and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dVertexesObject(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Create a new JunOpenGL3dObject and initialize it with an array of 3d points.
	 * 
	 * @param aClass java.lang.Class
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	protected static JunOpenGL3dObject Vertexes_(Class aClass, Jun3dPoint[] anArrayOf3dPoints) {
		JunOpenGL3dVertexesObject a3dObject = (JunOpenGL3dVertexesObject) _New(aClass);
		a3dObject.vertexes_(anArrayOf3dPoints);
		return a3dObject;
	}

	/**
	 * Create a new JunOpenGL3dObject and initialize it with an array of 3d points.
	 * 
	 * @param aClass java.lang.Class
	 * @param aCollectionOf3dPoints java.util.Vector
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	protected static JunOpenGL3dObject Vertexes_(Class aClass, Collection aCollectionOf3dPoints) {
		JunOpenGL3dVertexesObject a3dObject = (JunOpenGL3dVertexesObject) _New(aClass);
		a3dObject.vertexes_(aCollectionOf3dPoints);
		return a3dObject;
	}

	/**
	 * Create a new JunOpenGL3dObject and initialize it with an array of 3d
	 * points and a Color.
	 * 
	 * @param aClass java.lang.Class
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	protected static JunOpenGL3dObject Vertexes_paint_(Class aClass, Jun3dPoint[] anArrayOf3dPoints, Color aColor) {
		JunOpenGL3dObject anObject = Vertexes_(aClass, anArrayOf3dPoints);
		anObject.paint_(aColor);
		return anObject;
	}

	/**
	 * Initialize the JunOpenGL3dVertexesObject.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		vertexes = null;
		colors = null;
		normalVectors = null;
	}

	/**
	 * Answer my vertexes.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] vertexes() {
		return vertexes;
	}

	/**
	 * Set the vertexes with an array of 3d points.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public void vertexes_(Jun3dPoint[] anArrayOf3dPoints) {
		int size = anArrayOf3dPoints.length;
		vertexes = new Jun3dPoint[size];
		System.arraycopy(anArrayOf3dPoints, 0, vertexes, 0, size);
		this.flushBounds();
		this.flushNormalVectors();
	}

	/**
	 * Set the vertexes with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @category accessing
	 */
	public void vertexes_(Collection aCollectionOf3dPoints) {
		vertexes = new Jun3dPoint[aCollectionOf3dPoints.size()];
		aCollectionOf3dPoints.toArray(vertexes);
		this.flushBounds();
		this.flushNormalVectors();
	}

	/**
	 * Answer the receiver's average point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint averagePoint() {
		final StValueHolder totalPointHolder = new StValueHolder(new Jun3dPoint(0, 0, 0));
		this.pointsDo_(new StBlockClosure() {
			public Object value_(Object eachPoint) {
				Jun3dPoint totalPoint = (Jun3dPoint) totalPointHolder.value();
				totalPoint = totalPoint.plus_((Jun3dPoint) eachPoint);
				totalPointHolder.value_(totalPoint);
				return null;
			}
		});
		Jun3dPoint totalPoint = (Jun3dPoint) totalPointHolder.value();
		Jun3dPoint averagePoint = totalPoint.dividedBy_(this.numberOfVertexes());
		return averagePoint;
	}

	/**
	 * Answer my colors for Gouraud Shading.
	 * 
	 * @return java.awt.Color[]
	 * @category accessing
	 */
	public Color[] colors() {
		return colors;
	}

	/**
	 * Set my new colors for Gouraud Shading.
	 * 
	 * @param anArrayOfColors java.awt.Color[]
	 * @category accessing
	 */
	public void colors_(Color[] anArrayOfColors) {
		colors = anArrayOfColors;
	}

	/**
	 * Answer the number of the vertexes.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfVertexes() {
		return this.vertexes().length;
	}

	/**
	 * Answer the vertex at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint vertexAt(int index) {
		return this.vertexes()[index];
	}

	/**
	 * Answer the index of the specified vertex.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return int
	 * @category accessing
	 */
	public int indexOfVertex(Jun3dPoint aPoint) {
		Jun3dPoint[] vertexes = this.vertexes();
		for (int i = 0; i < vertexes.length; i++) {
			if (vertexes[i].equal_(aPoint)) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Answer the previous vertex of the specified vertex.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint previousVertexOf(Jun3dPoint aPoint) {
		int index = this.indexOfVertex(aPoint);
		if (index < 0) {
			return null;
		}

		index--;
		if (index < 0) {
			index = this.numberOfVertexes() - 1;
		}
		return this.vertexAt(index);
	}

	/**
	 * Answer the next vertex of the specified vertex.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint nextVertexOf(Jun3dPoint aPoint) {
		int index = this.indexOfVertex(aPoint);
		if (index < 0) {
			return null;
		}

		index++;
		if (index >= this.numberOfVertexes()) {
			index = 0;
		}
		return this.vertexAt(index);
	}

	/**
	 * Answer the current normal vectors.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category vector accessing
	 */
	public Jun3dPoint[] normalVectors() {
		if (normalVectors == null) {
			normalVectors = new Jun3dPoint[] { this.normalVector() };
		}
		return normalVectors;
	}

	/**
	 * Set the new normal vectors.
	 * 
	 * @param newNormalVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category vector accessing
	 */
	public void normalVectors_(Jun3dPoint[] newNormalVectors) {
		normalVectors = newNormalVectors;
	}

	/**
	 * Answer the current normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category vector accessing
	 */
	public Jun3dPoint normalVector() {
		if (preferredNormalVector == null) {
			preferredNormalVector = this.preferredNormalVector();
		}
		return preferredNormalVector;
	}

	/**
	 * Set the new normal vector.
	 * 
	 * @param newNormalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category vector accessing
	 */
	public void normalVector_(Jun3dPoint aPoint) {
		preferredNormalVector = aPoint;
	}

	/**
	 * Answer the normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category vector accessing
	 */
	public Jun3dPoint normalUnitVector() {
		return this.normalVector().normalUnitVector();
	}

	/**
	 * Establish all normal vectors with the specified smoothing angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#establishAllNormalVectorsWithSmoothingAngle_(jp.co.sra.jun.geometry.basic.JunAngle)
	 * @category vector accessing
	 */
	public void establishAllNormalVectorsWithSmoothingAngle_(JunAngle anAngle) {
		Jun3dPoint[] vectors = new Jun3dPoint[this.numberOfVertexes()];
		for (int i = 0; i < vectors.length; i++) {
			vectors[i] = this.normalVector();
		}
		this.normalVectors_(vectors);
	}

	/**
	 * Flush my colors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushColors()
	 * @category flushing
	 */
	public void flushColors() {
		colors = null;
	}

	/**
	 * Flush my normal vectors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushNormalVectors()
	 * @category flushing
	 */
	public void flushNormalVectors() {
		normalVectors = null;
	}

	/**
	 * Answer true if the receiver is a vertexes object.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#isVertexesObject()
	 * @category testing
	 */
	public boolean isVertexesObject() {
		return true;
	}

	/**
	 * Answer true if the receiver has colors.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasColors()
	 * @category testing
	 */
	public boolean hasColors() {
		return (colors != null);
	}

	/**
	 * Answer true if the receiver has normal vectors.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasNormalVectors()
	 * @category testing
	 */
	public boolean hasNormalVectors() {
		return (normalVectors != null);
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		if (vertexes != null) {
			Jun3dPoint[] newVertexes = new Jun3dPoint[vertexes.length];
			System.arraycopy(vertexes, 0, newVertexes, 0, vertexes.length);
			vertexes = newVertexes;
		}

		if (colors != null) {
			Color[] newColors = new Color[colors.length];
			System.arraycopy(colors, 0, newColors, 0, colors.length);
			colors = newColors;
		}

		if (normalVectors != null) {
			Jun3dPoint[] newNormalVectors = new Jun3dPoint[normalVectors.length];
			System.arraycopy(normalVectors, 0, newNormalVectors, 0, normalVectors.length);
			normalVectors = newNormalVectors;
		}

		if (preferredNormalVector != null) {
			preferredNormalVector = new Jun3dPoint(preferredNormalVector);
		}

		return this;
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		for (int i = 0; i < vertexes.length; i++) {
			aBlock.value_(vertexes[i]);
		}
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		Jun3dPoint[] points = this.vertexes();
		Jun3dPoint[] collection = new Jun3dPoint[points.length];
		for (int i = 0; i < collection.length; i++) {
			collection[i] = new Jun3dPoint(points[i]);
		}
		Jun3dPolyline polyline = new Jun3dPolyline(collection);
		return polyline.asArrayOfLines();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asArrayOfPoints() {
		return this.vertexes();
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dVertexesObject transformedObject = (JunOpenGL3dVertexesObject) this.copy();

		transformedObject.vertexes = new Jun3dPoint[vertexes.length];
		for (int i = 0; i < vertexes.length; i++) {
			transformedObject.vertexes[i] = (Jun3dPoint) vertexes[i].transform_(aTransformation);
		}

		if (normalVectors != null) {
			transformedObject.normalVectors = new Jun3dPoint[normalVectors.length];
			if (normalVectors.length == 1) {
				Jun3dPoint averagePoint = this.averageVertex();
				Jun3dLine aVector = new Jun3dLine(averagePoint, averagePoint.plus_(normalVectors[0]));
				transformedObject.normalVectors[0] = aVector.transform_(aTransformation).asUnitVector();
			} else {
				for (int i = 0; i < normalVectors.length; i++) {
					Jun3dPoint endPoint = vertexes[i].plus_(normalVectors[i]).transform_(aTransformation);
					transformedObject.normalVectors[i] = endPoint.minus_(transformedObject.vertexes[i]).unitVector();
				}
			}
		}

		return transformedObject;
	}

	/**
	 * Convert the receiver as a triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#asTriangle()
	 * @category converting
	 */
	public JunPlane asPlane() {
		Jun3dPoint[] points = this.vertexes();
		if (points == null || points.length < 3) {
			return null;
		}
		int oneThirds = points.length / 3;
		Jun3dPoint point1 = points[0];
		Jun3dPoint point2 = points[0].plus_(oneThirds);
		Jun3dPoint point3 = points[0].plus_(oneThirds * 2);
		return new JunPlane(point1, point2, point3);
	}

	/**
	 * Convert the receiver as a triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#asTriangle()
	 * @category converting
	 */
	public Jun3dTriangle asTriangle() {
		Jun3dPoint[] points = this.vertexes();
		if (points == null || points.length != 3) {
			return null;
		}

		return Jun3dTriangle.On_on_on_(points[0], points[1], points[2]);
	}

	/**
	 * Answer the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#reversed()
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		JunOpenGL3dVertexesObject reversedObject = (JunOpenGL3dVertexesObject) this.copy();

		if (vertexes != null) {
			reversedObject.vertexes = new Jun3dPoint[vertexes.length];
			for (int i = 0; i < vertexes.length; i++) {
				reversedObject.vertexes[i] = vertexes[vertexes.length - i - 1];
			}
		}

		if (colors != null) {
			reversedObject.colors = new Color[colors.length];
			for (int i = 0; i < colors.length; i++) {
				reversedObject.colors[i] = colors[colors.length - i - 1];
			}
		}

		if (normalVectors != null) {
			reversedObject.normalVectors = new Jun3dPoint[normalVectors.length];
			for (int i = 0; i < normalVectors.length; i++) {
				reversedObject.normalVectors[i] = normalVectors[normalVectors.length - i - 1];
			}
		}

		return reversedObject;
	}

	/**
	 * Answer the normal vector object of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject normalVectorObject() {
		Jun3dPoint startPoint = this.averageVertex();
		JunOpenGL3dObject normalVectorObject = JunOpenGL3dObject.ConeFrom_to_width_(startPoint, startPoint.plus_(this.normalVector()), 0.15);
		normalVectorObject.flushAllPaints();
		normalVectorObject.flushAllAlphas();
		return normalVectorObject;
	}

	/**
	 * Answer the normal vector object of the receiver which is applied the transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject normalVectorObjectAppliedTransformation_(Jun3dTransformation aTransformation) {
		Jun3dPoint startPoint = this.transform_(aTransformation).averageVertex();

		boolean aBoolean = this.hasNormalVectors();
		if (aBoolean == false) {
			this.establishAllNormalVectors();
		}
		JunOpenGL3dVertexesObject aPolygon;
		try {
			aPolygon = (JunOpenGL3dVertexesObject) this.transform_(aTransformation);
		} finally {
			if (aBoolean == false) {
				this.flushNormalVectors();
			}
		}
		Jun3dPoint normalVector = aPolygon.normalVector();

		JunOpenGL3dObject normalVectorObject = JunOpenGL3dObject.ConeFrom_to_width_(startPoint, startPoint.plus_(normalVector), 0.15);
		normalVectorObject.flushAllPaints();
		normalVectorObject.flushAllAlphas();
		return normalVectorObject;
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (!this.hasName() && !this.hasColor() && !this.hasStipple() && !this.hasTexture() && !this.hasColors() && !this.hasNormalVectors()) {
			for (int i = 0; i < vertexes.length; i++) {
				list.add_(vertexes[i]);
			}
			return list;
		}

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasStipple()) {
			list.add_(this.stippleToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.vertexesToLispList());
		if (this.hasColors()) {
			list.add_(this.colorsToLispList());
		}
		if (this.hasNormalVectors()) {
			list.add_(this.normalVectorsToLispList());
		}

		return list;
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected abstract JunLispList stippleToLispList();

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected abstract JunLispList textureToLispList();

	/**
	 * Convert the receiver's vertexes as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList vertexesToLispList() {
		Object[] collection = new Object[vertexes.length + 1];
		collection[0] = $("points");
		for (int i = 0; i < vertexes.length; i++) {
			collection[i + 1] = vertexes[i];
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's colors as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList colorsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("colors"));
		if (colors != null) {
			for (int i = 0; i < colors.length; i++) {
				list.add_(JunLispCons.List_(new float[] { colors[i].getRed() / 255f, colors[i].getGreen() / 255f, colors[i].getBlue() / 255f }));
			}
		}
		return list;
	}

	/**
	 * Convert the receiver's normal vectors as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList normalVectorsToLispList() {
		Object[] collection = new Object[normalVectors.length + 1];
		collection[0] = $("normalVectors");
		if (normalVectors != null) {
			for (int i = 0; i < normalVectors.length; i++) {
				collection[i + 1] = normalVectors[i];
			}
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.stippleFromLispList(aList);
		this.textureFromLispList(aList);
		this.vertexesFromLispList(aList);
		this.colorsFromLispList(aList);
		this.normalVectorsFromLispList(aList);
	}

	/**
	 * Get my stipple from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected abstract void stippleFromLispList(JunLispList aList);

	/**
	 * Get my texture from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected abstract void textureFromLispList(JunLispList aList);

	/**
	 * Get my vertexes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void vertexesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("points")));
			}
		}, new StBlockClosure());

		if (list == null) {
			this.vertexes_(((JunLispList) aList.tail()).toVector());
		} else {
			this.vertexes_(((JunLispList) list.tail()).toVector());
		}
	}

	/**
	 * Get my colors from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void colorsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("colors")));
			}
		}, new StBlockClosure());

		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Color[] colors = new Color[anArray.length];
		for (int i = 0; i < colors.length; i++) {
			JunLispCons rgbList = (JunLispCons) anArray[i];
			float red = ((Number) rgbList.nth_(1)).floatValue();
			float green = ((Number) rgbList.nth_(2)).floatValue();
			float blue = ((Number) rgbList.nth_(3)).floatValue();
			colors[i] = new Color(red, green, blue);
		}
		this.colors_(colors);
	}

	/**
	 * Get my normalVectors from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void normalVectorsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("normalVectors")));
			}
		}, new StBlockClosure());

		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Jun3dPoint[] normalVectors = new Jun3dPoint[anArray.length];
		System.arraycopy(anArray, 0, normalVectors, 0, normalVectors.length);
		this.normalVectors_(normalVectors);
	}

	/**
	 * Write my VRML1.0 string on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException DOCUMENT ME!
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			bw.write("DEF ");
			bw.write(this.legalName());
			bw.newLine();
			bw.write("Separator {");
			bw.newLine();
			this.vrml10ColorOn_(bw);
			this.vrml10VertexesOn_(bw);
			bw.write("} #Separator");
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write my vertexes as VRML1.0 on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category vrml support
	 */
	protected void vrml10VertexesOn_(Writer aWriter) {
		Jun3dPoint[] points = this.vrmlVertexes();
		if (points.length == 0) {
			return;
		}

		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			if (this.hasTexture()) {
				bw.write("\t");
				bw.write("Texture2");
				bw.newLine();
				bw.write("\t\t");
				bw.write("image ");
				this.texture().vrmlSFImageOn_(bw);
				bw.newLine();
				bw.write("\t");
				bw.write("} #Texture2");
				bw.newLine();
			}
			bw.write("\t");
			bw.write("Coordinate3 {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("point [");
			bw.newLine();
			for (int index = 0; index < points.length; index++) {
				Jun3dPoint point = points[index];
				bw.write("\t\t\t");
				bw.write(String.valueOf((float) point.x()));
				bw.write(" ");
				bw.write(String.valueOf((float) point.y()));
				bw.write(" ");
				bw.write(String.valueOf((float) point.z()));
				if (index != (points.length - 1)) {
					bw.write(",");
				}
				bw.write(" #");
				bw.write(String.valueOf(index));
				bw.newLine();
			}
			bw.write("\t\t");
			bw.write("] #point");
			bw.newLine();
			bw.write("\t");
			bw.write("} #Coordinate3");
			bw.newLine();
			if (this.hasTexture() && (this.texture().hasCoordinates())) {
				bw.write("\t");
				bw.write("TextureCoordinate2 {");
				bw.newLine();
				bw.write("\t\t");
				bw.write("point [");
				bw.newLine();
				Jun2dPoint[] coordinates = this.texture().coordinates();
				for (int index = 0; index < coordinates.length; index++) {
					Jun2dPoint point = coordinates[index];
					bw.write("\t\t\t");
					bw.write(String.valueOf((float) point.x()));
					bw.write(" ");
					bw.write(String.valueOf((float) point.y()));
					if (index != (coordinates.length - 1)) {
						bw.write(",");
					}
					bw.write(" #");
					bw.write(String.valueOf(index));
					bw.newLine();
				}
				bw.write("\t\t");
				bw.write("]");
				bw.newLine();
				bw.write("\t");
				bw.write("} #TextureCoordinate2");
				bw.newLine();
			}
			bw.write("\t");
			bw.write(this.vrml10IndexedSetName());
			bw.write(" {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("coordIndex [");
			bw.newLine();
			bw.write("\t\t\t");
			for (int index = 0; index < points.length; index++) {
				bw.write(String.valueOf(index));
				bw.write(", ");
			}
			bw.write("-1");
			bw.newLine();
			bw.write("\t\t");
			bw.write("] #coordIndex");
			bw.newLine();
			if (this.hasTexture()) {
				bw.write("\t");
				bw.write("textureCoordIndex [");
				bw.newLine();
				bw.write("\t\t");
				for (int index = 0; index < points.length; index++) {
					bw.write(String.valueOf(index));
					bw.write(", ");
				}
				bw.write("-1");
				bw.newLine();
				bw.write("\t");
				bw.write("] #textureCoordIndex");
			}
			bw.write("\t");
			bw.write("} #");
			bw.write(this.vrml10IndexedSetName());
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write my material as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20MaterialOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20MaterialOn_(PrintWriter pw, String leader) {
		pw.println(leader + "material Material {");

		pw.print(leader + INDENT + "emissiveColor ");
		Color color = this.paint();
		pw.print(color.getRed() / 255f);
		pw.print(' ');
		pw.print(color.getGreen() / 255f);
		pw.print(' ');
		pw.print(color.getBlue() / 255f);
		pw.println();

		pw.println(leader + "} #Material");
	}

	/**
	 * Write my geometry as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		if (this.vrmlVertexes().length == 0) {
			return;
		}

		pw.println(leader + "geometry " + this.vrml20IndexedSetName() + " {");

		this.vrml20CoordOn_(pw, leader + INDENT);
		this.vrml20CoordIndexOn_(pw, leader + INDENT);

		if (this.hasNormalVectors()) {
			this.vrml20NormalOn_(pw, leader + INDENT);
		}

		if (this.hasTexture() && this.texture().hasCoordinates()) {
			this.vrml20TexCoordOn_(pw, leader + INDENT);
			this.vrml20TexCoordIndexOn_(pw, leader + INDENT);
		}

		pw.println(leader + "} #" + this.vrml20IndexedSetName());
	}

	/**
	 * Write my coordinate as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20CoordOn_(PrintWriter pw, String leader) {
		pw.println(leader + "coord Coordinate {");
		pw.println(leader + INDENT + "point [");

		Jun3dPoint[] points = this.vrmlVertexes();
		for (int i = 0; i < points.length; i++) {
			pw.print(leader + INDENT + INDENT);
			pw.print((float) points[i].x());
			pw.print(' ');
			pw.print((float) points[i].y());
			pw.print(' ');
			pw.print((float) points[i].z());
			if (i < points.length - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #point");
		pw.println(leader + "} #Coordinate");
	}

	/**
	 * Write my coordinate index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20CoordIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "coordIndex [");

		pw.print(leader + INDENT);
		Jun3dPoint[] points = this.vrmlVertexes();
		for (int i = 0; i < points.length; i++) {
			pw.print(i);
			pw.print(", ");
		}
		pw.println(-1);

		pw.println(leader + "] #coordIndex");
	}

	/**
	 * Write my normal vectors as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20NormalOn_(PrintWriter pw, String leader) {
		pw.println(leader + "normal Normal {");
		pw.println(leader + INDENT + "vector [");

		Jun3dPoint[] points = this.normalVectors();
		for (int i = 0; i < points.length; i++) {
			pw.print(leader + INDENT + INDENT);
			pw.print((float) points[i].x());
			pw.print(' ');
			pw.print((float) points[i].y());
			pw.print(' ');
			pw.print((float) points[i].z());
			if (i < points.length - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #vector");
		pw.println(leader + "} #Normal");
	}

	/**
	 * Write my texture coordinate as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20TexCoordOn_(PrintWriter pw, String leader) {
		pw.println(leader + "texCoord TextureCoordinate {");
		pw.println(leader + INDENT + "point [");

		Jun2dPoint[] coordinates = this.texture().coordinates();
		for (int i = 0; i < coordinates.length; i++) {
			pw.print(leader + INDENT + INDENT);
			pw.print((float) coordinates[i].x());
			pw.print(' ');
			pw.print((float) coordinates[i].y());
			if (i < coordinates.length - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #point");
		pw.println(leader + "} #TextureCoordinate");
	}

	/**
	 * Write my texture coordinate index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20TexCoordIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "texCoordIndex [");

		pw.print(leader + INDENT);
		Jun3dPoint[] points = this.vrmlVertexes();
		for (int i = 0; i < points.length; i++) {
			pw.print(i);
			pw.print(", ");
		}
		pw.println(-1);

		pw.println(leader + "] #texCoordIndex");
	}

	/**
	 * Answer the indexed set name for VRML1.0.
	 * 
	 * @return String java.lang.String
	 * @category vrml support
	 */
	protected abstract String vrml10IndexedSetName();

	/**
	 * Answer the indexed set name for VRML2.0.
	 * 
	 * @return String java.lang.String
	 * @category vrml support
	 */
	protected abstract String vrml20IndexedSetName();

	/**
	 * Answer the vertexes for VRML.
	 * 
	 * @return Jun3dPoint[] jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category vrml support
	 */
	protected Jun3dPoint[] vrmlVertexes() {
		return this.vertexes();
	}

	/**
	 * Calculate the preferred normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint preferredNormalVector() {
		StBlockClosure prevPointBlock = new StBlockClosure() {
			public Object value_(Object argument) {
				int startIndex = ((Integer) argument).intValue();
				Jun3dPoint startPoint = vertexes[startIndex];
				int index = startIndex - 1;
				int length = vertexes.length;
				while ((startIndex + index + length) > 0) {
					int i = (index % length + length) % length;
					if (startPoint.distance_(vertexes[i]) >= Jun3dPoint.ACCURACY) {
						return vertexes[i];
					}
					index--;
				}
				return null;
			}
		};

		StBlockClosure nextPointBlock = new StBlockClosure() {
			public Object value_(Object argument) {
				int startIndex = ((Integer) argument).intValue();
				Jun3dPoint startPoint = vertexes[startIndex];
				int index = startIndex + 1;
				int length = vertexes.length;
				while ((startIndex - index + length) > 0) {
					int i = (index % length + length) % length;
					if (startPoint.distance_(vertexes[i]) >= Jun3dPoint.ACCURACY) {
						return vertexes[i];
					}
					index++;
				}
				return null;
			}
		};

		Jun3dPoint vector = Jun3dPoint.Zero();
		for (int i = 0; i < vertexes.length; i++) {
			Jun3dPoint prevPoint = (Jun3dPoint) prevPointBlock.value_(new Integer(i));
			Jun3dPoint nextPoint = (Jun3dPoint) nextPointBlock.value_(new Integer(i));
			if ((prevPoint != null) && (nextPoint != null)) {
				Jun3dPoint thePoint = vertexes[i];
				vector = (Jun3dPoint) vector.plus_(((Jun3dPoint) thePoint.minus_(prevPoint)).vectorProduct_((Jun3dPoint) nextPoint.minus_(thePoint)));
			}
		}

		return (Jun3dPoint) vector.unitVector();
	}

}
