package jp.co.sra.jun.opengl.objects;

import java.io.PrintWriter;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunOpenGL3dCone class
 * 
 *  @author    nisinaka
 *  @created   2004/06/01 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dCone.java,v 8.13 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dCone extends JunOpenGL3dCompoundObject {

	protected static final Jun3dPoint DefaultApex = new Jun3dPoint(0, 0, 2);
	protected static final Jun3dPoint DefaultBottomCenter = new Jun3dPoint(0, 0, 0);
	protected static final double DefaultBottomRadius = 1;
	protected static final int DefaultSlices = 20;

	protected Jun3dPoint apex;
	protected Jun3dPoint bottomCenter;
	protected double bottomRadius;
	protected int slices;

	/**
	 * Create a new instance of JunOpenGL3dCone.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dCone() {
		super();
		this.flushComponents();
	}

	/**
	 * Create a new instance of JunOpenGL3dCone and initialize it.
	 * Mainly used for conversion from VRML.
	 * 
	 * @param bottomRadius double
	 * @param height double
	 * @category Instance creation
	 */
	public JunOpenGL3dCone(double bottomRadius, double height) {
		this(new Jun3dPoint(0, height / 2, 0), new Jun3dPoint(0, -height / 2, 0), bottomRadius, DefaultSlices);
	}

	/**
	 * Create a new instance of JunOpenGL3dCone and initialize it.
	 * 
	 * @param apex jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param bottomCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param bottomRadius double
	 * @param slices int
	 * @category Instance creation
	 */
	public JunOpenGL3dCone(Jun3dPoint apex, Jun3dPoint bottomCenter, double bottomRadius, int slices) {
		super();
		this.apex_(apex);
		this.bottomCenter_(bottomCenter);
		this.bottomRadius_(bottomRadius);
		this.slices_(slices);
	}

	/**
	 * Create a new instance of JunOpenGL3dCone and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dCone(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		apex = null;
		bottomCenter = null;
		bottomRadius = Double.NaN;
		slices = DefaultSlices;
	}

	/**
	 * Answer my current apex point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint apex() {
		if (apex == null) {
			apex = DefaultApex;
		}
		return apex;
	}

	/**
	 * Set my new apex point.
	 * 
	 * @param newApex jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void apex_(Jun3dPoint newApex) {
		apex = newApex;
		this.flushComponents();
	}

	/**
	 * Answer my current bottomCenter point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint bottomCenter() {
		if (bottomCenter == null) {
			bottomCenter = DefaultBottomCenter;
		}
		return bottomCenter;
	}

	/**
	 * Set my new bottomCenter point.
	 * 
	 * @param newBottomCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void bottomCenter_(Jun3dPoint newBottomCenter) {
		bottomCenter = newBottomCenter;
		this.flushComponents();
	}

	/**
	 * Answer my current bottomRadius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double bottomRadius() {
		if (Double.isNaN(bottomRadius)) {
			bottomRadius = DefaultBottomRadius;
		}
		return bottomRadius;
	}

	/**
	 * Set my new bottomRadius.
	 * 
	 * @param newRadius double
	 * @category accessing
	 */
	public void bottomRadius_(double newRadius) {
		bottomRadius = Math.max(newRadius, 0.001);
		this.flushComponents();
	}

	/**
	 * Answer my current number of slices.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int slices() {
		return slices;
	}

	/**
	 * Set my new number of slices.
	 * 
	 * @param newSlices int
	 * @category accessing
	 */
	public void slices_(int newSlices) {
		slices = Math.max(newSlices, 4);
		this.flushComponents();
	}

	/**
	 * Answer my current height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#height()
	 * @category accessing
	 */
	public double height() {
		return this.apex().distance_(this.bottomCenter());
	}

	/**
	 * Answer my current components as ArrayList.
	 * If not exist, create one.
	 * 
	 * @return java.util.ArrayList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#_components()
	 * @category accessing
	 */
	protected ArrayList _components() {
		if (components == null) {
			components = new ArrayList();
			this.flushBounds();

			Jun3dTransformation basicTransformation = this.basicTransformation();

			int slices = this.slices();
			Jun3dPoint[] points = new Jun3dPoint[slices];
			double dPhi = Math.PI * 2 / this.slices();
			for (int i = 0; i < slices; i++) {
				double phi = dPhi * i;
				double x = this.bottomRadius() * Math.sin(phi);
				if (Math.abs(x) < JunGeometry.ACCURACY) {
					x = 0.0d;
				}
				double y = this.bottomRadius() * Math.cos(phi);
				if (Math.abs(y) < JunGeometry.ACCURACY) {
					y = 0.0d;
				}
				points[i] = basicTransformation.applyTo_(new Jun3dPoint(x, y, 0));
			}

			JunOpenGL3dPolygon[] polygons = new JunOpenGL3dPolygon[slices];
			Jun3dPoint[] preferredNormalVectors = new Jun3dPoint[slices];
			for (int i = 0, prev = slices - 1; i < slices; prev = i, i++) {
				polygons[i] = new JunOpenGL3dPolygon(new Jun3dPoint[] { this.apex(), points[i], points[prev] });
				polygons[i].paint_(null);
				components.add(polygons[i]);
				preferredNormalVectors[i] = polygons[i].preferredNormalVector();
			}

			Jun3dPoint[] normalVectors = new Jun3dPoint[slices];
			for (int i = 0, prev = slices - 1; i < slices; prev = i, i++) {
				normalVectors[prev] = preferredNormalVectors[i].plus_(preferredNormalVectors[prev]).unitVector();
			}
			for (int i = 0, prev = slices - 1; i < slices; prev = i, i++) {
				polygons[i].normalVectors_(new Jun3dPoint[] { preferredNormalVectors[i], normalVectors[i], normalVectors[prev] });
			}

			JunOpenGL3dPolygon base = new JunOpenGL3dPolygon(points);
			base.paint_(null);
			base.normalVectors(); // to set its preferred normal vector.
			components.add(base);
		}
		return components;
	}

	/**
	 * Create a new JunOpenGL3dCone transformed with the transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming 
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dCone aCone = (JunOpenGL3dCone) this.copy();

		Jun3dPoint oldPoint = this.basicTransformation().applyTo_(new Jun3dPoint(this.bottomRadius(), 0, 0));
		Jun3dPoint newPoint = oldPoint.transform_(aTransformation);
		Jun3dPoint newBottomCenter = this.bottomCenter().transform_(aTransformation);
		double newBottomRadius = newBottomCenter.distance_(newPoint);

		aCone.apex_(this.apex().transform_(aTransformation));
		aCone.bottomCenter_(newBottomCenter);
		aCone.bottomRadius_(newBottomRadius);
		return aCone;
	}

	/**
	 * Answer the basic transformation which translate the basic cone to the receiver.
	 * Basic cone means a cone which bottom radius and height are the same but on the origin of X-Y plane.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	protected Jun3dTransformation basicTransformation() {
		Jun3dTransformation aTransformation;

		Jun3dPoint aPoint = this.apex().minus_(this.bottomCenter());
		if (aPoint.x() == 0 && aPoint.y() == 0 && aPoint.z() < 0) {
			aPoint = new Jun3dPoint(0, 0, -aPoint.z());
			aTransformation = Jun3dTransformation.MirrorY();
			aTransformation = aTransformation.product_(Jun3dTransformation.AlignVector_to_(new Jun3dPoint(0, 0, this.height()), aPoint));
		} else {
			aTransformation = Jun3dTransformation.AlignVector_to_(new Jun3dPoint(0, 0, this.height()), aPoint);
		}

		aTransformation = aTransformation.product_(Jun3dTransformation.Translate_(this.bottomCenter()));
		return aTransformation;
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		if (apex != null) {
			apex = new Jun3dPoint(apex.x(), apex.y(), apex.z());
		}

		if (bottomCenter != null) {
			bottomCenter = new Jun3dPoint(bottomCenter.x(), bottomCenter.y(), bottomCenter.z());
		}

		return this;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Cone");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}

		list.add_(this.apexToLispList());
		list.add_(this.bottomCenterToLispList());
		list.add_(this.bottomRadiusToLispList());
		list.add_(this.slicesToLispList());

		return list;
	}

	/**
	 * Convert the receiver's apex as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList apexToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("apex"));
		list.tail_(this.apex());
		return list;
	}

	/**
	 * Convert the receiver's bottom center as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList bottomCenterToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("bottomCenter"));
		list.tail_(this.bottomCenter());
		return list;
	}

	/**
	 * Convert the receiver's bottom radius as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList bottomRadiusToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("bottomRadius"));
		list.tail_(new Double(this.bottomRadius()));
		return list;
	}

	/**
	 * Convert the receiver's number of slices as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList slicesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("slices"));
		list.tail_(new Integer(this.slices()));
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.apexFromLispList(aList);
		this.bottomCenterFromLispList(aList);
		this.bottomRadiusFromLispList(aList);
		this.slicesFromLispList(aList);
	}

	/**
	 * Get my apex from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void apexFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("apex")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.apex_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my bottom center from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void bottomCenterFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("bottomCenter")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.bottomCenter_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my bottom radius from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void bottomRadiusFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("bottomRadius")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.bottomRadius_(((Number) list.tail()).doubleValue());
	}

	/**
	 * Get my slices from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void slicesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("slices")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.slices_(((Number) list.tail()).intValue());
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml20On_(PrintWriter pw, String leader) {
		Jun3dLine l = new Jun3dLine(this.apex(), this.bottomCenter());
		Jun3dPoint t = l.center();
		l = l.translatedBy_(t.negated());
		Jun3dLine y_axes = new Jun3dLine(new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0, 0));
		Jun3dPoint r = (new JunPlane(y_axes.to(), y_axes.from(), l.from())).normalVector();
		JunAngle a = y_axes.angleWithLine_(l);

		pw.println(leader + "Transform {");
		pw.println(leader + INDENT + "rotation " + r.x() + ' ' + r.y() + ' ' + r.z() + ' ' + a.rad());
		pw.println(leader + INDENT + "translation " + t.x() + ' ' + t.y() + ' ' + t.z());
		pw.println(leader + INDENT + "children [");
		this.vrml20ShapeOn_(pw, leader + INDENT + INDENT);
		pw.println(leader + INDENT + "] # children");
		pw.println(leader + "} #Transform");
		pw.flush();
	}

	/**
	 * Write my geometry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		pw.println(leader + "geometry Cone {");
		pw.println(leader + INDENT + "bottomRadius " + this.bottomRadius());
		pw.println(leader + INDENT + "height " + this.height());
		pw.println(leader + "} #Cone");
	}

}
