package jp.co.sra.jun.opengl.flux;

import java.util.ArrayList;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dTransformedObject;

/**
 * JunOpenGLFluxMutableObject class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/26 (by MATSUDA Ryouichi)
 *  @updated   2004/11/12 (by Mitsuhiro Asada)
 *  @updated   2006/10/12 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLFluxMutableObject.java,v 8.12 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGLFluxMutableObject extends JunOpenGLFluxAbstract {

	protected ArrayList transformationSequence;
	protected Jun3dBoundingBox boundingBox;

	/**
	 * Create a new instance of JunOpenGLFluxMutableObject and initialize it.
	 *
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.opengl.flux.JunOpenGLFluxMutableObject
	 * @category Instance creation
	 * @deprecated since Jun628, use the constructor.
	 */
	public static JunOpenGLFluxMutableObject Object_(JunOpenGL3dObject a3dObject) {
		return new JunOpenGLFluxMutableObject(a3dObject);
	}

	/**
	 * Create a new instance of JunOpenGLFluxMutableObject and initialize it.
	 *
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public JunOpenGLFluxMutableObject(JunOpenGL3dObject a3dObject) {
		super(a3dObject);
	}

	/**
	 * Create a new instance of JunOpenGLFluxMutableObject and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGLFluxMutableObject(JunLispList aList) {
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.flux.JunOpenGLFluxAbstract#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		transformationSequence = null;
		boundingBox = null;
	}

	/***
	 * Answer my current transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category accessing
	 */
	public Jun3dTransformation[] transformations() {
		return (Jun3dTransformation[]) this.transformationSequence().toArray(new Jun3dTransformation[this.transformationSequence().size()]);
	}

	/**
	 * Answer transformation sequence.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	protected ArrayList transformationSequence() {
		if (transformationSequence == null) {
			transformationSequence = new ArrayList();
		}
		return transformationSequence;
	}

	/**
	 * Answer the compound object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category accessing
	 */
	public JunOpenGL3dCompoundObject compoundObject() {
		if (this.isEmptyTransformation()) {
			return this.originalObject().asCompoundObject();
		}

		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		Jun3dTransformation[] transformations = this.transformations();
		for (int i = 0; i < transformations.length; i++) {
			compoundObject.add_(this.originalObject().transform_(transformations[i]));
		}
		return compoundObject;
	}

	/**
	 * Answer the object at normalized number.
	 * 
	 * @param normalizedNumber double
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject at_(double normalizedNumber) {
		if (this.originalObject() == null) {
			return null;
		}

		int index = (int) Math.round(normalizedNumber * (this.size() - 1)) + 1;
		if (this.isEmptyTransformation()) {
			if (this.originalObject() instanceof JunOpenGL3dCompoundObject) {
				return ((JunOpenGL3dCompoundObject) this.originalObject()).components()[index - 1];
			} else {
				return this.originalObject();
			}
		} else {
			if (0 < index && index <= this.size()) {
				Jun3dTransformation transformation = (Jun3dTransformation) this.transformationSequence().get(index - 1);
				return new JunOpenGL3dTransformedObject(this.originalObject(), transformation);
			} else {
				return this.originalObject();
			}
		}
	}

	/**
	 * Answer the number of all elements.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		if (this.originalObject() == null) {
			return 0;
		}

		if (this.isEmptyTransformation()) {
			if (this.originalObject() instanceof JunOpenGL3dCompoundObject) {
				return ((JunOpenGL3dCompoundObject) this.originalObject()).components().length;
			} else {
				return 1;
			}
		} else {
			return this.transformationSequence().size();
		}
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.flux.JunOpenGLFluxAbstract#boundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = this.preferredBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer my preferred bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.flux.JunOpenGLFluxAbstract#preferredBoundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox preferredBoundingBox() {
		if (this.isEmptyTransformation()) {
			return this.originalObject().boundingBox();
		}

		final double[] min = new double[] { Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY };
		final double[] max = new double[] { Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY };
		this.pointsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun3dPoint aPoint = (Jun3dPoint) anObject;
				min[0] = Math.min(min[0], aPoint.x());
				min[1] = Math.min(min[1], aPoint.y());
				min[2] = Math.min(min[2], aPoint.z());
				max[0] = Math.max(max[0], aPoint.x());
				max[1] = Math.max(max[1], aPoint.y());
				max[2] = Math.max(max[2], aPoint.z());
				return null;
			}
		});

		Jun3dPoint originPoint = new Jun3dPoint(min[0], min[1], min[2]);
		Jun3dPoint cornerPoint = new Jun3dPoint(max[0], max[1], max[2]);
		return Jun3dBoundingBox.Origin_corner_(originPoint, cornerPoint);
	}

	/**
	 * Flush my bounds.
	 * 
	 * @see jp.co.sra.jun.opengl.flux.JunOpenGLFluxAbstract#flushBounds()
	 * @category bounds accessing
	 */
	public void flushBounds() {
		boundingBox = null;
	}

	/**
	 * Add a transformation..
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category adding
	 */
	public void add_(Jun3dTransformation aTransformation) {
		this.transformationSequence().add(aTransformation);
	}

	/**
	 * Enumerate every points and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.flux.JunOpenGLFluxAbstract#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(final StBlockClosure aBlock) {
		if (this.originalObject() == null) {
			return;
		}

		if (this.isEmptyTransformation()) {
			this.originalObject().pointsDo_(aBlock);
		} else {
			Jun3dTransformation[] transformations = this.transformations();
			for (int i = 0; i < transformations.length; i++) {
				final Jun3dTransformation transformation = transformations[i];
				this.originalObject().pointsDo_(new StBlockClosure() {
					public Object value_(Object o) {
						Jun3dPoint point = (Jun3dPoint) o;
						aBlock.value_(point.transform_(transformation));
						return null;
					}
				});
			}
		}
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		throw SmalltalkException.Error("Please use \"isEmptyTransformation\" instead of \"isEmpty\"");
	}

	/**
	 * Answer true if the receiver is empty transformation, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmptyTransformation() {
		return this.transformationSequence().size() < 1;
	}

	/**
	 * Show the receiver.
	 * 
	 * @category utilities
	 */
	public void show() {
		JunOpenGLFluxObject fluxObject = new JunOpenGLFluxObject();
		fluxObject.addMutable_(this);
		fluxObject.show();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.objectFromLispList(aList);
		this.transformationsFromLispList(aList);
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(this.objectToLispList());
		list.add_(this.transformationsToLispList());
		return list;
	}

	/**
	 * Get my object from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void objectFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("object")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.originalObject_(JunOpenGL3dCompoundObject.ObjectFromLispList_((JunLispCons) list.tail()));
	}

	/**
	 * Convert the object as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons objectToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("object"));

		if (this.originalObject() != null) {
			list.tail_(this.originalObject().toLispList());
		}
		return list;
	}

	/**
	 * Get my transformations from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void transformationsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("transformations")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		((JunLispList) list.tail()).do_(new StBlockClosure() {
			public Object value_(Object object) {
				JunLispCons each = (JunLispCons) object;
				JunLispCons cons = JunLispCons.Cell();
				cons.head_(each);

				Jun3dTransformation transformation = JunOpenGL3dObject.TransformationFromLispList_(cons);
				add_(transformation);
				return null;
			}
		});
	}

	/**
	 * Convert the transformations as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispList transformationsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("transformations"));

		Jun3dTransformation[] transformations = this.transformations();
		for (int i = 0; i < transformations.length; i++) {
			list.add_(this.transformationToLispList_(transformations[i]));
		}
		return list;
	}

}
