package jp.co.sra.jun.opengl.flux;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;

/**
 * JunCoordinateAngles class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/12/02 (by MATSUDA Ryouichi)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCoordinateAngles.java,v 8.13 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunCoordinateAngles extends JunGeometry {

	protected JunAngle omegaAngle;
	protected JunAngle phiAngle;
	protected JunAngle thetaAngle;

	/**
	 * Create a new instance of <code>JunCoordinateAngles</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunCoordinateAngles() {
		super();
	}

	/**
	 * Create a new instance of <code>JunCoordinateAngles</code> with omega, phi and theta.
	 * 
	 * @param omegaAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param phiAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param thetaAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles
	 * @category Instance creation
	 */
	public static JunCoordinateAngles Omega_phi_theta_(JunAngle omegaAngle, JunAngle phiAngle, JunAngle thetaAngle) {
		JunCoordinateAngles eulerAngles;

		eulerAngles = new JunCoordinateAngles();
		eulerAngles.omega_(omegaAngle);
		eulerAngles.phi_(phiAngle);
		eulerAngles.theta_(thetaAngle);

		return eulerAngles;
	}

	/**
	 * Create a new instance of <code>JunCoordinateAngles</code> with three points.
	 * 
	 * @param pointA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointC jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles
	 * @category Instance creation
	 */
	public static JunCoordinateAngles P1_p2_p3_(Jun3dPoint pointA, Jun3dPoint pointB, Jun3dPoint pointC) {
		JunCoordinateAngles instance;
		instance = new JunCoordinateAngles();
		instance.p1_p2_p3_(pointA, pointB, pointC);
		return instance;
	}

	/**
	 * Create a new instance of <code>JunCoordinateAngles</code> with four points.
	 * 
	 * @param pointA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointC jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointD jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles
	 * @category Instance creation
	 */
	public static JunCoordinateAngles P1_p2_p3_p4_(Jun3dPoint pointA, Jun3dPoint pointB, Jun3dPoint pointC, Jun3dPoint pointD) {
		JunCoordinateAngles instance;
		instance = new JunCoordinateAngles();
		instance.p1_p2_p3_p4_(pointA, pointB, pointC, pointD);
		return instance;
	}

	/**
	 * Create a new instance of <code>JunCoordinateAngles</code> with two vectors.
	 * 
	 * @param vectorA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param vectorB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles
	 * @category Instance creation
	 */
	public static JunCoordinateAngles V1_v2_(Jun3dPoint vectorA, Jun3dPoint vectorB) {
		JunCoordinateAngles instance;
		instance = new JunCoordinateAngles();
		instance.v1_v2_(vectorA, vectorB);
		return instance;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.omega_(0);
		this.phi_(0);
		this.theta_(0);
	}

	/**
	 * Answer omega angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final JunAngle omega() {
		return omegaAngle;
	}

	/**
	 * Set omega angle.
	 * 
	 * @param anAngle double
	 * @category accessing
	 */
	public void omega_(double anAngle) {
		omegaAngle = JunAngle.FromRad_(anAngle);
	}

	/**
	 * Set omega angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final void omega_(JunAngle anAngle) {
		omegaAngle = anAngle;
	}

	/**
	 * Compute angles from three points.
	 * 
	 * @param pointA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointC jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void p1_p2_p3_(Jun3dPoint pointA, Jun3dPoint pointB, Jun3dPoint pointC) {
		Jun3dPoint aVector;
		Jun3dPoint normalVector;
		aVector = (Jun3dPoint) pointB.minus_(pointA);
		normalVector = this.outerProduct_with_(aVector, (Jun3dPoint) pointC.minus_(pointA));
		normalVector = this.outerProduct_with_(normalVector, aVector);
		this.computeAnglesFrom_and_(aVector, normalVector);
	}

	/**
	 * Compute angles from four points.
	 * 
	 * @param pointA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointC jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointD jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void p1_p2_p3_p4_(Jun3dPoint pointA, Jun3dPoint pointB, Jun3dPoint pointC, Jun3dPoint pointD) {
		Jun3dPoint aVector;
		Jun3dPoint normalVector;

		aVector = (Jun3dPoint) pointB.minus_(pointA);
		normalVector = this.outerProduct_with_(aVector, (Jun3dPoint) pointD.minus_(pointC));
		this.computeAnglesFrom_and_(aVector, normalVector);
	}

	/**
	 * Answer phi angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final JunAngle phi() {
		return phiAngle;
	}

	/**
	 * Set phi angle.
	 * 
	 * @param anAngle double
	 * @category accessing
	 */
	public void phi_(double anAngle) {
		phiAngle = JunAngle.FromRad_(anAngle);
	}

	/**
	 * Set phi angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final void phi_(JunAngle anAngle) {
		phiAngle = anAngle;
	}

	/**
	 * Answer theta angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final JunAngle theta() {
		return thetaAngle;
	}

	/**
	 * Set theta angle.
	 * 
	 * @param anAngle double
	 * @category accessing
	 */
	public void theta_(double anAngle) {
		thetaAngle = JunAngle.FromRad_(anAngle);
	}

	/**
	 * Set theta angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public final void theta_(JunAngle anAngle) {
		thetaAngle = anAngle;
	}

	/**
	 * Compute angles from two vectors.
	 * 
	 * @param vectorA jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param vectorB jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void v1_v2_(Jun3dPoint vectorA, Jun3dPoint vectorB) {
		this.p1_p2_p3_(new Jun3dPoint(0, 0, 0), vectorA, vectorB);
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCoordinateAngles aCoordinateAngles = (JunCoordinateAngles) anObject;
		return this.omega().equal_(aCoordinateAngles.omega()) && this.phi().equal_(aCoordinateAngles.phi()) && this.theta().equal_(aCoordinateAngles.theta());
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCoordinateAngles aCoordinateAngles = (JunCoordinateAngles) anObject;
		return this.omega().equals(aCoordinateAngles.omega()) && this.phi().equals(aCoordinateAngles.phi()) && this.theta().equals(aCoordinateAngles.theta());
	}

	/**
	 * Answer inverse transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category converting
	 */
	public Jun3dTransformation asInverseTransformation() {
		Jun3dTransformation aTransformationAroundX = Jun3dTransformation.RotateX_(this.omega().negated());
		Jun3dTransformation aTransformationAroundY = Jun3dTransformation.RotateY_(this.phi().negated());
		Jun3dTransformation aTransformationAroundZ = Jun3dTransformation.RotateZ_(this.theta().negated());
		Jun3dTransformation aTransformation = Jun3dTransformation.Unity();
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundZ);
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundY);
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundX);
		return aTransformation;
	}

	/**
	 * Answer transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category converting
	 */
	public Jun3dTransformation asTransformation() {
		Jun3dTransformation aTransformationAroundX = Jun3dTransformation.RotateX_(this.omega());
		Jun3dTransformation aTransformationAroundY = Jun3dTransformation.RotateY_(this.phi());
		Jun3dTransformation aTransformationAroundZ = Jun3dTransformation.RotateZ_(this.theta());
		Jun3dTransformation aTransformation = Jun3dTransformation.Unity();
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundX);
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundY);
		aTransformation = (Jun3dTransformation) aTransformation.product_(aTransformationAroundZ);
		return aTransformation;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Coordinate Angles (");
		this.omega().printOn_(aWriter);
		aWriter.write(' ');
		this.phi().printOn_(aWriter);
		aWriter.write(' ');
		this.theta().printOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Compute angles from vector and normalVector.
	 * 
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void computeAnglesFrom_and_(Jun3dPoint aVector, Jun3dPoint normalVector) {
		Jun3dPoint vectorB;
		Jun3dPoint vectorN;
		double sinTheta;
		double thetaRadians;
		double sinPhi;
		double phiRadians;
		double sinOmega;
		double omegaRadians;
		Jun3dTransformation aTransformation;
		vectorB = aVector;
		vectorN = normalVector;

		double aLength = this.vectorLength_(new Jun3dPoint(vectorB.x(), vectorB.y(), 0));
		if (aLength < Accuracy()) {
			thetaRadians = 0;
		} else {
			sinTheta = vectorB.y() / aLength;
			thetaRadians = Math.acos(vectorB.x() / aLength);
			if (sinTheta > 0) {
				thetaRadians = thetaRadians * -1;
			}
		}
		aTransformation = Jun3dTransformation.RotateZ_(JunAngle.FromRad_(thetaRadians));
		vectorB = (Jun3dPoint) vectorB.transform_(aTransformation);
		vectorN = (Jun3dPoint) vectorN.transform_(aTransformation);
		aLength = this.vectorLength_(vectorB);
		sinPhi = vectorB.z() / aLength;
		phiRadians = Math.acos(this.vectorLength_(new Jun3dPoint(vectorB.x(), vectorB.y(), 0)) / aLength);
		if (sinPhi < 0) {
			phiRadians = phiRadians * -1;
		}
		aTransformation = Jun3dTransformation.RotateY_(JunAngle.FromRad_(phiRadians));
		vectorB = (Jun3dPoint) vectorB.transform_(aTransformation);
		vectorN = (Jun3dPoint) vectorN.transform_(aTransformation);
		aLength = this.vectorLength_(new Jun3dPoint(0, vectorN.y(), vectorN.z()));
		sinOmega = vectorN.y() / aLength;
		omegaRadians = Math.acos(vectorN.z() / aLength);
		if (sinOmega > 0) {
			omegaRadians = omegaRadians * -1;
		}
		thetaAngle = JunAngle.FromRad_(thetaRadians * -1);
		phiAngle = JunAngle.FromRad_(phiRadians * -1);
		omegaAngle = JunAngle.FromRad_(omegaRadians * -1);
	}

	/**
	 * Answer outer product vector with vector.
	 * 
	 * @param aVector1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aVector2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint outerProduct_with_(Jun3dPoint aVector1, Jun3dPoint aVector2) {
		return new Jun3dPoint((aVector1.y() * aVector2.z()) - (aVector1.z() * aVector2.y()), (aVector1.z() * aVector2.x()) - (aVector1.x() * aVector2.z()), (aVector1.x() * aVector2.y()) - (aVector1.y() * aVector2.x()));
	}

	/**
	 * Answer vector length.
	 * 
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category private
	 */
	protected double vectorLength_(Jun3dPoint aVector) {
		return Math.sqrt((aVector.x() * aVector.x()) + (aVector.y() * aVector.y()) + (aVector.z() * aVector.z()));
	}

}
