package jp.co.sra.jun.opengl.display;

import java.awt.Color;
import java.awt.Component;
import java.awt.Frame;
import java.awt.Point;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.surfaces.JunSphereSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.opengl.texture.JunOpenGLStipple;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSmallCompiler;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLShowModel class
 * 
 *  @author    He Weijie
 *  @created   1998/11/26 (by He Weijie)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2000/01/06 (by MATSUDA Ryouichi)
 *  @updated   2003/05/08 (by nisinaka)
 *  @updated   2007/08/24 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun642 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLShowModel.java,v 8.17 2008/02/20 06:32:19 nisinaka Exp $
 */
public class JunOpenGLShowModel extends JunOpenGLDisplayModel {

	protected JunOpenGLDisplayModel displayModel;
	protected Jun3dPoint viewportSight;
	protected double viewportHeight;
	protected boolean useTransparency;

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object and open it.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayModel Show_(JunOpenGL3dObject a3dObject) {
		return Show_eyePoint_(a3dObject, null);
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object and open it.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint) {
		return Show_eyePoint_sightPoint_(a3dObject, eyePoint, null);
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object and open it.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint) {
		return Show_eyePoint_sightPoint_upVector_(a3dObject, eyePoint, sightPoint, null);
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object and open it.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_upVector_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector) {
		return Show_eyePoint_sightPoint_upVector_viewFactor_(a3dObject, eyePoint, sightPoint, upVector, null);
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object and open it.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param viewFactor java.lang.Double
	 * @category Instance creation
	 */
	public static JunOpenGLShowModel Show_eyePoint_sightPoint_upVector_viewFactor_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, Double viewFactor) {
		JunOpenGLShowModel displayModel = new JunOpenGLShowModel(a3dObject);
		if (eyePoint != null) {
			displayModel.defaultEyePoint_(eyePoint);
		}
		if (sightPoint != null) {
			displayModel.defaultSightPoint_(sightPoint);
		}
		if (upVector != null) {
			displayModel.defaultUpVector_(upVector);
		}
		if (viewFactor != null) {
			displayModel.defaultViewFactor_(viewFactor.doubleValue());
		}
		displayModel.open();
		return displayModel;
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLShowModel() {
		super();
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified display model.
	 * 
	 * @param anOpenGLDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Instance creation
	 */
	public JunOpenGLShowModel(JunOpenGLDisplayModel anOpenGLDisplayModel) {
		this.displayModel_(anOpenGLDisplayModel);
	}

	/**
	 * Create a new instance of <code>JunOpenGLShowModel</code> with the specified 3D object.
	 *
	 * @param anOpenGL3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public JunOpenGLShowModel(JunOpenGL3dObject anOpenGL3dObject) {
		this.displayObject_(anOpenGL3dObject);
	}

	/**
	 * Initialize the JunOpenGLShowModel.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		displayModel = null;
		viewportSight = null;
		viewportHeight = Double.NaN;
		useTransparency = true;
	}

	/**
	 * Answer my current display model.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category accessing
	 */
	public JunOpenGLDisplayModel displayModel() {
		if (displayModel == null) {
			this.displayModel_(new JunOpenGLDisplayModel());
		}
		return displayModel;
	}

	/**
	 * Set my new display model.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category accessing
	 */
	public void displayModel_(JunOpenGLDisplayModel aDisplayModel) {
		if (displayModel != aDisplayModel) {
			displayModel = aDisplayModel;
			if (displayModel.myDependents() == null || displayModel.myDependents().contains(this) == false) {
				displayModel.addDependent_(this);
			}
		}
	}

	/**
	 * Answer my current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject displayObject() {
		return this.displayModel().displayObject();
	}

	/**
	 * Set my new display object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public void displayObject_(JunOpenGL3dObject a3dObject) {
		this.displayModel().displayObject_(a3dObject);
	}

	/**
	 * Answer the object to show.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject showObject() {
		JunOpenGLProjection aProjection = this.displayModel().displayProjection();
		double currentHeight = this.computeViewportHeight();
		Jun3dPoint currentSight = aProjection.sightPoint();
		double nearHeight = 2 * aProjection.sightPoint().distance_(aProjection.nearCenter());
		if (currentHeight > nearHeight) {
			currentHeight = nearHeight;
		}

		if (openGL3dObject == null || Double.isNaN(viewportHeight) || Math.abs(currentHeight - viewportHeight) > JunGeometry.ACCURACY || viewportSight == null || currentSight.distance_(viewportSight) > JunGeometry.ACCURACY) {
			viewportHeight = currentHeight;
			viewportSight = currentSight;

			// JunScavenger aScavenger = new JunScavenger();
			// if (showObject == null) {
			// 	aScavenger.run();
			// } else {
			// 	JunOpenGL3dObject aSphere;
			// 	if (this.displayObject() == null) {
			// 		aSphere = showObject.components().first();
			// 	} else {
			// 		aSphere = showObject.components().second();
			//	}
			// 	aSphere do :[: each | each stipple notNil ifTrue :[each stipple flushPointerOfPixels]].aScavenger.run();
			// }

			JunOpenGL3dObject aSphere = JunOpenGL3dObject.Globe_radius_center_(15, viewportHeight / 2, viewportSight);
			aSphere.paint_(Color.cyan);

			if (this.useTransparency()) {
				aSphere.alpha_(0.25f);
			} else {
				JunOpenGLStipple aStipple = JunOpenGLStipple.Halftone_(0.25);
				if (aSphere instanceof JunOpenGL3dCompoundObject) {
					JunOpenGL3dObject[] components = ((JunOpenGL3dCompoundObject) aSphere).components();
					for (int i = 0; i < components.length; i++) {
						components[i].stipple_(aStipple);
					}
				} else {
					aSphere.stipple_(aStipple);
				}
			}

			JunOpenGL3dObject xyzAxes = JunOpenGL3dObject.Axes2();
			xyzAxes = xyzAxes.scaledBy_(viewportHeight / 2);

			JunOpenGL3dCompoundObject aBody = new JunOpenGL3dCompoundObject();
			if (this.displayObject() != null) {
				aBody.add_(this.displayObject());
			}
			aBody.add_(aSphere);
			aBody.add_(xyzAxes);
			openGL3dObject = aBody;
		}

		return new JunOpenGL3dCompoundObject(openGL3dObject, this.viewportObject());
	}

	/**
	 * Answer the current projection of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public JunOpenGLProjection showProjection() {
		return this.showProjector().projection();
	}

	/**
	 * Set the new projection for the receiver.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public void showProjection_(JunOpenGLProjection aProjection) {
		this.showProjector().projection_(aProjection);
	}

	/**
	 * Answer the projector for the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	public JunOpenGLProjector showProjector() {
		if (openGLProjector == null) {
			openGLProjector = new JunOpenGLProjector();
			openGLProjector.eyePoint_(this.defaultEyePoint());
			openGLProjector.sightPoint_(this.defaultSightPoint());
			openGLProjector.upVector_(this.defaultUpVector());
			openGLProjector.viewFactor_(this.defaultViewFactor());
			openGLProjector.smoothShading();
		}
		return openGLProjector;
	}

	/**
	 * Answer true if I use the transparency.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean useTransparency() {
		return useTransparency;
	}

	/**
	 * Set a boolean whether I use the transparency or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void useTransparency_(boolean aBoolean) {
		if (useTransparency == aBoolean) {
			return;
		}

		useTransparency = aBoolean;
		this.flushObject();
		this.changed_($("object"));
	}

	/**
	 * Answer the viewport beam.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject viewportBeam() {
		JunOpenGLProjection aProjection = this.displayModel().displayProjection();
		Jun3dPoint far = aProjection.farCenter();
		Jun3dPoint sight = aProjection.sightPoint();
		Jun3dPoint near = aProjection.nearCenter();
		Jun3dPoint eye = aProjection.eyePoint();
		Jun3dPoint[] twoPoints = new Jun3dPoint[2];

		JunOpenGL3dCompoundObject eyeBeam = new JunOpenGL3dCompoundObject();
		JunOpenGL3dPolyline aPolyline;
		if (far.distance_(near) > sight.distance_(near)) {
			twoPoints[0] = far;
			twoPoints[1] = sight;
			aPolyline = new JunOpenGL3dPolyline(twoPoints);
			aPolyline.paint_(Color.magenta);
			eyeBeam.add_(aPolyline);
		}

		twoPoints[0] = sight;
		twoPoints[1] = near;
		aPolyline = new JunOpenGL3dPolyline(twoPoints);
		aPolyline.paint_(Color.blue);
		eyeBeam.add_(aPolyline);

		if (eye.distance_(sight) > near.distance_(sight)) {
			twoPoints[0] = near;
			twoPoints[1] = eye;
			aPolyline = new JunOpenGL3dPolyline(twoPoints);
			aPolyline.paint_(Color.yellow);
			eyeBeam.add_(aPolyline);
		}

		eyeBeam.name_("viewport eye beam");
		return eyeBeam;
	}

	/**
	 * Answer the viewport object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject viewportObject() {
		JunOpenGL3dCompoundObject aPlate = this.viewportPlate();
		JunOpenGLProjection aProjection = this.displayModel().displayProjection();

		if (viewportHeight == Double.NaN) {
			viewportHeight = this.computeViewportHeight();
		}

		double aRadius = viewportHeight / 2;
		double aDistance = aProjection.sightPoint().distance_(aProjection.eyePoint());
		Jun3dLine aLine = new Jun3dLine(aProjection.sightPoint(), aProjection.eyePoint());
		Jun3dPoint aPoint = (Jun3dPoint) aLine.atT_(aRadius / aDistance);
		aPlate = (JunOpenGL3dCompoundObject) aPlate.translatedBy_(aPoint);

		JunOpenGL3dObject[] components = aPlate.components();
		JunOpenGL3dPolylineLoop aPolyline = new JunOpenGL3dPolylineLoop(((JunOpenGL3dVertexesObject) components[components.length - 1]).vertexes());
		Jun3dPoint[] vertexes = aPolyline.vertexes();
		Jun3dPoint p1 = vertexes[0];
		Jun3dPoint p2 = vertexes[1];
		Jun3dPoint p3 = vertexes[2];
		Jun3dPoint p4 = vertexes[3];
		Jun3dLine xLine = new Jun3dLine((Jun3dPoint) p1.plus_(((Jun3dPoint) p4.minus_(p1)).dividedBy_(2)), (Jun3dPoint) p2.plus_(((Jun3dPoint) p3.minus_(p2)).dividedBy_(2)));
		Jun3dLine yLine = new Jun3dLine((Jun3dPoint) p1.plus_(((Jun3dPoint) p2.minus_(p1)).dividedBy_(2)), (Jun3dPoint) p4.plus_(((Jun3dPoint) p3.minus_(p4)).dividedBy_(2)));
		JunOpenGL3dCompoundObject xAxis = new JunOpenGL3dCompoundObject();
		Jun3dPoint[] twoPoints = new Jun3dPoint[2];
		Jun3dPoint[] fourPoints = new Jun3dPoint[4];
		twoPoints[0] = (Jun3dPoint) xLine.atT_(0);
		twoPoints[1] = (Jun3dPoint) xLine.atT_(1);
		xAxis.add_(new JunOpenGL3dPolyline(twoPoints));
		fourPoints[0] = (Jun3dPoint) xLine.atT_(1);
		fourPoints[1] = (Jun3dPoint) xLine.atT_(0.9).minus_(((Jun3dPoint) ((Jun3dPoint) yLine.atT_(1)).minus_(yLine.atT_(0.9))).dividedBy_(4));
		fourPoints[2] = (Jun3dPoint) xLine.atT_(0.9).plus_(((Jun3dPoint) ((Jun3dPoint) yLine.atT_(1)).minus_(yLine.atT_(0.9))).dividedBy_(4));
		fourPoints[3] = (Jun3dPoint) xLine.atT_(1);
		xAxis.add_(new JunOpenGL3dPolyline(fourPoints));
		xAxis.paint_(Color.red);
		xAxis.name_(aPlate.name() + " x axis");

		JunOpenGL3dCompoundObject yAxis = new JunOpenGL3dCompoundObject();
		twoPoints[0] = (Jun3dPoint) yLine.atT_(0);
		twoPoints[1] = (Jun3dPoint) yLine.atT_(1);
		yAxis.add_(new JunOpenGL3dPolyline(twoPoints));
		fourPoints[0] = (Jun3dPoint) yLine.atT_(1);
		fourPoints[1] = (Jun3dPoint) yLine.atT_(0.9).minus_(((Jun3dPoint) ((Jun3dPoint) xLine.atT_(1)).minus_(xLine.atT_(0.9))).dividedBy_(4));
		fourPoints[2] = (Jun3dPoint) yLine.atT_(0.9).plus_(((Jun3dPoint) ((Jun3dPoint) xLine.atT_(1)).minus_(xLine.atT_(0.9))).dividedBy_(4));
		fourPoints[3] = (Jun3dPoint) yLine.atT_(1);
		yAxis.add_(new JunOpenGL3dPolyline(fourPoints));
		yAxis.paint_(Color.green);
		yAxis.name_(aPlate.name() + " y axis");

		JunOpenGL3dCompoundObject aBody = new JunOpenGL3dCompoundObject();
		aBody.add_(aPlate);
		aBody.add_(aPolyline);
		aBody.add_(xAxis);
		aBody.add_(yAxis);
		aBody.name_("viewport");

		return aBody;
	}

	/**
	 * Answer the viewport plate.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category accessing
	 */
	public JunOpenGL3dCompoundObject viewportPlate() {
		JunOpenGLProjection aProjection = this.displayModel().displayProjection();

		if (Double.isNaN(viewportHeight)) {
			viewportHeight = this.computeViewportHeight();
		}

		double height = viewportHeight;
		double width;
		if (this.displayModel().builder().windows().length > 0) {
			Component aComponent = this.displayModel().getOpenGLView().toComponent();
			Rectangle aRectangle = aComponent.getBounds();
			width = height * aRectangle.width / aRectangle.height;
		} else {
			width = height;
		}

		Jun3dPoint p1 = new Jun3dPoint(0, 0, 0);
		Jun3dPoint p2 = (Jun3dPoint) aProjection.unitRightVector().scaledBy_(new Jun3dPoint(width, width, width));
		Jun3dPoint p4 = (Jun3dPoint) aProjection.unitUpVector().scaledBy_(new Jun3dPoint(height, height, height));
		Jun3dPoint p3 = (Jun3dPoint) p2.plus_(((Jun3dPoint) p4.minus_(p2)).dividedBy_(2));
		p3 = (Jun3dPoint) p3.plus_(p3);

		Jun3dPoint[] anArray = new Jun3dPoint[] { p1, p2, p3, p4 };
		JunOpenGL3dCompoundObject aPlate = new JunOpenGL3dCompoundObject();

		if (this.useTransparency()) {
			Jun3dPoint[] reversedArray = new Jun3dPoint[] { p4, p3, p2, p1 };
			aPlate.add_(new JunOpenGL3dPolygon(reversedArray));
		}

		aPlate.add_(new JunOpenGL3dPolygon(anArray));

		Jun3dPoint p5 = (Jun3dPoint) p2.plus_(((Jun3dPoint) p4.minus_(p2)).dividedBy_(2));
		Jun3dPoint p6 = (Jun3dPoint) (new Jun3dPoint(0, 0, 0)).minus_(p5);
		aPlate = (JunOpenGL3dCompoundObject) aPlate.translatedBy_(p6);
		aPlate.paint_(Color.yellow);

		if (this.useTransparency()) {
			aPlate.alpha_(0.2f);
		} else {
			aPlate.components()[0].stipple_(JunOpenGLStipple.Halftone_(0.125));
		}

		aPlate.name_("viewport plate");

		return aPlate;
	}

	/**
	 * Answer the JunButtonModel for the bull button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel bullButton() {
		if (this.pushButtons().containsKey($("bull")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.BullCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					if (model.value()) {
						JunOpenGLShowModel.this.dragButton().value_(false);
						JunOpenGLShowModel.this.grabButton().value_(false);
					}
					JunOpenGLShowModel.this.changed_($("state"));
					return model;
				}
			});

			this.pushButtons().put($("bull"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("bull"));
	}

	/**
	 * Answer the JunButtonModel for the drag button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel dragButton() {
		if (this.pushButtons().containsKey($("drag")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.QuartersCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					if (model.value()) {
						JunOpenGLShowModel.this.grabButton().value_(false);
						JunOpenGLShowModel.this.bullButton().value_(false);
					}
					JunOpenGLShowModel.this.changed_($("state"));
					return model;
				}
			});

			this.pushButtons().put($("drag"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("drag"));
	}

	/**
	 * Answer the JunButtonModel for the grab button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel grabButton() {
		if (this.pushButtons().containsKey($("grab")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.HandCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					if (model.value()) {
						JunOpenGLShowModel.this.dragButton().value_(false);
						JunOpenGLShowModel.this.bullButton().value_(false);
					}
					JunOpenGLShowModel.this.changed_($("state"));
					return model;
				}
			});

			this.pushButtons().put($("grab"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("grab"));
	}

	/**
	 * Answer the default eye point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultEyePoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultEyePoint() {
		return this.displayModel().defaultEyePoint();
	}

	/**
	 * Set the default eye point.
	 * 
	 * @param eyePoint jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultEyePoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultEyePoint_(Jun3dPoint eyePoint) {
		this.displayModel().defaultEyePoint_(eyePoint);
	}

	/**
	 * Answer the default sight point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultSightPoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultSightPoint() {
		return this.displayModel().defaultSightPoint();
	}

	/**
	 * Set the default sight point.
	 * 
	 * @param sightPoint jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultSightPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultSightPoint_(Jun3dPoint sightPoint) {
		this.displayModel().defaultSightPoint_(sightPoint);
	}

	/**
	 * Answer the default up vector.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultUpVector()
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		return this.displayModel().defaultUpVector();
	}

	/**
	 * Set the default up vector.
	 * 
	 * @param upVector jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultUpVector_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultUpVector_(Jun3dPoint upVector) {
		this.displayModel().defaultUpVector_(upVector);
	}

	/**
	 * Answer the default view factor.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultViewFactor()
	 * @category defaults
	 */
	public double defaultViewFactor() {
		return this.displayModel().defaultViewFactor();
	}

	/**
	 * Set the default view factor.
	 * 
	 * @param viewFactor jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultViewFactor_(double)
	 * @category defaults
	 */
	public void defaultViewFactor_(double viewFactor) {
		this.displayModel().defaultViewFactor_(viewFactor);
	}

	/**
	 * Create a Frame for the application model.
	 * Do everything but open the frame.
	 *
	 * @param aView jp.co.sra.smalltalk.StView
	 * @return java.awt.Frame
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#allButOpenView_(jp.co.sra.smalltalk.StView)
	 * @category interface opening
	 */
	protected Frame allButOpenView_(StView aView) {
		Frame aFrame = super.allButOpenView_(aView);
		this.spawnViewfinder();
		return aFrame;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLShowViewAwt(this);
		} else {
			return new JunOpenGLShowViewSwing(this);
		}
	}

	/**
	 * Answer this window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Viewport");
	}

	/**
	 * Bull the show object.
	 * Called from the controller.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void bull_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLDisplayModel aModel = this.displayModel();
		JunOpenGLProjection aProjection = aModel.displayProjection();
		JunSphereSurface grabSphere = aModel.grabSphere();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		JunAngle rotationAngle = (new Jun3dLine(grabSphere.center(), toPoint)).angleWithLine_(new Jun3dLine(grabSphere.center(), fromPoint));

		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(grabSphere.center(), (Jun3dPoint) grabSphere.center().minus_(((Jun3dPoint) fromPoint.minus_(grabSphere.center())).product_((Jun3dPoint) toPoint.minus_(grabSphere.center()))));
			Jun3dTransformation transformation = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.mul_(-1), rotationAxis);
			Jun3dPoint upPoint = (Jun3dPoint) aProjection.sightPoint().plus_(aProjection.unitUpVector());
			aProjection.eyePoint_(transformationInv.applyTo_(aProjection.eyePoint()));
			aProjection.upVector_((Jun3dPoint) transformation.applyTo_(upPoint).minus_(aProjection.sightPoint()));
			aModel.displayProjection_(aProjection);
			aModel.changed_($("projection"));
		}
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void drag_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.showObject() == null) {
			return;
		}

		//if (StController._ShiftDown()) {
		//	this.look_xy_(from2dPoint, to2dPoint);
		//} else {
		this.slide_xy_(from2dPoint, to2dPoint);
		//}
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void grab_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.showProjection();
		JunSphereSurface grabSphere = this.grabSphere();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		JunAngle rotationAngle = (new Jun3dLine(grabSphere.center(), toPoint)).angleWithLine_(new Jun3dLine(grabSphere.center(), fromPoint));

		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(grabSphere.center(), (Jun3dPoint) grabSphere.center().minus_(((Jun3dPoint) fromPoint.minus_(grabSphere.center())).product_((Jun3dPoint) toPoint.minus_(grabSphere.center()))));
			Jun3dTransformation transformation = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.mul_(-1), rotationAxis);
			Jun3dPoint upPoint = (Jun3dPoint) aProjection.sightPoint().plus_(aProjection.unitUpVector());
			aProjection.eyePoint_(transformationInv.applyTo_(aProjection.eyePoint()));
			aProjection.upVector_((Jun3dPoint) transformation.applyTo_(upPoint).minus_(aProjection.sightPoint()));
			this.showProjection_(aProjection);
			this.changed_($("projection"));
		}
	}

	/**
	 * Change the eye point.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#look_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void look_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.showProjection();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		this.showProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Slide the eye beam.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#slide_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void slide_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.showProjection();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		aProjection.sightPoint_((Jun3dPoint) aProjection.sightPoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		this.showProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateViewMenuIndication();
	}

	/**
	 * Update the view menu indication.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#updateViewMenuIndication()
	 * @category menu accessing
	 */
	public void updateViewMenuIndication() {
		StMenu viewMenu = (StMenu) this._menuBar().atNameKey_($("viewMenu"));
		if (viewMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Zooming
		menuItem = viewMenu.atNameKey_($("zoomingMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Pointing
		menuItem = viewMenu.atNameKey_($("pointingMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Reset
		menuItem = viewMenu.atNameKey_($("resetMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Preset
		menuItem = viewMenu.atNameKey_($("presetMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty && _presetProjections != null && _presetProjections.length > 0);
		}

		// Projection
		menuItem = viewMenu.atNameKey_($("projectionMenu"));
		if (menuItem != null) {
			StMenu projectionMenu = (StMenu) menuItem;
			if (this.showProjector().projection() instanceof JunOpenGLPerspectiveProjection) {
				menuItem = projectionMenu.atNameKey_($("perspectiveProjectionMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else {
				menuItem = projectionMenu.atNameKey_($("parallelProjectionMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Presentation
		menuItem = viewMenu.atNameKey_($("presentationMenu"));
		if (menuItem != null) {
			StMenu presentationMenu = (StMenu) menuItem;
			if (this.showProjector().presentation() == $("solidPresentation")) {
				menuItem = presentationMenu.atNameKey_($("solidPresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else if (this.showProjector().presentation() == $("wireframePresentation")) {
				menuItem = presentationMenu.atNameKey_($("wireframePresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else if (this.showProjector().presentation() == $("hiddenlinePresentation")) {
				menuItem = presentationMenu.atNameKey_($("hiddenlinePresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Shading
		menuItem = viewMenu.atNameKey_($("shadingMenu"));
		if (menuItem != null) {
			StMenu shadingMenu = (StMenu) menuItem;
			if (this.showProjector().shading() == $("flatShading")) {
				menuItem = shadingMenu.atNameKey_($("flatShadingMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else {
				menuItem = shadingMenu.atNameKey_($("smoothShadingMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Smoothing
		menuItem = viewMenu.atNameKey_($("smoothingMenu"));
		if (menuItem != null) {
			StMenu smoothingMenu = (StMenu) menuItem;
			menuItem = smoothingMenu.atNameKey_($("lineSmoothMenu"));
			if (menuItem != null) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.showProjector().lineSmooth());
			}
			menuItem = smoothingMenu.atNameKey_($("polygonSmoothMenu"));
			if (menuItem != null) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.showProjector().polygonSmooth());
			}
		}
	}

	/**
	 * Change the eye point.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeEyePoint()
	 * @category menu messages
	 */
	public void changeEyePoint() {
		Jun3dPoint point = this.showProjector().eyePoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new eye point.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.eyePoint_(point);
	}

	/**
	 * Menu message: Change the sight point.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeSightPoint()
	 * @category menu messages
	 */
	public void changeSightPoint() {
		Jun3dPoint point = this.showProjector().sightPoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new sight point.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.sightPoint_(point);
	}

	/**
	 * Menu message: Change the up vector.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeUpVector()
	 * @category menu messages
	 */
	public void changeUpVector() {
		Jun3dPoint point = this.showProjector().upVector();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new up vector.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.upVector_(point);
	}

	/**
	 * Menu message: Change the view factor.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeViewFactor()
	 * @category menu messages
	 */
	public void changeViewFactor() {
		double value = this.showProjector().viewFactor();
		String string = JunDialog.Request_($String("Input a new view factor."), String.valueOf((float) value));
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			value = Double.valueOf(string).doubleValue();
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid value."));
		}

		this.viewFactor_(value);
	}

	/**
	 * Menu message: Change the zoom height.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeZoomHeight()
	 * @category menu messages
	 */
	public void changeZoomHeight() {
		double value = this.showProjector().zoomHeight();
		String string = JunDialog.Request_($String("Input a new zoom height."), String.valueOf((float) value));
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			value = Double.valueOf(string).doubleValue();
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid value."));
		}

		this.zoomHeight_(value);
	}

	/**
	 * Menu message: Toggle a line smooth effect on a projector. 
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#lineSmooth()
	 * @category menu messages
	 */
	public void lineSmooth() {
		this.showProjector().lineSmooth_(!this.showProjector().lineSmooth());
		this.updateViewMenuIndication();
		this.changed_($("smoothing"));
	}

	/**
	 * Menu message: Toggle a polygon smooth effect on a projector.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#polygonSmooth()
	 * @category menu messages
	 */
	public void polygonSmooth() {
		this.showProjector().polygonSmooth_(!this.showProjector().polygonSmooth());
		this.updateViewMenuIndication();
		this.changed_($("smoothing"));
	}

	/**
	 * Spawn a show object and open it with JunOpenGLDisplayModel.
	 * 
	 * @category menu messages
	 */
	public void spawnObject() {
		JunOpenGLDisplayModel aDisplayModel = null;
		if (StInputState.Default().shiftDown()) {
			aDisplayModel = new JunOpenGLDisplayModel(new JunOpenGL3dCompoundObject(this.showObject(), this.viewportBeam()));
		} else {
			aDisplayModel = new JunOpenGLDisplayModel(this.showObject());
		}
		aDisplayModel.displayProjection_((JunOpenGLProjection) this.showProjection().copy());

		StView view = this.getView();
		if (view == null) {
			aDisplayModel.open();
		} else {
			StRectangle box = new StRectangle(view.topComponent().getBounds());
			StRectangle area = new StRectangle(0, 0, box.width(), box.height());
			area = area.align_with_(area.topLeft(), new Point(box.right() + 5, box.top()));
			aDisplayModel.openIn_(area.toRectangle());
		}

		aDisplayModel.changed_($("object"));
	}

	/**
	 * Spawn a Viewfinder.
	 * 
	 * @category menu messages
	 */
	public void spawnViewfinder() {
		JunOpenGLDisplayModel model = this.displayModel();
		if (model.builder().windows().length > 0) {
			model._windowExpandAndRaise();
		} else {
			model.open();
		}
	}

	/**
	 * Toggle between stipple and transparency.
	 * 
	 * @category menu messages
	 */
	public void switchStippleAndTransparency() {
		this.useTransparency_(!this.useTransparency());
	}

	/**
	 * Change the presentation mode to hidden-line.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#hiddenlinePresentation()
	 * @category presentation
	 */
	public void hiddenlinePresentation() {
		this.showProjector().hiddenlinePresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("hiddenlinePresentation"));
		}
	}

	/**
	 * Change the presentation mode to solid.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#solidPresentation()
	 * @category presentation
	 */
	public void solidPresentation() {
		this.showProjector().solidPresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("solidPresentation"));
		}
	}

	/**
	 * Change the presentation mode to wireframe.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#wireframePresentation()
	 * @category presentation
	 */
	public void wireframePresentation() {
		this.showProjector().wireframePresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("wireframePresentation"));
		}
	}

	/**
	 * Set my new eye point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#eyePoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void eyePoint_(Jun3dPoint aPoint) {
		this.showProjector().eyePoint_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("eyePoint")) == false) {
			this.defaultEyePoint_(aPoint);
		}
	}

	/**
	 * Fit the display object to the view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fit()
	 * @category projection
	 */
	public void fit() {
		if (this.showObject() == null) {
			return;
		}

		this.fitSilently();
		this.changed_($("projection"));
	}

	/**
	 * Fit the sight point to the current view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitSight()
	 * @category projection
	 */
	public void fitSight() {
		if (this.showObject() == null) {
			return;
		}

		Jun3dPoint sightPoint = this.computeSightPoint();
		this.sightPoint_(sightPoint);
	}

	/**
	 * Fit the 3d object to the current view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitSilently()
	 * @category projection
	 */
	public void fitSilently() {
		if (this.showObject() == null) {
			return;
		}

		Jun3dPoint sightPoint = this.computeSightPoint();
		this.showProjector().sightPoint_(sightPoint);
		double zoomHeight = this.computeZoomHeight();
		this.showProjector().zoomHeight_(zoomHeight);
	}

	/**
	 * Menu message: Fit to the zoom factor.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitZoom()
	 * @category projection
	 */
	public void fitZoom() {
		if (this.showObject() == null) {
			return;
		}

		double zoomHeight = this.computeZoomHeight();
		this.zoomHeight_(zoomHeight);
	}

	/**
	 * Change the projection to parallel.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#parallelProjection()
	 * @category projection
	 */
	public void parallelProjection() {
		this.showProjector().parallelProjection();
		this.updateViewMenuIndication();
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("projection")) == false) {
			this.defaultProjection_(JunOpenGLParallelProjection.Type);
		}
	}

	/**
	 * Change the projection to perspective.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#perspectiveProjection()
	 * @category projection
	 */
	public void perspectiveProjection() {
		this.showProjector().perspectiveProjection();
		this.updateViewMenuIndication();
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("projection")) == false) {
			this.defaultProjection_(JunOpenGLPerspectiveProjection.Type);
		}
	}

	/**
	 * Set my new sight point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#sightPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void sightPoint_(Jun3dPoint aPoint) {
		this.showProjector().sightPoint_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("sightPoint")) == false) {
			this.defaultSightPoint_(aPoint);
		}
	}

	/**
	 * Set my new up vector.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#upVector_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void upVector_(Jun3dPoint aPoint) {
		this.showProjector().upVector_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("upVector")) == false) {
			this.defaultUpVector_(aPoint);
		}
	}

	/**
	 * Set my new view factor.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#viewFactor_(double)
	 * @category projection
	 */
	public void viewFactor_(double factor) {
		this.showProjector().viewFactor_(factor);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("viewFactor")) == false) {
			this.defaultViewFactor_(factor);
		}
	}

	/**
	 * Do the zoom with the specified factor.
	 * 
	 * @param factor double
	 * @category projection
	 */
	public void zoom_(double factor) {
		if (StInputState.Default().shiftDown()) {
			if (factor >= 1.0) {
				factor = 1.0 + (factor - 1.0) * 0.1;
			} else {
				factor = 1.0 - (1.0 - factor) * 0.1;
			}
		}
		this.showProjector().zoom_(factor);
		this.changed_($("projection"));
	}

	/**
	 * Set my new zoom height.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#zoomHeight_(double)
	 * @category projection
	 */
	public void zoomHeight_(double aNumber) {
		this.showProjector().zoomHeight_(aNumber);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("zoomHeight")) == false) {
			this.defaultZoomHeight_(aNumber);
		}
	}

	/**
	 * Render myself on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.objects.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(final JunOpenGLRenderingContext renderingContext) {
		renderingContext.clear();

		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLLight[] lightCollection = this.displayModel().displayLightCollection();
		this.showProjector().project_withLights_on_(this.showObject(), lightCollection, renderingContext);
		this.showProjector().project_on_(this.viewportBeam(), renderingContext);
		renderingContext.flush();
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu(JunSystem.$String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem(JunSystem.$String("Spawn"), $("spawnMenu"), new MenuPerformer(this, "spawnObject")));
		miscMenu.add(new StMenuItem(JunSystem.$String("Viewport"), $("viewportMenu"), new MenuPerformer(this, "spawnViewport")));
		miscMenu.addSeparator();
		miscMenu.add(new StMenuItem(JunSystem.$String("Stipple or Transparency"), new MenuPerformer(this, "switchStippleAndTransparency")));
		return miscMenu;
	}

	/**
	 * Change the shading mode to flat.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#flatShading()
	 * @category shading
	 */
	public void flatShading() {
		this.showProjector().flatShading();
		this.updateViewMenuIndication();
		this.changed_($("shading"));

		if (this.defaultProjectionTable().containsKey($("shading")) == false) {
			this.defaultShading_($("flatShading"));
		}
	}

	/**
	 * Change the shading mode to smooth.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#smoothShading()
	 * @category shading
	 */
	public void smoothShading() {
		this.showProjector().smoothShading();
		this.updateViewMenuIndication();
		this.changed_($("shading"));

		if (this.defaultProjectionTable().containsKey($("shading")) == false) {
			this.defaultShading_($("smoothShading"));
		}
	}

	/**
	 * Called when a changed method is invoked on a dependence.
	 * 
	 * @param e jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.StObject#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent e) {
		if (e.getSource() == this.displayModel()) {
			this.changed_(e.getAspect());
		}
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeSightPoint()
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		return this.displayModel().displayProjection().sightPoint();
	}

	/**
	 * Compute the current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeZoomHeight()
	 * @category private
	 */
	protected double computeZoomHeight() {
		double depth = this.showProjection().distance();
		Jun3dPoint up = this.showProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, -1), depth);
		Jun3dPoint down = this.showProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, 1), depth);
		Jun3dPoint right = this.showProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(1, 0), depth);
		Jun3dPoint left = this.showProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(-1, 0), depth);
		final JunPlane horizontal = new JunPlane(this.showProjection().eyePoint(), left, right);
		final JunPlane vertical = new JunPlane(this.showProjection().eyePoint(), up, down);
		final double maxOffset[] = new double[] { 0.0d, 0.0d };
		this.showObject().pointsDo_(new StBlockClosure() {
			public Object value_(Object p) {
				double offsetX = vertical.distanceFromPoint_((Jun3dPoint) p);
				double offsetY = horizontal.distanceFromPoint_((Jun3dPoint) p);
				if (offsetX > maxOffset[0]) {
					maxOffset[0] = offsetX;
				}
				if (offsetY > maxOffset[1]) {
					maxOffset[1] = offsetY;
				}
				return null;
			}
		});
		double maxOffsetX = maxOffset[0];
		double maxOffsetY = maxOffset[1];
		double factor = 2.5;
		return (maxOffsetX > maxOffsetY) ? maxOffsetX * factor : maxOffsetY * factor;
	}

	/**
	 * Compute the current viewport height.
	 * 
	 * @return double
	 * @category private
	 */
	protected double computeViewportHeight() {
		JunOpenGLProjection aProjection = this.displayModel().displayProjection();
		double currentHeight = aProjection.regularHeight();
		double nearHeight = 2 * aProjection.sightPoint().distance_(aProjection.nearCenter());
		if (currentHeight > nearHeight) {
			currentHeight = nearHeight;
		}
		return currentHeight;
	}

	/**
	 * Grab at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab3dPoint_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category private
	 */
	protected Jun3dPoint grab3dPoint_(Jun2dPoint aPoint) {
		JunOpenGLProjection projection = this.showProjection();
		JunSphereSurface handleSphere = this.grabSphere();
		JunOpenGLParallelProjection prj = (JunOpenGLParallelProjection) projection.asParallelProjection();
		Jun3dLine line = new Jun3dLine(prj.translateTo3dPointInNearBoundaryPlaneFromPoint_(aPoint), prj.translateTo3dPointInFarBoundaryPlaneFromPoint_(aPoint));
		Jun3dPoint[] candidates = handleSphere.crossPointsWithLine_(line);
		if (candidates.length == 0) {
			return prj.translateTo3dPointFromPoint_(aPoint);
		} else {
			return candidates[0];
		}
	}

	/**
	 * Create a sphere for grabbing.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunSphereSurface
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grabSphere()
	 * @category private
	 */
	protected JunSphereSurface grabSphere() {
		JunOpenGLProjection projection = this.showProjection();
		JunSphereSurface grabSphere = new JunSphereSurface(projection.sightPoint(), projection.regularHeight() / 2);
		return grabSphere;
	}

}
