package jp.co.sra.jun.opengl.cosmos;

import java.awt.event.WindowEvent;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;

import jp.co.sra.gl4jun.GLjInterface;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLHypercosmModel class
 * 
 *  @author    nisinaka
 *  @created   2004/01/06 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLHypercosmModel.java,v 8.12 2008/02/20 06:32:18 nisinaka Exp $
 */
public class JunOpenGLHypercosmModel extends JunOpenGLDisplayModel {

	protected Vector hyperDoors;
	protected Vector hyperSpaces;
	protected Hashtable stencilTable;

	/**
	 * Create a new instance of JunOpenGLHypercosmModel and initialize it.
	 * 
	 * @param displayObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public JunOpenGLHypercosmModel(JunOpenGL3dObject displayObject) {
		super(displayObject);
	}

	/**
	 * Initialize the JunOpenGLDisplayModel.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		// this.useDisplayList_(false);
		hyperDoors = null;
		hyperSpaces = null;
		stencilTable = null;
	}

	/**
	 * Answer my current hyper doors.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector hyperDoors() {
		if (hyperDoors == null) {
			hyperDoors = new Vector();
		}

		return hyperDoors;
	}

	/**
	 * Answer my current hyper doors with no duplication.
	 * 
	 * @return jp.co.sra.jun.opengl.object.JunOpenGL3dObject[]
	 * @category accessing
	 */
	public JunOpenGL3dObject[] hyperDoorsNoDupulication() {
		Vector aCollection = new Vector(this.hyperDoors().size());
		for (Enumeration e = this.hyperDoors().elements(); e.hasMoreElements();) {
			JunOpenGL3dObject hyperDoor = (JunOpenGL3dObject) e.nextElement();
			if (aCollection.contains(hyperDoor) == false) {
				aCollection.add(hyperDoor);
			}
		}
		return (JunOpenGL3dObject[]) aCollection.toArray(new JunOpenGL3dObject[aCollection.size()]);
	}

	/**
	 * Answer my current hyper spaces.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector hyperSpaces() {
		if (hyperSpaces == null) {
			hyperSpaces = new Vector();
		}

		return hyperSpaces;
	}

	/**
	 * Answer my current hyper spaces with no duplication.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel[]
	 * @category accessing
	 */
	public JunOpenGLDisplayModel[] hyperSpacesNoDupulication() {
		Vector aCollection = new Vector(this.hyperSpaces().size());
		for (Enumeration e = this.hyperSpaces().elements(); e.hasMoreElements();) {
			JunOpenGLDisplayModel hyperSpace = (JunOpenGLDisplayModel) e.nextElement();
			if (aCollection.contains(hyperSpace) == false) {
				aCollection.add(hyperSpace);
			}
		}
		return (JunOpenGLDisplayModel[]) aCollection.toArray(new JunOpenGLDisplayModel[aCollection.size()]);
	}

	/**
	 * Add a pair of a hyper door and a hyper space.
	 * 
	 * @param stencilObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param displayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category adding
	 */
	public void add(JunOpenGL3dObject stencilObject, JunOpenGLDisplayModel displayModel) {
		if (displayModel == this) {
			return;
		}

		this.hyperDoors().add(stencilObject);
		this.hyperSpaces().add(displayModel);

		if (displayModel.myDependents() == null || displayModel.myDependents().contains(this) == false) {
			displayModel.addDependent_(this);
		}
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#boundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (this.displayObject() == null) {
			return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 0));
		}

		Jun3dBoundingBox boundingBox = this.displayObject().boundingBox();
		for (Enumeration e = this.hyperDoors().elements(); e.hasMoreElements();) {
			JunOpenGL3dObject hyperDoor = (JunOpenGL3dObject) e.nextElement();
			boundingBox = boundingBox.merge_(hyperDoor.boundingBox());
		}
		return boundingBox;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Hypercosm");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);

		JunOpenGLDisplayModel[] hyperSpaces = this.hyperSpacesNoDupulication();
		for (int i = 0; i < hyperSpaces.length; i++) {
			hyperSpaces[i].closeRequest();
		}
	}

	/**
	 * Answer the array of all display lights.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLights()
	 * @category lighting
	 */
	public synchronized JunOpenGLDisplayLight[] displayLights() {
		if (displayLights == null) {
			displayLights = new JunOpenGLDisplayLight[5];
			displayLights[0] = JunOpenGLDisplayLight.ParallelLight_color_position_(false, StColorValue.Brightness_(0.1), this.defaultLightPoint());
			displayLights[1] = new JunOpenGLDisplayLight();
			displayLights[2] = new JunOpenGLDisplayLight();
			displayLights[3] = new JunOpenGLDisplayLight();
			displayLights[4] = JunOpenGLDisplayLight.AmbientLight_color_(true, this.defaultLightColor());

			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].compute_(new StBlockClosure() {
					public Object value() {
						JunOpenGLHypercosmModel.this.updateLightMenuIndication();
						JunOpenGLHypercosmModel.this.changed_($("light"));
						return null;
					}
				});
			}
		}

		return displayLights;
	}

	/**
	 * Render the display objects on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.objects.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext renderingContext) {
		renderingContext.clear();
		GLjInterface.Current().glEnable(GLjInterface.GL_STENCIL_TEST);
		GLjInterface.Current().glStencilMask(GLjInterface.GL_TRUE);
		GLjInterface.Current().glClear(GLjInterface.GL_STENCIL_BUFFER_BIT);
		this.computeStencilTable();

		try {
			Enumeration eDoors = this.hyperDoors().elements();
			Enumeration eSpaces = this.hyperSpaces().elements();
			while (eDoors.hasMoreElements() && eSpaces.hasMoreElements()) {
				JunOpenGL3dObject hyperDoor = (JunOpenGL3dObject) eDoors.nextElement();
				JunOpenGLDisplayModel hyperSpace = (JunOpenGLDisplayModel) eSpaces.nextElement();
				this.renderHyperDoor_on_(hyperDoor, renderingContext);
				this.renderHyperSpace_on_(hyperSpace, (JunOpenGLRenderingContext) renderingContext.clone());
				this.renderHyperDoorAux_on_(hyperDoor, renderingContext);
			}
			this.renderHyperDoorsOn_(renderingContext);
			this.renderDisplayObjectOn_(renderingContext);
		} finally {
			GLjInterface.Current().glDisable(GLjInterface.GL_STENCIL_TEST);
			GLjInterface.Current().glStencilMask(GLjInterface.GL_FALSE);
		}
	}

	/**
	 * Render the hyper door on the rendering context.
	 * 
	 * @param hyperDoor jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderHyperDoor_on_(JunOpenGL3dObject hyperDoor, JunOpenGLRenderingContext renderingContext) {
		GLjInterface.Current().glColorMask(false, false, false, false);
		renderingContext.disableDepthMask();

		try {
			GLjInterface.Current().glStencilFunc(GLjInterface.GL_ALWAYS, 1, 1);
			GLjInterface.Current().glStencilOp(GLjInterface.GL_REPLACE, GLjInterface.GL_REPLACE, GLjInterface.GL_REPLACE);
			JunOpenGL3dPolygon[] polygons = (JunOpenGL3dPolygon[]) stencilTable.get(hyperDoor);
			for (int i = 0; i < polygons.length; i++) {
				this.displayProjector().project_on_(polygons[i], renderingContext);
			}
		} finally {
			renderingContext.enableDepthMask();
			GLjInterface.Current().glColorMask(true, true, true, true);
		}
	}

	/**
	 * Render the hyper space on the rendering context.
	 * 
	 * @param hyperSpace jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderHyperSpace_on_(JunOpenGLDisplayModel hyperSpace, JunOpenGLRenderingContext renderingContext) {
		GLjInterface.Current().glStencilFunc(GLjInterface.GL_EQUAL, 1, 1);
		GLjInterface.Current().glStencilOp(GLjInterface.GL_KEEP, GLjInterface.GL_KEEP, GLjInterface.GL_KEEP);
		hyperSpace.displayProjector().project_withLights_on_(hyperSpace.displayObject(), hyperSpace.displayLightCollection(), renderingContext);
	}

	/**
	 * Render the hyper door aux on the rendering context.
	 * 
	 * @param hyperDoor jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderHyperDoorAux_on_(JunOpenGL3dObject hyperDoor, JunOpenGLRenderingContext renderingContext) {
		GLjInterface.Current().glColorMask(false, false, false, false);
		renderingContext.disableDepthMask();

		try {
			GLjInterface.Current().glStencilFunc(GLjInterface.GL_EQUAL, 1, 1);
			GLjInterface.Current().glStencilOp(GLjInterface.GL_ZERO, GLjInterface.GL_ZERO, GLjInterface.GL_ZERO);
			JunOpenGL3dPolygon[] polygons = (JunOpenGL3dPolygon[]) stencilTable.get(hyperDoor);
			for (int i = 0; i < polygons.length; i++) {
				this.displayProjector().project_on_(polygons[i], renderingContext);
			}
		} finally {
			renderingContext.enableDepthMask();
			GLjInterface.Current().glColorMask(true, true, true, true);
		}
	}

	/**
	 * Render the hyper doors on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderHyperDoorsOn_(JunOpenGLRenderingContext renderingContext) {
		GLjInterface.Current().glColorMask(false, false, false, false);
		renderingContext.disableDepthMask();

		try {
			GLjInterface.Current().glStencilFunc(GLjInterface.GL_ALWAYS, 1, 1);
			GLjInterface.Current().glStencilOp(GLjInterface.GL_REPLACE, GLjInterface.GL_REPLACE, GLjInterface.GL_REPLACE);
			for (Enumeration e = this.hyperDoors().elements(); e.hasMoreElements();) {
				JunOpenGL3dPolygon[] polygons = (JunOpenGL3dPolygon[]) stencilTable.get(e.nextElement());
				for (int i = 0; i < polygons.length; i++) {
					this.displayProjector().project_on_(polygons[i], renderingContext);
				}
			}
		} finally {
			renderingContext.enableDepthMask();
			GLjInterface.Current().glColorMask(true, true, true, true);
		}

	}

	/**
	 * Render the display object on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderDisplayObjectOn_(JunOpenGLRenderingContext renderingContext) {
		GLjInterface.Current().glStencilFunc(GLjInterface.GL_NOTEQUAL, 1, 1);
		GLjInterface.Current().glStencilOp(GLjInterface.GL_KEEP, GLjInterface.GL_KEEP, GLjInterface.GL_KEEP);
		JunOpenGLLight[] lightCollection = this.displayLightCollection();
		JunOpenGL3dObject renderingObject = this.displayObject();
		if (renderingObject == null) {
			this.displayProjector().projectOn_(renderingContext);
		} else {
			if (this.selectedObjects().isEmpty()) {
				this.displayProjector().project_withLights_on_(renderingObject, lightCollection, renderingContext);
			} else {
				JunOpenGLProjector aProjector = (JunOpenGLProjector) this.displayProjector().copy();
				if (this.displayProjector().presentation() == $("solidPresentation")) {
					aProjector.wireframePresentation();
				} else if (this.displayProjector().presentation() == $("wireframePresentation")) {
					aProjector.wireframePresentation();
				} else if (this.displayProjector().presentation() == $("hiddenlinePresentation")) {
					aProjector.hiddenlinePresentation();
				}

				aProjector.project_withLights_on_(renderingObject, lightCollection, renderingContext);

				if (this.displayProjector().presentation() == $("solidPresentation")) {
					aProjector.solidPresentation();
				} else if (this.displayProjector().presentation() == $("wireframePresentation")) {
					aProjector.solidPresentation();
				} else if (this.displayProjector().presentation() == $("hiddenlinePresentation")) {
					aProjector.solidPresentation();
				}

				Object[] objects = this.selectedObjects().toArray();
				for (int i = 0; i < objects.length; i++) {
					aProjector.project_on_((JunOpenGL3dObject) objects[i], renderingContext);
				}

			}
		}
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument evt.getAspect() is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		super.update_(evt);

		if (evt.getAspect() != $("state")) {
			this.changed_(evt);
		}
	}

	/**
	 * Compute the stencil table.
	 * 
	 * @category private
	 */
	protected void computeStencilTable() {
		stencilTable = new Hashtable();
		for (Enumeration e = this.hyperDoors().elements(); e.hasMoreElements();) {
			JunOpenGL3dObject hyperDoor = (JunOpenGL3dObject) e.nextElement();
			final Vector stencilPolygons = new Vector(32);
			hyperDoor.polygonsDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunOpenGL3dPolygon aPolygon = (JunOpenGL3dPolygon) anObject;
					Jun3dPoint[] vertexes = aPolygon.vertexes();
					double valueF = (new JunPlane(vertexes[0], vertexes[1], vertexes[2])).valueF_(JunOpenGLHypercosmModel.this.eyePoint());
					if (valueF > 0) {
						stencilPolygons.add(aPolygon);
					}
					return null;
				}
			});

			stencilTable.put(hyperDoor, (JunOpenGL3dPolygon[]) stencilPolygons.toArray(new JunOpenGL3dPolygon[stencilPolygons.size()]));
		}
	}

}
