package jp.co.sra.jun.opengl.chart;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.Comparator;
import java.util.TreeSet;
import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunChartDataWithNumericalKeys class
 * 
 *  @author    nisinaka
 *  @created   1999/01/13 (by nisinaka)
 *  @updated   2007/06/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunChartDataWithNumericalKeys.java,v 8.13 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunChartDataWithNumericalKeys extends JunChartData {

	protected double[] keyRange = null;
	protected double minimumKey = Double.NaN;
	protected double maximumKey = Double.NaN;

	/**
	 * Create a new instance of JunChartDataWithNumericalKeys and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunChartDataWithNumericalKeys() {
		super();
	}

	/**
	 * Create a new instance of a ChartData and initialize it with the
	 * collection of sample.. This method corresponds to the class method
	 * "samples:" of JunChartData in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category Instance creation
	 */
	public JunChartDataWithNumericalKeys(java.util.Vector aCollectionOfSample) {
		super(aCollectionOfSample);
	}

	/**
	 * Create a new JunChartData with a collection of sample and a number of
	 * keys. This method corresponds to the class method
	 * "samples:numberOfKeys:" of JunChartData in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @param numberOfKeys int
	 * @category Instance creation
	 */
	public JunChartDataWithNumericalKeys(java.util.Vector aCollectionOfSample, int numberOfKeys) {
		super(aCollectionOfSample, numberOfKeys);
	}

	/**
	 * Set the samples of the receiver.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category accessing
	 */
	public void samples_(Vector aCollectionOfSample) {
		Object[][][] savedSamples = this.samples();
		super.samples_(aCollectionOfSample);

		if (this.hasValidKeys() == false) {
			JunDialog.Warn_(JunSystem.$String("Invalid keys:  Could not change samples."));
			super.samples_(savedSamples);
		}
	}

	/**
	 * Answer the array of JunChartDataSheet.
	 * 
	 * @return jp.co.sra.jun.opengl.chart.JunChartDataSheet[]
	 * @category accessing
	 */
	public JunChartDataSheet[] sheets() {
		if (sheets == null) {
			if (samples == null) {
				return null;
			}
			if (!this.hasValidSamples()) {
				return null;
			}

			Comparator aComparator = new Comparator() {
				public int compare(Object arg0, Object arg1) {
					double v0 = ((Number) ((Object[]) arg0)[0]).doubleValue();
					double v1 = ((Number) ((Object[]) arg1)[0]).doubleValue();
					return (v0 < v1) ? -1 : 1;
				}
			};

			int size = samples.length;
			sheets = new JunChartDataSheet[size];
			for (int i = 0; i < size; i++) {
				TreeSet aTreeSet = new TreeSet(aComparator);
				for (int j = 0; j < samples[i].length; j++) {
					aTreeSet.add(samples[i][j]);
				}

				Object[][] collectionOfArray = (Object[][]) aTreeSet.toArray(new Object[aTreeSet.size()][]);
				sheets[i] = new JunChartDataSheet(collectionOfArray, this.numberOfKeys());
			}
		}
		return sheets;
	}

	/**
	 * Answer the interval of the key range.
	 * 
	 * @return double
	 * @category key accessing
	 */
	public double intervalOfKeyRange() {
		return this.keyRange()[2];
	}

	/**
	 * Answer the information about the key range.
	 * 
	 * @return double[]
	 * @category key accessing
	 */
	public double[] keyRange() {
		if (keyRange == null) {
			keyRange = this.defaultKeyRange();
		}
		return keyRange;
	}

	/**
	 * Set the information about the key range.
	 * 
	 * @param anArray double[]
	 * @category key accessing
	 */
	public void keyRange_(double[] anArray) {
		keyRange = anArray;
	}

	/**
	 * Set the information about the key range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @category key accessing
	 */
	public void keyRangeFrom_to_(double minimumNumber, double maximumNumber) {
		this.keyRangeFrom_to_by_(minimumNumber, maximumNumber, this.defaultIntervalFrom_to_(minimumNumber, maximumNumber));
	}

	/**
	 * Set the information about the key range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @param interval double
	 * @category key accessing
	 */
	public void keyRangeFrom_to_by_(double minimumNumber, double maximumNumber, double interval) {
		if (minimumNumber >= maximumNumber) {
			return;
		}

		double[] anArray = { minimumNumber, maximumNumber, interval };
		this.keyRange_(anArray);
	}

	/**
	 * Answer the maximum key of the chart data.
	 * 
	 * @return double
	 * @category key accessing
	 */
	public double maximumKey() {
		if (Double.isNaN(maximumKey)) {
			final StValueHolder maximumValueHolder = new StValueHolder(Double.NEGATIVE_INFINITY);
			JunChartDataSheet[] sheets = this.sheets();
			for (int i = 0; i < sheets.length; i++) {
				sheets[i].keysDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						double aNumber = ((Number) anObject).doubleValue();
						if (aNumber > maximumValueHolder._doubleValue()) {
							maximumValueHolder.value_(aNumber);
						}
						return null;
					}
				});
			}
			maximumKey = maximumValueHolder._doubleValue();
		}
		return maximumKey;
	}

	/**
	 * Answer the maximum of the key range.
	 * 
	 * @return double
	 * @category key accessing
	 */
	public double maximumOfKeyRange() {
		return this.keyRange()[1];
	}

	/**
	 * Answer the minimum key of the chart data.
	 * 
	 * @return double
	 * @category key accessing
	 */
	public double minimumKey() {
		if (Double.isNaN(minimumKey)) {
			final StValueHolder minimumValueHolder = new StValueHolder(Double.MAX_VALUE);
			JunChartDataSheet[] sheets = this.sheets();
			for (int i = 0; i < sheets.length; i++) {
				sheets[i].keysDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						double aNumber = ((Number) anObject).doubleValue();
						if (aNumber < minimumValueHolder._doubleValue()) {
							minimumValueHolder.value_(anObject);
						}
						return null;
					}
				});
			}
			minimumKey = minimumValueHolder._doubleValue();
		}
		return minimumKey;
	}

	/**
	 * Answer the minimum of the key range.
	 * 
	 * @return double
	 * @category key accessing
	 */
	public double minimumOfKeyRange() {
		return this.keyRange()[0];
	}

	/**
	 * Answer the default key range.
	 *      0 : minimum value
	 *      1 : maximum value
	 *      2 : interval
	 * 
	 * @return double[]
	 * @category defaults
	 */
	protected double[] defaultKeyRange() {
		double max = this.maximumKey();
		double min = this.minimumKey();
		double[] range = { min, max, this.defaultIntervalFrom_to_(min, max) };
		return range;
	}

	/**
	 * Answer the default number of keys.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultNumberOfKeys() {
		return 1;
	}

	/**
	 * Flush the cached values.
	 * 
	 * @category flushing
	 */
	protected void flushCaches() {
		super.flushCaches();
		minimumKey = Double.NaN;
		maximumKey = Double.NaN;
	}

	/**
	 * Normalize the number in the key range.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category normalizing
	 */
	public double normalizeKey_(double aNumber) {
		double[] range = { 0, 1 };
		return this.mapNumber_fromRange_toRange_(aNumber, this.keyRange(), range);
	}

	/**
	 * Denormalize the number in the key range.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category normalizing
	 */
	public double denormalizeKey_(double aNumber) {
		double[] range = { 0, 1 };
		return this.mapNumber_fromRange_toRange_(aNumber, range, this.keyRange());
	}

	/**
	 * Print my string representation on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);

		BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
		bw.write('(');
		bw.newLine();
		bw.write("     keyRange :  ");
		if (keyRange == null) {
			bw.write("null");
		} else {
			bw.write(" from ");
			bw.write(String.valueOf(keyRange[0]));
			bw.write(" to ");
			bw.write(String.valueOf(keyRange[1]));
			bw.write(" by ");
			bw.write(String.valueOf(keyRange[2]));
		}
		bw.newLine();
		bw.write("     minimumKey :  ");
		bw.write(String.valueOf(minimumKey));
		bw.newLine();
		bw.write("     maximumKey :  ");
		bw.write(String.valueOf(maximumKey));
		bw.newLine();
		bw.write(')');
		bw.flush();
	}

	/**
	 * Answer true if the receiver has a valid keys, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasValidKeys() {
		if (this.numberOfKeys() != 1) {
			return false;
		}

		JunChartDataSheet[] sheets = this.sheets();
		for (int i = 0; i < sheets.length; i++) {
			Object result = sheets[i].keysDo_(new StBlockClosure() {
				public Object value_(Object key) {
					if (key instanceof Number == false) {
						return Boolean.FALSE;
					}
					return null;
				}
			});
			if (result == Boolean.FALSE) {
				return false;
			}
		}
		return true;
	}

}
