package jp.co.sra.jun.octree.nodes;

import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.breps.basic.JunBreps;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.solid.abstracts.JunSolidModel;
import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.topology.elements.JunBody;
import jp.co.sra.jun.topology.elements.JunEdge;
import jp.co.sra.jun.topology.elements.JunLoop;
import jp.co.sra.jun.topology.elements.JunVertex;
import jp.co.sra.jun.topology.euleroperators.JunKEL;
import jp.co.sra.jun.topology.euleroperators.JunKEML;
import jp.co.sra.jun.topology.euleroperators.JunKEV;
import jp.co.sra.jun.topology.euleroperators.JunKEVVL;
import jp.co.sra.jun.topology.euleroperators.JunMEL;
import jp.co.sra.jun.topology.euleroperators.JunMVE;

/**
 * JunOctreeNode class
 * 
 *  @author    nisinaka
 *  @created   1999/12/28 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOctreeNode.java,v 8.11 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunOctreeNode extends JunAbstractObject {
	//
	protected static final Jun3dPoint[] PolygonNormals = new Jun3dPoint[] { new Jun3dPoint(0, 0, 1), new Jun3dPoint(0, 0, -1), new Jun3dPoint(1, 0, 0), new Jun3dPoint(-1, 0, 0), new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, -1, 0), };
	protected static final Jun3dPoint[][] PolygonPoints = new Jun3dPoint[][] {
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 1), new Jun3dPoint(1, 0, 1), new Jun3dPoint(1, 1, 1), new Jun3dPoint(0, 1, 1), },
			new Jun3dPoint[] { new Jun3dPoint(0, 1, 0), new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 0, 0), new Jun3dPoint(0, 0, 0), },
			new Jun3dPoint[] { new Jun3dPoint(1, 0, 0), new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 1, 1), new Jun3dPoint(1, 0, 1), },
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 1), new Jun3dPoint(0, 1, 1), new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0, 0), },
			new Jun3dPoint[] { new Jun3dPoint(0, 1, 1), new Jun3dPoint(1, 1, 1), new Jun3dPoint(1, 1, 0), new Jun3dPoint(0, 1, 0), },
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 0), new Jun3dPoint(1, 0, 0), new Jun3dPoint(1, 0, 1), new Jun3dPoint(0, 0, 1), }, };
	protected static final Jun3dPoint VertexPoint = new Jun3dPoint(0.5, 0.5, 0.5);
	protected static final Jun3dPoint[][] WireframePoints = new Jun3dPoint[][] {
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 1), new Jun3dPoint(1, 0, 1), new Jun3dPoint(1, 1, 1), new Jun3dPoint(0, 1, 1), new Jun3dPoint(0, 0, 1), },
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 0), new Jun3dPoint(1, 0, 0), new Jun3dPoint(1, 1, 0), new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0, 0), },
			new Jun3dPoint[] { new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 1), },
			new Jun3dPoint[] { new Jun3dPoint(1, 0, 0), new Jun3dPoint(1, 0, 1), },
			new Jun3dPoint[] { new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 1, 1), },
			new Jun3dPoint[] { new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 1, 1), }, };

	/** The contents of the node. */
	protected Object contents = null;

	/**
	 * Create a new instance of <code>JunOctreeNode</code> and initialize it.
	 * 
	 * @param aBoolean java.lang.Boolean
	 * @category Instance creation
	 */
	public JunOctreeNode(Boolean aBoolean) {
		this(aBoolean.booleanValue());
	}

	/**
	 * Create a new instance of <code>JunOctreeNode</code> and initialize it.
	 * 
	 * @param aBoolean boolean
	 * @category Instance creation
	 */
	public JunOctreeNode(boolean aBoolean) {
		super();
		if (aBoolean == true) {
			this.isFilled_(Boolean.TRUE);
		} else {
			this.isFilled_(Boolean.FALSE);
		}
	}

	/**
	 * Create a new instance of JunOctreeNode as an empty node.
	 * 
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category Instance creation
	 */
	public static JunOctreeNode Empty() {
		return new JunOctreeNode(false);
	}

	/**
	 * Create a new instance of JunOctreeNode as a filled node.
	 * 
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category Instance creation
	 */
	public static JunOctreeNode Filled() {
		return new JunOctreeNode(true);
	}

	/**
	 * Convert to JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		if (this.isFilled()) {
			return JunOpenGL3dObject.Origin_extent_(Jun3dPoint.Zero(), Jun3dPoint.Unity());
		} else if (this.isEmpty()) {
			return null;
		} else if (contents instanceof JunBreps) {
			return ((JunBreps) contents).asJunOpenGL3dObject();
		}

		throw new SmalltalkException("unknown status of the node");
	}

	/**
	 * Make the node to be compact.
	 */
	public void beCompact() {
		if (this.isNode()) {
			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;

			for (int i = 0; i < _compositionalContents.length; i++) {
				_compositionalContents[i].beCompact();
			}

			this.basicBeCompact();
		}
	}

	/**
	 * Set the contents to be divided.
	 */
	public void beDivided() {
		JunOctreeNode[] newContents = new JunOctreeNode[8];

		for (int i = 0; i < 8; i++) {
			newContents[i] = new JunOctreeNode((Boolean) contents);
		}

		contents = newContents;
	}

	/**
	 * Set the contents to be empty.
	 */
	public void beEmpty() {
		this.isFilled_(Boolean.FALSE);
	}

	/**
	 * Set the contents to be filled.
	 */
	public void beFilled() {
		this.isFilled_(Boolean.TRUE);
	}

	/**
	 * Set myself as a node of b-reps octree.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void brepsOrigin_extent_accuracy_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, double aNumber, StBlockClosure aBlock) {
		if ((aJun3dPoint2.x() < aNumber) && (aJun3dPoint2.y() < aNumber) && (aJun3dPoint2.z() < aNumber)) {
			this.brepsOrigin_extent_block_(aJun3dPoint1, aJun3dPoint2, aBlock);
		} else {
			this.beDivided();

			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, aNumber, aBlock);
			_compositionalContents[1].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, aNumber, aBlock);
			_compositionalContents[2].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, aNumber, aBlock);
			_compositionalContents[3].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, aNumber, aBlock);
			_compositionalContents[4].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, aNumber, aBlock);
			_compositionalContents[5].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, aNumber, aBlock);
			_compositionalContents[6].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, aNumber, aBlock);
			_compositionalContents[7].brepsOrigin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, aNumber, aBlock);
			this.basicBeCompact();
		}
	}

	/**
	 * Set myself as a node of b-reps octree.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param depth int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void brepsOrigin_extent_depth_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, int depth, StBlockClosure aBlock) {
		if (depth <= 1) {
			this.brepsOrigin_extent_block_(aJun3dPoint1, aJun3dPoint2, aBlock);
		} else {
			this.beDivided();

			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[1].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[2].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[3].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[4].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[5].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[6].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[7].brepsOrigin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, depth - 1, aBlock);
			this.basicBeCompact();
		}
	}

	/**
	 * Answer the child node with the specified index.
	 * 
	 * @param index int
	 * 
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 */
	public JunOctreeNode childAt_(int index) {
		if (contents instanceof JunOctreeNode[] == false) {
			return null;
		}

		return ((JunOctreeNode[]) contents)[index];
	}

	/**
	 * Answer true if the Jun3dPoint is contained in the node.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return boolean
	 */
	public boolean containsPoint_(Jun3dPoint aJun3dPoint) {
		if (contents instanceof Boolean) {
			return ((Boolean) contents).booleanValue();
		} else if (contents instanceof JunBreps) {
			return ((JunBreps) contents).containsPoint_(aJun3dPoint);
		}

		return false;
	}

	/**
	 * Answer the contents.
	 * 
	 * @return java.lang.Object
	 */
	public Object contents() {
		return contents;
	}

	/**
	 * Answer true if the recevier has children nodes, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean hasChildren() {
		return contents instanceof JunOctreeNode[];
	}

	/**
	 * Answer true if the recevier is an empty node, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isEmpty() {
		return contents == Boolean.FALSE;
	}

	/**
	 * Answer true if the recevier is a filled node, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isFilled() {
		return contents == Boolean.TRUE;
	}

	/**
	 * Set the contents to be filled or not.
	 * 
	 * @param aBoolean java.lang.Boolean
	 */
	public void isFilled_(Boolean aBoolean) {
		contents = aBoolean;
	}

	/**
	 * Answer true if the recevier is a node, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isNode() {
		return this.hasChildren();
	}

	/**
	 * Answer true if the recevier is a tip, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isTip() {
		return (this.isNode() == false);
	}

	/**
	 * Evaluate the block with the specified conditions.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param accuracy double
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void origin_extent_accuracy_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, double accuracy, StBlockClosure aBlock) {
		if ((aJun3dPoint2.x() <= accuracy) && (aJun3dPoint2.y() <= accuracy) && (aJun3dPoint2.z() <= accuracy)) {
			this.origin_extent_block_(aJun3dPoint1, aJun3dPoint2, aBlock);
		} else {
			this.beDivided();

			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].origin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, accuracy, aBlock);
			_compositionalContents[1].origin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, accuracy, aBlock);
			_compositionalContents[2].origin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, accuracy, aBlock);
			_compositionalContents[3].origin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, accuracy, aBlock);
			_compositionalContents[4].origin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, accuracy, aBlock);
			_compositionalContents[5].origin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, accuracy, aBlock);
			_compositionalContents[6].origin_extent_accuracy_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, accuracy, aBlock);
			_compositionalContents[7].origin_extent_accuracy_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, accuracy, aBlock);
			this.basicBeCompact();
		}
	}

	/**
	 * Set myself to be filled or not.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void origin_extent_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, StBlockClosure aBlock) {
		Number aNumber = (Number) aBlock.value_(aJun3dPoint2.dividedBy_(2.0d).plus_(aJun3dPoint1));

		if (aNumber.doubleValue() >= 0) {
			this.beFilled();
		} else {
			this.beEmpty();
		}
	}

	/**
	 * Set myself to be filled or to be empty.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param depth int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void origin_extent_depth_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, int depth, StBlockClosure aBlock) {
		if (depth <= 1) {
			this.origin_extent_block_(aJun3dPoint1, aJun3dPoint2, aBlock);
		} else {
			this.beDivided();

			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].origin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[1].origin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[2].origin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[3].origin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, depth - 1, aBlock);
			_compositionalContents[4].origin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[5].origin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[6].origin_extent_depth_block_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, depth - 1, aBlock);
			_compositionalContents[7].origin_extent_depth_block_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, depth - 1, aBlock);
			this.basicBeCompact();
		}
	}

	/**
	 * Enumerate the nodes and evaluate the block closure.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object origin_extent_nodesDo_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, StBlockClosure aBlock) {
		aBlock.value_value_value_(aJun3dPoint1, aJun3dPoint2, this);

		if (this.isNode()) {
			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].origin_extent_nodesDo_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, aBlock);
			_compositionalContents[1].origin_extent_nodesDo_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, aBlock);
			_compositionalContents[2].origin_extent_nodesDo_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, aBlock);
			_compositionalContents[3].origin_extent_nodesDo_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, aBlock);
			_compositionalContents[4].origin_extent_nodesDo_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, aBlock);
			_compositionalContents[5].origin_extent_nodesDo_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, aBlock);
			_compositionalContents[6].origin_extent_nodesDo_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, aBlock);
			_compositionalContents[7].origin_extent_nodesDo_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, aBlock);
		}

		return null;
	}

	/**
	 * Enumerate all nodes except aCollectionOfJunOctreeNode and evaluate the
	 * block.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunOctreeNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object origin_extent_nodesWithout_do_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, JunOctreeNode aJunOctreeNode, StBlockClosure aBlock) {
		if (this != aJunOctreeNode) {
			aBlock.value_value_value_(aJun3dPoint1, aJun3dPoint2, this);

			if (this.isNode()) {
				JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
				Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
				Jun3dPoint origin1 = aJun3dPoint1;
				Jun3dPoint origin2 = origin1.plus_(extent);
				_compositionalContents[0].origin_extent_nodesWithout_do_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[1].origin_extent_nodesWithout_do_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[2].origin_extent_nodesWithout_do_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[3].origin_extent_nodesWithout_do_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[4].origin_extent_nodesWithout_do_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[5].origin_extent_nodesWithout_do_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[6].origin_extent_nodesWithout_do_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, aJunOctreeNode, aBlock);
				_compositionalContents[7].origin_extent_nodesWithout_do_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, aJunOctreeNode, aBlock);
			}
		}

		return null;
	}

	/**
	 * Enumerate all nodes except aCollectionOfJunOctreeNode and evaluate the
	 * block.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aCollectionOfJunOctreeNode java.util.Vector
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object origin_extent_nodesWithoutAll_do_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, Vector aCollectionOfJunOctreeNode, StBlockClosure aBlock) {
		if (aCollectionOfJunOctreeNode.contains(this) == false) {
			aBlock.value_value_value_(aJun3dPoint1, aJun3dPoint2, this);

			if (this.isNode()) {
				JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
				Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
				Jun3dPoint origin1 = aJun3dPoint1;
				Jun3dPoint origin2 = origin1.plus_(extent);
				_compositionalContents[0].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[1].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[2].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[3].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[4].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[5].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[6].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, aCollectionOfJunOctreeNode, aBlock);
				_compositionalContents[7].origin_extent_nodesWithoutAll_do_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, aCollectionOfJunOctreeNode, aBlock);
			}
		}

		return null;
	}

	/**
	 * Enumerate the tips and evaluate the block closure.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object origin_extent_tipsDo_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, StBlockClosure aBlock) {
		if (this.isTip()) {
			return aBlock.value_value_value_(aJun3dPoint1, aJun3dPoint2, this);
		} else {
			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;
			Jun3dPoint extent = aJun3dPoint2.dividedBy_(2.0d);
			Jun3dPoint origin1 = aJun3dPoint1;
			Jun3dPoint origin2 = origin1.plus_(extent);
			_compositionalContents[0].origin_extent_tipsDo_(new Jun3dPoint(origin1.x(), origin1.y(), origin1.z()), extent, aBlock);
			_compositionalContents[1].origin_extent_tipsDo_(new Jun3dPoint(origin2.x(), origin1.y(), origin1.z()), extent, aBlock);
			_compositionalContents[2].origin_extent_tipsDo_(new Jun3dPoint(origin1.x(), origin2.y(), origin1.z()), extent, aBlock);
			_compositionalContents[3].origin_extent_tipsDo_(new Jun3dPoint(origin2.x(), origin2.y(), origin1.z()), extent, aBlock);
			_compositionalContents[4].origin_extent_tipsDo_(new Jun3dPoint(origin1.x(), origin1.y(), origin2.z()), extent, aBlock);
			_compositionalContents[5].origin_extent_tipsDo_(new Jun3dPoint(origin2.x(), origin1.y(), origin2.z()), extent, aBlock);
			_compositionalContents[6].origin_extent_tipsDo_(new Jun3dPoint(origin1.x(), origin2.y(), origin2.z()), extent, aBlock);
			_compositionalContents[7].origin_extent_tipsDo_(new Jun3dPoint(origin2.x(), origin2.y(), origin2.z()), extent, aBlock);

			return null;
		}
	}

	/**
	 * Render the receiver on the rendering context as polygon.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 */
	public void renderPolygonOn_(JunOpenGLRenderingContext renderingContext) {
		renderingContext.enableLighting();

		if (this.isNode() || this.isEmpty() || this.isFilled()) {
			for (int i = 0; i < PolygonPoints.length; i++) {
				renderingContext.displayPolygon_normalVector_(PolygonPoints[i], PolygonNormals[i]);
			}
		} else {
			throw new SmalltalkException("never reached");
		}
	}

	/**
	 * Render the receiver on the rendering context as vertex.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 */
	public void renderVertexOn_(JunOpenGLRenderingContext renderingContext) {
		renderingContext.disableLighting();

		if (this.isFilled() || this.isEmpty() || this.isNode()) {
			renderingContext.displayPoint_(VertexPoint);
		} else {
			throw new SmalltalkException("never reached");
		}
	}

	/**
	 * Render the receiver on the rendering context as wireframe.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 */
	public void renderWireframeOn_(JunOpenGLRenderingContext renderingContext) {
		renderingContext.enableLighting();

		if (this.isNode() || this.isEmpty() || this.isFilled()) {
			for (int i = 0; i < WireframePoints.length; i++) {
				renderingContext.displayLines_(WireframePoints[i]);
			}
		} else {
			throw new SmalltalkException("never reached");
		}
	}

	/**
	 * Set aJunSolid as contents of the node.
	 * 
	 * @param aJunSolid jp.co.sra.jun.solid.abstracts.JunSolid
	 */
	public void solid_(JunSolidModel aJunSolid) {
		contents = aJunSolid;
	}

	/**
	 * Make a monolithic contents as much as possible.
	 */
	protected void basicBeCompact() {
		if (this.isNode()) {
			JunOctreeNode[] _compositionalContents = (JunOctreeNode[]) contents;

			if (_compositionalContents[0].isFilled()) {
				for (int i = 1; i < _compositionalContents.length; i++) {
					if (_compositionalContents[i].isFilled() == false) {
						return;
					}
				}

				this.beFilled();
			} else if (_compositionalContents[0].isEmpty()) {
				for (int i = 1; i < _compositionalContents.length; i++) {
					if (_compositionalContents[i].isEmpty() == false) {
						return;
					}
				}

				this.beEmpty();
			}
		}
	}

	/**
	 * Set myself as a node of b-reps octree.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	protected void brepsOrigin_extent_block_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, StBlockClosure aBlock) {
		Jun3dPoint corner = aJun3dPoint1.plus_(aJun3dPoint2);
		double[] values = new double[8];
		values[0] = ((Number) aBlock.value_(new Jun3dPoint(aJun3dPoint1.x(), aJun3dPoint1.y(), aJun3dPoint1.z()))).doubleValue();
		values[1] = ((Number) aBlock.value_(new Jun3dPoint(corner.x(), aJun3dPoint1.y(), aJun3dPoint1.z()))).doubleValue();
		values[2] = ((Number) aBlock.value_(new Jun3dPoint(aJun3dPoint1.x(), corner.y(), aJun3dPoint1.z()))).doubleValue();
		values[3] = ((Number) aBlock.value_(new Jun3dPoint(corner.x(), corner.y(), aJun3dPoint1.z()))).doubleValue();
		values[4] = ((Number) aBlock.value_(new Jun3dPoint(aJun3dPoint1.x(), aJun3dPoint1.y(), corner.z()))).doubleValue();
		values[5] = ((Number) aBlock.value_(new Jun3dPoint(corner.x(), aJun3dPoint1.y(), corner.z()))).doubleValue();
		values[6] = ((Number) aBlock.value_(new Jun3dPoint(aJun3dPoint1.x(), corner.y(), corner.z()))).doubleValue();
		values[7] = ((Number) aBlock.value_(new Jun3dPoint(corner.x(), corner.y(), corner.z()))).doubleValue();

		//
		Hashtable pointsAndValues = new Hashtable(8);
		pointsAndValues.put(new Jun3dPoint(0.0d, 0.0d, 0.0d), new Double(values[0]));
		pointsAndValues.put(new Jun3dPoint(1.0d, 0.0d, 0.0d), new Double(values[1]));
		pointsAndValues.put(new Jun3dPoint(0.0d, 1.0d, 0.0d), new Double(values[2]));
		pointsAndValues.put(new Jun3dPoint(1.0d, 1.0d, 0.0d), new Double(values[3]));
		pointsAndValues.put(new Jun3dPoint(0.0d, 0.0d, 1.0d), new Double(values[4]));
		pointsAndValues.put(new Jun3dPoint(1.0d, 0.0d, 1.0d), new Double(values[5]));
		pointsAndValues.put(new Jun3dPoint(0.0d, 1.0d, 1.0d), new Double(values[6]));
		pointsAndValues.put(new Jun3dPoint(1.0d, 1.0d, 1.0d), new Double(values[7]));

		//
		boolean containsPositive = false;
		boolean containsNegative = false;

		for (int i = 0; i < 8; i++) {
			if (values[i] >= 0) {
				containsPositive = true;
			} else {
				containsNegative = true;
			}
		}

		if (containsPositive == false) {
			contents = Boolean.FALSE;
		}

		if (containsNegative == false) {
			contents = Boolean.TRUE;
		}

		this.solid_(this.createBrepsBodyFrom_(pointsAndValues));
	}

	/**
	 * Create a breps body from the specified points and values.
	 * 
	 * @param pointValues java.util.Hashtable
	 * 
	 * @return jp.co.sra.jun.breps.basic.JunBreps
	 */
	protected JunBreps createBrepsBodyFrom_(Hashtable pointValues) {
		JunBody body = JunBody.CubeOrigin_corner_(new Jun3dPoint(0.0d, 0.0d, 0.0d), new Jun3dPoint(1.0d, 1.0d, 1.0d));
		JunVertex[] originalVertices = body.vertexes(); // asSet???
		Hashtable vertexValues = new Hashtable();

		for (int i = 0; i < originalVertices.length; i++) {
			JunVertex vertex = originalVertices[i];
			vertexValues.put(vertex, pointValues.get(vertex.point()));
		}

		JunEdge[] edges = body.edges();

		for (int i = 0; i < edges.length; i++) {
			JunEdge edge = edges[i];
			JunVertex vertex1 = edge.startVertex();
			JunVertex vertex2 = edge.endVertex();
			double value1 = ((Number) vertexValues.get(vertex1)).doubleValue();
			double value2 = ((Number) vertexValues.get(vertex2)).doubleValue();

			if ((value1 < 0 && value2 >= 0) || (value1 >= 0 && value2 < 0)) {
				Jun3dPoint point = vertex2.point().minus_(vertex1.point()).multipliedBy_(value1 / (value1 - value2)).plus_(vertex1.point());
				JunMVE mve = JunMVE.Body_edge_point_(body, edge, point);
				mve.doOperation();
			}
		}

		JunLoop[] loops = body.loops();

		for (int i = 0; i < loops.length; i++) {
			JunLoop loop = loops[i];
			int numVertices = loop.numberOfVertexes();

			if (numVertices == 6) {
				JunVertex[] vertices = loop.vertexes();
				Vector selectedVertices = new Vector();

				for (int j = 0; j < vertices.length; j++) {
					JunVertex vertex = vertices[j];
					boolean containedInOriginalVertices = true;

					for (int k = 0; k < originalVertices.length; k++) {
						if (vertex.equals(originalVertices[k])) {
							containedInOriginalVertices = false;

							break;
						}
					}

					if (containedInOriginalVertices) {
						selectedVertices.addElement(vertex);
					}
				}

				JunVertex firstVertex = (JunVertex) selectedVertices.elementAt(0);
				JunVertex lastVertex = (JunVertex) selectedVertices.elementAt(selectedVertices.size() - 1);
				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, firstVertex, lastVertex);
				mel.doOperation();
			}

			if (numVertices == 8) {
				JunLoop theLoop = loop;
				JunVertex[] vertices = loop.vertexes();
				int size = vertices.length;

				for (int j = 0; j < size; j++) {
					Number aNumber = (Number) vertexValues.get(vertices[j]);

					if (aNumber != null) {
						double value = aNumber.doubleValue();

						if (value < 0) {
							JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, theLoop, vertices[(j - 1 + size) % size], vertices[(j + 1) % size]);
							mel.doOperation();

							if (mel.newLoop().numberOfVertexes() > 3) {
								theLoop = mel.newLoop();
							}
						}
					}
				}
			}
		}

		for (int i = 0; i < originalVertices.length; i++) {
			JunVertex vertex = originalVertices[i];

			if ((vertex.isKilled() == false) && (((Number) vertexValues.get(vertex)).doubleValue() < 0)) {
				edges = vertex.edges();

				for (int j = 0; j < edges.length; j++) {
					JunEdge edge = edges[j];

					if (edge.leftLoop() == edge.rightLoop()) {
						if ((edge.startRightEdge() == edge) && (edge.endLeftEdge() == edge)) {
							JunKEVVL kevvl = JunKEVVL.Body_edge_(body, edge);
							kevvl.doOperation();
						} else {
							JunVertex removalVertex = null;

							if (edge.nextEdgeWithVertex_(edge.startVertex()) == edge) {
								removalVertex = edge.startVertex();
							}

							if (edge.nextEdgeWithVertex_(edge.endVertex()) == edge) {
								removalVertex = edge.endVertex();
							}

							if (removalVertex != null) {
								JunKEV kev = JunKEV.Body_edge_vertex_(body, edge, removalVertex);
								kev.doOperation();
							} else {
								JunKEML keml = JunKEML.Body_edge_(body, edge);
								keml.doOperation();
							}
						}
					} else {
						JunKEL kel = JunKEL.Body_edge_(body, edge);
						kel.doOperation();
					}
				}
			}
		}

		loops = body.loops();

		for (int i = 0; i < loops.length; i++) {
			JunLoop loop = loops[i];

			if (loop.numberOfVertexes() > 3) {
				JunVertex[] vertices = loop.vertexes();
				int index1 = vertices.length - 1;
				int index2 = 1;

				while ((index1 - index2) > 1) {
					JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, vertices[index2], vertices[index1]);
					mel.doOperation();

					if ((vertices.length - 1 - index1) < index2) { // what it should be??
						index1--;
					} else {
						index2++;
					}
				}
			}
		}

		loops = body.loops();

		for (int i = 0; i < loops.length; i++) {
			loops[i].surface_(loops[i].surface());
		}

		return new JunBreps(body);
	}
}
