package jp.co.sra.jun.metaball.abstracts;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;

/**
 * JunDistanceMetaball class
 * 
 *  @author    nishihara
 *  @created   1999/11/05 (by nishihara)
 *  @updated   2006/10/25 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDistanceMetaball.java,v 8.12 2008/02/20 06:32:16 nisinaka Exp $
 */
public abstract class JunDistanceMetaball extends JunMetaball {

	/** number of weight. */
	protected double weight;

	/** number of order. */
	protected int order;

	/**
	 * Create a new instance of JunDistanceMetaball and initialize it.
	 * 
	 * @return jp.co.sra.jun.metaball.abstracts.JunDistanceMetaball
	 * @category Instance creation
	 */
	public JunDistanceMetaball() {
		super();
	}

	/**
	 * Create a new instance of JunDistanceMetaball and initialize it.
	 * 
	 * @param number1 int
	 * @param number2 double
	 * @category Instance creation
	 */
	public JunDistanceMetaball(int number1, double number2) {
		this();
		this.order_(number1);
		this.weight_(number2);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		weight = this.defaultWeight();
		order = this.defaultOrder();
	}

	/**
	 * Answer my current order.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int order() {
		return order;
	}

	/**
	 * Set my new order.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void order_(int aNumber) {
		order = aNumber;
	}

	/**
	 * Answer my current weight.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double weight() {
		return this.weight;
	}

	/**
	 * Set my new weight.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void weight_(double aNumber) {
		weight = aNumber;
	}

	/**
	 * Answer a value at the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @see jp.co.sra.jun.metaball.abstracts.JunMetaball#valueAt_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category accessing
	 */
	public double valueAt_(Jun3dPoint aJun3dPoint) {
		double distance = this.distance_(aJun3dPoint);
		if (distance < this.accuracy()) {
			distance = this.accuracy();
		}
		int sign = (this.weight == 0) ? 0 : (this.weight > 0) ? 1 : -1;
		return Math.pow((Math.abs(this.weight) / distance), this.order) * sign;
	}

	/**
	 * Answer the distance with the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category accessing
	 */
	public abstract double distance_(Jun3dPoint aJun3dPoint);

	/**
	 * Answer the default order.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultOrder() {
		return 2;
	}

	/**
	 * Answer the default weight.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultWeight() {
		return 1.0d;
	}

	/**
	 * Answer my accuracy.
	 * 
	 * @return double
	 * @category private
	 */
	protected double accuracy() {
		return JunGeometry.ACCURACY;
	}

}
