package jp.co.sra.jun.graphics.abstracts;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispNil;
import jp.co.sra.jun.goodies.texteditor.JunTextEditor;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunAbstractVisual class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2003/12/15 (by Ryouichi Matsuda)
 *  @updated   2006/11/22 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAbstractVisual.java,v 8.10 2008/02/20 06:32:15 nisinaka Exp $
 */
public abstract class JunAbstractVisual extends StObject implements StDisplayable {
	/**
	 * Answer the background color.
	 * 
	 * @return java.awt.Color
	 * @category Accessing
	 */
	public static Color BackgroundColor() {
		return Color.white;
	}

	/**
	 * Answer the foreground color.
	 * 
	 * @return java.awt.Color
	 * @category Accessing
	 */
	public static Color ForegroundColor() {
		return Color.black;
	}

	/**
	 * Answer the selection background color.
	 * 
	 * @return java.awt.Color
	 * @category Accessing
	 */
	public static Color SelectionBackgroundColor() {
		return new Color((int) (0.0391894 * 255), (int) (0.141131 * 255), (int) (0.415578 * 255));
	}

	/**
	 * Answer the selection foreground color.
	 * 
	 * @return java.awt.Color
	 * @category Accessing
	 */
	public static Color SelectionForegroundColor() {
		return Color.white;
	}

	/**
	 * Answer a contract string for display.
	 * 
	 * @param aString java.lang.String
	 * @param aWidth int
	 * @param aTextStyle java.awt.FontMetrics
	 * @return java.lang.String
	 * @category Utilities
	 */
	public static String ContractString_compositionWidth_textStyle_(String aString, int aWidth, FontMetrics aTextStyle) {
		String targetString = aString;
		int charCount = targetString.length();
		while (charCount > 3 && aTextStyle.stringWidth(targetString) > aWidth) {
			charCount--;
			targetString = JunStringUtility._ContractString_to_(targetString, charCount);
		}
		return targetString;
	}

	/**
	 * Answer a generated unique number.
	 * 
	 * @return long
	 * @category Utilities
	 */
	public static long GenerateUniqueNumber() {
		long aNumber = JunSystem.GenerateUniqueNumber();
		aNumber = aNumber / 10;
		return aNumber;
	}

	/**
	 * Create a new instance of JunAbstractVisual and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunAbstractVisual() {
		super();
		this.initialize();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		// nothing to do
	}

	/**
	 * Answer the receiver's current background color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color backgroundColor() {
		return BackgroundColor();
	}

	/**
	 * Answer the receiver's current foreground color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color foregroundColor() {
		return ForegroundColor();
	}

	/**
	 * Answer the receiver's current selection background color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color selectionBackgroundColor() {
		return SelectionBackgroundColor();
	}

	/**
	 * Answer the receiver's current selection foreground color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color selectionForegroundColor() {
		return SelectionForegroundColor();
	}

	/**
	 * Answer the receiver's rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		// Answer a Rectangle that represents the receiver's actual bounding rectangle.
		return this.preferredBounds();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return new Rectangle(0, 0, 0, 0);
	}

	/**
	 * Convert to image and answer it.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		int width = this.bounds().width;
		int height = this.bounds().height;
		StImage anImage = new StImage(width, height);
		Graphics aGraphics = null;
		try {
			aGraphics = anImage.image().getGraphics();
			aGraphics.setClip(0, 0, width, height);
			aGraphics.setColor(Color.white);
			aGraphics.fillRect(0, 0, width, height);
			aGraphics.setColor(Color.black);
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public abstract void displayOn_(Graphics aGraphics);

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		// Display the receiver on aGraphicsContext translated by aPoint.
		Graphics gc = aGraphics.create();
		try {
			gc.translate(aPoint.x, aPoint.y);
			displayOn_(gc);
		} finally {
			if (gc != null) {
				gc.dispose();
			}
		}
	}

	/**
	 * Enumerate all items and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object itemsDo_(StBlockClosure aBlock) {
		// no operation
		return this;
	}

	/**
	 * Enumerate all items and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object allItemsDo_(StBlockClosure aBlock) {
		return this.allItemsDo_nestLevel_(aBlock, 0);
	}

	/**
	 * Enumerate all items and evaluate the block with nest level.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param anInteger int
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object allItemsDo_nestLevel_(final StBlockClosure aBlock, final int anInteger) {
		return this.itemsDo_(new StBlockClosure() {
			public Object value_(Object o) {
				JunAbstractVisual item = (JunAbstractVisual) o;
				if (aBlock.numArgs() == 1) {
					aBlock.value_(item);
				}
				if (aBlock.numArgs() == 2) {
					aBlock.value_value_(item, new Integer(anInteger));
				}
				if (item.isComposite()) {
					item.allItemsDo_nestLevel_(aBlock, anInteger + 1);
				}
				return null;
			}
		});
	}

	/**
	 * Enumerate with all items and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object withAllItemsDo_(StBlockClosure aBlock) {
		return this.withAllItemsDo_nestLevel_(aBlock, 0);
	}

	/**
	 * Enumerate with all items and evaluate the block with nest level.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param anInteger int
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object withAllItemsDo_nestLevel_(StBlockClosure aBlock, int anInteger) {
		if (aBlock.numArgs() == 1) {
			aBlock.value_(this);
		}
		if (aBlock.numArgs() == 2) {
			aBlock.value_value_(this, new Integer(anInteger));
		}
		return this.allItemsDo_nestLevel_(aBlock, anInteger + 1);
	}

	/**
	 * Answer true if the receiver is composite object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isComposite() {
		return false;
	}

	/**
	 * Answer true if the receiver is item object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isItem() {
		return false;
	}

	/**
	 * Answer true if the receiver is map object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isMap() {
		return false;
	}

	/**
	 * Answer true if the receiver is network object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNetwork() {
		return false;
	}

	/**
	 * Answer true if the receiver is primitive object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPrimitive() {
		return false;
	}

	/**
	 * Answer true if the receiver is suquence object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSequence() {
		return false;
	}

	/**
	 * Answer true if the receiver is spatial object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSpatial() {
		return false;
	}

	/**
	 * Answer the kind name of the element.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	protected StSymbol kindName() {
		return this._className();
	}

	/**
	 * Answer a new lisp cons cell.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons lispCons() {
		return JunLispCons.Cell();
	}

	/**
	 * Answer a new lisp nil.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispNil lispNil() {
		return JunLispNil.NullList();
	}

	/**
	 * Convert the receiver to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		return list;
	}

	/**
	 * Open a new window with the receiver's lisp list.
	 * 
	 * @category lisp support
	 */
	public void showLispList() {
		this.showLispList_(this.toLispList());
	}

	/**
	 * Open a new window with the specified lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void showLispList_(JunLispList aList) {
		Writer aWriter = null;
		try {
			aWriter = new StringWriter();
			aList.saveOn_(aWriter);
		} catch (IOException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (aWriter != null) {
				try {
					aWriter.flush();
					aWriter.close();
				} catch (IOException e) {
					System.err.println(e.getMessage());
					e.printStackTrace();
				}
			}
		}

		JunTextEditor aModel = new JunTextEditor(aWriter.toString());
		aModel.beEditable(false);
		aModel.openIn_(new Rectangle(100, 100, 340, 180));
	}

	/**
	 * Save the receiver on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category lisp support
	 */
	public void saveOn_(Writer aWriter) throws IOException {
		JunCursors cursor = new JunCursors(new Cursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();
			this.toLispList().saveOn_(aWriter);
		} finally {
			cursor._restore();
		}
	}
}
