package jp.co.sra.jun.goodies.utilities;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel;
import jp.co.sra.jun.goodies.image.streams.JunBmpImageStream;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.multimedia.JunMultiMediaImageModel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunImageUtility class
 * 
 *  @author    nisinaka
 *  @created   2005/10/26 (by nisinaka)
 *  @updated   2006/04/07 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun681 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageUtility.java,v 8.16 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunImageUtility extends JunAbstractObject {
	/**
	 * Create an image from display.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category Creating
	 */
	public static BufferedImage FromDisplay() {
		return FromDisplay_(new Rectangle(new Point(0, 0), Toolkit.getDefaultToolkit().getScreenSize()));
	}

	/**
	 * Create an image from display with the specified bounds.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return java.awt.image.BufferedImage
	 * @category Creating
	 */
	public static BufferedImage FromDisplay_(Rectangle aRectangle) {
		return StImage._OfArea(aRectangle).image();
	}

	/**
	 * Create an image from user.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category Creating
	 */
	public static BufferedImage FromUser() {
		return StImage._FromUser().image();
	}

	/**
	 * Create an image with the specified extent.
	 * 
	 * @param imageExtent java.awt.Dimension
	 * @return java.awt.image.BufferedImage
	 * @category Creating
	 */
	public static BufferedImage ImageExtent_(Dimension imageExtent) {
		return ImageExtent_displayBlock_(imageExtent, new StBlockClosure() {
			public Object value_(Object graphicsContext) {
				return null;
			}
		});
	}

	/**
	 * Create an image with the specified extent.
	 * Apply the display block to the image.
	 * 
	 * @param imageExtent java.awt.Dimension
	 * @param displayBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.awt.image.BufferedImage
	 * @category Creating
	 */
	public static BufferedImage ImageExtent_displayBlock_(Dimension imageExtent, StBlockClosure displayBlock) {
		BufferedImage anImage = null;

		Graphics aGraphics = null;
		try {
			anImage = new BufferedImage(imageExtent.width, imageExtent.height, BufferedImage.TYPE_INT_ARGB);
			aGraphics = anImage.getGraphics();
			aGraphics.setColor(Color.white);
			aGraphics.fillRect(0, 0, imageExtent.width, imageExtent.height);
			aGraphics.setColor(Color.black);
			if (displayBlock != null) {
				displayBlock.value_(aGraphics);
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return anImage;
	}

	/**
	 * Answer the default image quality value.
	 * 
	 * @return float
	 * @category Defaults
	 */
	public static float DefaultImageQuality() {
		return 0.95f;
	}

	/**
	 * Request image file, and answer it.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Reading
	 */
	public static StImage RequestImage() {
		File aFile = RequestImageFile();
		if (aFile == null) {
			return null;
		}
		JunMultiMediaImageModel imageModel = new JunMultiMediaImageModel(aFile);
		if (imageModel == null) {
			return null;
		}
		StImage anImage = imageModel.visual().asImage();
		return anImage;
	}

	/**
	 * Request image file, and answer it.
	 * 
	 * @return java.io.File
	 * @category Reading
	 */
	public static File RequestImageFile() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] {
				new JunFileModel.FileType(JunSystem.$String("<1p> files", null, JunSystem.$String("Image")), JunSystem.DefaultImageExtensionPatterns()),
				new JunFileModel.FileType(JunSystem.$String("<1p> files", null, JunSystem.$String("All")), new String[] { "*" }) };
		String message = JunSystem.$String("Select a <1p> file.", null, JunSystem.$String("Image"));
		File file = JunFileRequesterDialog.Request(message, fileTypes, fileTypes[0]);
		if (file == null) {
			return null;
		}
		return file;
	}

	/**
	 * Open the new application window with the specified image.
	 * 
	 * @param anImage java.awt.Image
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_(Image anImage) {
		return JunImageDisplayModel.Show_(anImage);
	}

	/**
	 * Open the new application window with the specified image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StDisplayable
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_(StDisplayable anImage) {
		return JunImageDisplayModel.Show_(anImage);
	}

	/**
	 * Open the new application window with the specified image at the specified point.
	 * 
	 * @param anImage java.awt.Image
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_at_(Image anImage, Point aPoint) {
		return JunImageDisplayModel.Show_at_(anImage, aPoint);
	}

	/**
	 * Open the new application window with the specified image at the specified point.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StDisplayable
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_at_(StDisplayable anImage, Point aPoint) {
		return JunImageDisplayModel.Show_at_(anImage, aPoint);
	}

	/**
	 * Open the new application window with the specified image in the specified area.
	 * 
	 * @param anImage java.awt.Image
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_in_(Image anImage, Rectangle aRectangle) {
		return JunImageDisplayModel.Show_in_(anImage, aRectangle);
	}

	/**
	 * Open the new application window with the specified image in the specified area.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StDisplayable
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category Showing
	 */
	public static JunImageDisplayModel Show_in_(StDisplayable anImage, Rectangle aRectangle) {
		return JunImageDisplayModel.Show_in_(anImage, aRectangle);
	}

	/**
	 * Request new image file, and answer it.
	 * 
	 * @return java.io.File
	 * @category Writing
	 */
	public static File RequestNewImageFile() {
		String[] values = new String[] { "*.jpg", "*.jpeg", "*.bmp", "*.png", "*.JPG", "*.JPEG", "*.BMP", "*.PNG" };
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, JunSystem.$String("Image")), values) };
		String message = JunSystem.$String("Input a <1p> file.", null, JunSystem.$String("Image"));
		File file = JunFileRequesterDialog.RequestNewFile(message, new File(JunSystem.DefaultBaseName() + ".jpg"), fileTypes, fileTypes[0]);
		if (file == null) {
			return null;
		}
		return file;
	}

	/**
	 * Request new image file, then write the specified image to a file.
	 * 
	 * @param anImage java.awt.Image
	 * @throws java.io.IOException
	 * @category Writing
	 */
	public static File RequestNewImageFileThenWriteImage_(Image anImage) throws IOException {
		return RequestNewImageFileThenWriteImage_(new StImage(anImage));
	}

	/**
	 * Request new image file, then write the specified image to a file.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @throws java.io.IOException
	 * @category Writing
	 */
	public static File RequestNewImageFileThenWriteImage_(StImage anImage) throws IOException {
		File aFile = RequestNewImageFile();
		if (aFile == null) {
			return null;
		}
		return WriteImage_to_(anImage, aFile);
	}

	/**
	 * Write the specified image to a file.
	 * 
	 * @param anImage java.awt.Image
	 * @param aFile java.io.File
	 * @return java.io.File
	 * @throws java.io.IOException
	 * @category Writing
	 */
	public static File WriteImage_to_(Image anImage, File aFile) throws IOException {
		return WriteImage_to_(new StImage(anImage), aFile);
	}

	/**
	 * Write the specified image to a file.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aFile java.io.File
	 * @return java.io.File
	 * @throws java.io.IOException
	 * @category Writing
	 */
	public static File WriteImage_to_(StImage anImage, File aFile) throws IOException {
		File theFile = aFile;
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		JunImageStream aStream = null;
		try {
			cursor._show();
			Class aClass = JunImageStream.ImageStreamClassForFileName_(aFile);
			if (aClass == null) {
				aClass = JunBmpImageStream.class;
				theFile = new File(aFile.getAbsoluteFile() + ".bmp");
			}

			aStream = (JunImageStream) StObject._PerformWith(aClass, "On_", new FileOutputStream(theFile));
			if (aStream.respondsTo_("quality_")) {
				aStream.perform_with_("quality_", new Float(DefaultImageQuality()));
			}
			aStream.nextPutImage_(anImage);
		} catch (NoSuchMethodException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (aStream != null) {
				aStream.flush();
				aStream.close();
			}
			cursor._restore();
		}
		return theFile;
	}
}
