package jp.co.sra.jun.goodies.texteditor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.TextArea;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;
import java.io.IOException;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.WindowSpecLayout;
import jp.co.sra.jun.system.framework.JunAbstractViewPanel;

/**
 * JunTextEditorViewAwt class
 * 
 *  @author    m-asada
 *  @created   2005/05/11 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextEditorViewAwt.java,v 8.11 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTextEditorViewAwt extends JunAbstractViewPanel implements JunTextEditorView, ClipboardOwner {
	protected TextArea textArea;

	/**
	 * Create a new instance of JunTextEditorViewAwt and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTextEditorViewAwt() {
		super();
	}

	/**
	 * Create a new instance of JunTextEditorViewAwt and initialize it.
	 * 
	 * @param newModel jp.co.sra.jun.goodies.texteditor.JunTextEditor
	 * @category Instance creation
	 */
	public JunTextEditorViewAwt(JunTextEditor newModel) {
		super(newModel);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		textArea = null;
	}

	/**
	 * Answer a default model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewPanel#defaultModel()
	 * @category model accessing
	 */
	protected StModel defaultModel() {
		return new JunTextEditor();
	}

	/**
	 * Answer the model as JunTextEditor.
	 * 
	 * @return jp.co.sra.jun.goodies.texteditor.JunTextEditor
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getModel()
	 * @category model accessing
	 */
	public JunTextEditor getModel() {
		return (JunTextEditor) this.model();
	}

	/**
	 * Notifies this object that it is no longer the owner of
	 * the contents of the clipboard.
	 *
	 * @param clipboard the clipboard that is no longer owned
	 * @param contents the contents which this owner had placed on the clipboard
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category events
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#buildComponent()
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.getTextArea().setText(this.getModel().text());
		this.setLayout(new WindowSpecLayout());
		this.add(this.getTextArea(), WindowSpecLayout.Position(0f, 0, 0f, 0, 1f, 0, 1f, 0));
		this.setSize(new Dimension(480, 300));
	}

	/**
	 * Rebuild this component.
	 * 
	 * @param oldModel jp.co.sra.smalltalk.StModel
	 * @param newModel jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewPanel#rebuildComponent(jp.co.sra.smalltalk.StModel, jp.co.sra.smalltalk.StModel)
	 * @category interface opening
	 */
	protected void rebuildComponent(StModel oldModel, StModel newModel) {
		JunTextEditor textEditor = (JunTextEditor) newModel;
		this.getTextArea().setText(textEditor.text());
	}

	/**
	 * Answer the text area.
	 * 
	 * @return java.awt.TextArea
	 * @category interface opening
	 */
	public TextArea getTextArea() {
		if (textArea == null) {
			textArea = new TextArea("", 0, 0, TextArea.SCROLLBARS_VERTICAL_ONLY);
			textArea.setFont(JunTextEditor.DefaultTextFont());
			textArea.setBackground(Color.white);
			textArea.setEditable(this.getModel().isEditable());
			textArea.add(this.popupMenuView());
			textArea.addMouseListener(new MouseAdapter() {
				public void mousePressed(MouseEvent event) {
					if (event.isMetaDown() || event.isPopupTrigger()) {
						_showPopupMenu(event.getX(), event.getY());
					}
				}
			});
			textArea.addTextListener(new TextListener() {
				public void textValueChanged(TextEvent e) {
					String newText = textArea.getText();
					if (newText.equals(JunTextEditorViewAwt.this.getModel().text()) == false) {
						JunTextEditorViewAwt.this.getModel().text_(newText);
					}
				}
			});
		}
		return textArea;
	}

	/**
	 * Answer the text in text area.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getText()
	 * @category text accessing
	 */
	public String getText() {
		return this.getTextArea().getText();
	}

	/**
	 * Answer the selected text in text area.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getSelectedText()
	 * @category text accessing
	 */
	public String getSelectedText() {
		String selectedText = null;
		try {
			selectedText = this.getTextArea().getSelectedText();
		} catch (IllegalArgumentException e) {
		}
		return selectedText;
	}

	/**
	 * Copy selected text to clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#copyText()
	 * @category text accessing
	 */
	public void copyText() {
		String selectedText = this.getSelectedText();
		if (selectedText != null && selectedText.length() > 0) {
			Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			StringSelection contents = new StringSelection(selectedText);
			clipboard.setContents(contents, this);
		}
	}

	/**
	 * Cut selected text to clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#cutText()
	 * @category text accessing
	 */
	public void cutText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		String selectedText = this.getSelectedText();
		if (selectedText != null && selectedText.length() > 0) {
			this.copyText();

			int start = this.getTextArea().getSelectionStart();
			int end = this.getTextArea().getSelectionEnd();
			String aText = this.getText();
			this.getTextArea().setText(aText.substring(0, start) + aText.substring(end));
		}
	}

	/**
	 * Paste selected text from clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#pasteText()
	 * @category text accessing
	 */
	public void pasteText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		String pasteString = null;
		try {
			Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			Transferable contents = clipboard.getContents(this);
			pasteString = (String) contents.getTransferData(DataFlavor.stringFlavor);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} catch (UnsupportedFlavorException e) {
		}

		if (pasteString != null && pasteString.length() > 0) {
			TextArea aTextArea = this.getTextArea();
			String aText = this.getText();
			String selectedText = this.getSelectedText();
			if (selectedText != null && selectedText.length() > 0) {
				int start = aTextArea.getSelectionStart();
				int end = aTextArea.getSelectionEnd();
				aTextArea.setText(aText.substring(0, start) + pasteString + aText.substring(end));
				aTextArea.setCaretPosition(start + pasteString.length() - 1);
			} else {
				int caret = aTextArea.getCaretPosition();
				aTextArea.setText(aText.substring(0, caret) + pasteString + aText.substring(caret));
				aTextArea.setCaretPosition(caret + pasteString.length() - 1);
			}
		}
	}

	/**
	 * Clear text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#clearText()
	 * @category text accessing
	 */
	public void clearText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		String aText = this.getText();
		if (aText != null && aText.length() > 0) {
			this.getTextArea().setText("");
		}
	}

	/**
	 * Undo text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditor.JunTextEditorView#undoText()
	 * @category text accessing
	 */
	public void undoText() {
		
	}

	/**
	 * Redo text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditor.JunTextEditorView#redoText()
	 * @category text accessing
	 */
	public void redoText() {
		
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		StSymbol aspect = evt.getAspect();
		if (aspect == $("text")) {
			if (this.getModel().text().equals(this.getTextArea().getText()) == false) {
				this.getTextArea().setText(this.getModel().text());
				this.getTextArea().setCaretPosition(0);
			}
			return;
		}
		if (aspect == $("editable")) {
			this.getTextArea().setEditable(this.getModel().isEditable());
			return;
		}

		super.update_(evt);
	}
}
