package jp.co.sra.jun.goodies.texteditor;

import java.awt.Font;
import java.awt.event.KeyEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunTextEditor class
 * 
 *  @author    m-asada
 *  @created   2005/05/11 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextEditor.java,v 8.13 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTextEditor extends JunApplicationModel {
	protected StValueHolder textHolder;
	protected boolean editable;

	protected StMenuBar _menuBar;
	protected StPopupMenu _popupMenu;
	protected static Font _DefaultFont = null;

	/**
	 * Answer a default font of JunTextEditor.
	 *
	 * @return java.awt.Font
	 * @category Defaults
	 */
	public static Font DefaultTextFont() {
		if (_DefaultFont == null) {
			boolean isMacOS = System.getProperty("os.name").toLowerCase().matches(".*mac.*os.*x.*");
			String fontName = "Monospaced";
			int fontSize = isMacOS ? 12 : SystemResourceSupport.getFont().getSize();
			_DefaultFont = new Font(fontName, Font.PLAIN, fontSize);
		}
		return _DefaultFont;
	}

	/**
	 * Create a new instance of JunTextEditor and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTextEditor() {
		super();
	}

	/**
	 * Create a new instance of JunTextEditor and initialize it.
	 * 
	 * @param aText java.lang.String
	 * @category Instance creation
	 */
	public JunTextEditor(String aText) {
		super();
		this.text_(aText);
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		textHolder = null;
		editable = true;
		_menuBar = null;
		_popupMenu = null;
	}

	/**
	 * Answer the receiver's text.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String text() {
		return (String) this.textValueHolder().value();
	}

	/**
	 * Set the receiver's text.
	 * 
	 * @param aText java.lang.String
	 * @category accessing
	 */
	public void text_(String aText) {
		if (this.isEditable() == false) {
			return;
		}

		this.textValueHolder().value_(aText);
		this.changed_($("text"));
	}

	/**
	 * Answer true if the receiver's text area is editable, ottherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean isEditable() {
		return editable;
	}

	/**
	 * Set the receiver's editable flag.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void beEditable(boolean aBoolean) {
		if (editable != aBoolean) {
			editable = aBoolean;
			this.changed_($("editable"));
		}
	}

	/**
	 * Answer the receiver's text value holder.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder textValueHolder() {
		if (textHolder == null) {
			textHolder = StValueHolder.With_(new String());
		}
		return textHolder;
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as
	 * the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return this.textValueHolder().compute_(aBlock);
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunTextEditorViewAwt(this);
		} else {
			return new JunTextEditorViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Text Editor");
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		StSymbol[] menuSymbols = new StSymbol[] { $("cutText"), $("pasteText"), $("clearText"), $("undoText"), $("redoText") };
		for (int i = 0; i < menuSymbols.length; i++) {
			StMenuItem menuItem = this._popupMenu().atNameKey_(menuSymbols[i]);
			if (menuItem != null) {
				menuItem.beEnabled(this.isEditable());
			}
		}
	}

	/**
	 * Copy selected text to clipboard.
	 * 
	 * @category menu messages
	 */
	public void copyText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().copyText();
	}

	/**
	 * Cut selected text to clipboard.
	 * 
	 * @category menu messages
	 */
	public void cutText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().cutText();
	}

	/**
	 * Paste text from clipboard.
	 * 
	 * @category menu messages
	 */
	public void pasteText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().pasteText();
	}

	/**
	 * Clear text.
	 * 
	 * @category menu messages
	 */
	public void clearText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().clearText();
	}

	/**
	 * Undo text.
	 * 
	 * @category menu messages
	 */
	public void undoText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().undoText();
	}

	/**
	 * Redo text.
	 * 
	 * @category menu messages
	 */
	public void redoText() {
		if (this.getView() == null) {
			return;
		}
		this.getTextEditorView().redoText();
	}

	/**
	 * Answer the receiver's popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem($String("Copy"), $("copyText"), new MenuPerformer(this, "copyText"), KeyEvent.VK_C, KeyEvent.CTRL_MASK));
			_popupMenu.add(new StMenuItem($String("Cut"), $("cutText"), new MenuPerformer(this, "cutText"), KeyEvent.VK_X, KeyEvent.CTRL_MASK));
			_popupMenu.add(new StMenuItem($String("Paste"), $("pasteText"), new MenuPerformer(this, "pasteText"), KeyEvent.VK_V, KeyEvent.CTRL_MASK));
			_popupMenu.add(new StMenuItem($String("Clear"), $("clearText"), new MenuPerformer(this, "clearText")));
			if (JunApplicationModel.DefaultViewMode != JunApplicationModel.VIEW_AWT) {
				_popupMenu.addSeparator();
				_popupMenu.add(new StMenuItem($String("Undo"), $("undoText"), new MenuPerformer(this, "undoText"), KeyEvent.VK_Z, KeyEvent.CTRL_MASK));
				_popupMenu.add(new StMenuItem($String("Redo"), $("redoText"), new MenuPerformer(this, "redoText"), KeyEvent.VK_Y, KeyEvent.CTRL_MASK));
			}
		}
		return _popupMenu;
	}

	/**
	 * Answer the receiver's  view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category view accessing
	 */
	public StView getView() {
		return this.getTextEditorView();
	}

	/**
	 * Answer the receiver's view as JunTextEditorView.
	 * 
	 * @return jp.co.sra.jun.goodies.texteditor.JunTextEditorView
	 * @category view accessing
	 */
	public JunTextEditorView getTextEditorView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunTextEditorView && ((JunTextEditorView) dependents[i]).model() == this) {
				return (JunTextEditorView) dependents[i];
			}
		}
		return null;
	}
}
