package jp.co.sra.jun.goodies.tables;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeSet;
import java.util.Map.Entry;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunAdjacencyMatrix class
 * 
 *  @author    nisinaka
 *  @created   2006/04/06 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAdjacencyMatrix.java,v 8.14 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunAdjacencyMatrix extends JunAbstractObject {

	protected boolean isUndirectedGraph;
	protected JunAdjacencyTable adjacencyTable;
	protected Object[] cachedNodes;
	protected Object[] cachedArcs;
	protected Object[] cachedRoots;
	protected HashMap cachedNodeToArcs;

	/**
	 * Create a new instance of <code>JunAdjacencyMatrix</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunAdjacencyMatrix() {
		super();
	}

	/**
	 * Create a new instance of <code>JunAdjacencyMatrix</code> and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunAdjacencyMatrix(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		isUndirectedGraph = false;
		adjacencyTable = new JunAdjacencyTable();
		this.flushCachedAll();
	}

	/**
	 * Set the receiver to be a directed graph.
	 * 
	 * @category accessing
	 */
	public void beDirectedGraph() {
		if (this.isDirectedGraph()) {
			return;
		}

		isUndirectedGraph = false;
	}

	/**
	 * Set the receiver to be an undirected graph.
	 * 
	 * @category accessing
	 */
	public void beUndirectedGraph() {
		if (this.isUndirectedGraph()) {
			return;
		}

		isUndirectedGraph = true;

		Object[] arcs = this.arcs();
		for (int i = 0; i < arcs.length; i++) {
			Object first = ((Object[]) arcs[i])[0];
			Object last = ((Object[]) arcs[i])[2];
			JunAttributeTable first2last = (JunAttributeTable) ((Object[]) arcs[i])[1];
			JunAttributeTable last2first = adjacencyTable.at_with_(last, first);
			JunAttributeTable attributes = first2last;
			if (first2last != null && last2first != null) {
				attributes = first2last.merge_(last2first);
			}
			adjacencyTable.at_with_put_(first, last, attributes);
			adjacencyTable.at_with_put_(last, first, attributes);
		}
		this.flushCachedAll();
	}

	/**
	 * Answer the number of nodes.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfNodes() {
		return this.nodes().length;
	}

	/**
	 * Answer the number of arcs.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfArcs() {
		int numberOfArcs = this.arcs().length;
		if (this.isUndirectedGraph()) {
			numberOfArcs /= 2;
		}
		return numberOfArcs;
	}

	/**
	 * Answer the number of roots.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfRoots() {
		return this.roots().length;
	}

	/**
	 * Answer the minimum number of arcs.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int minimumNumberOfArcs() {
		return 0;
	}

	/**
	 * Answer the maximum number of arcs.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int maximumNumberOfArcs() {
		int maximumNumberOfArcs = this.size() * (this.size() - 1);
		if (this.isUndirectedGraph()) {
			maximumNumberOfArcs /= 2;
		}
		return maximumNumberOfArcs;
	}

	/**
	 * Answer my size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return adjacencyTable.size();
	}

	/**
	 * Answer my current nodes.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodes() {
		if (cachedNodes == null) {
			TreeSet keys = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					if (o1 instanceof Comparable) {
						return ((Comparable) o1).compareTo(o2);
					} else if (o2 instanceof Comparable) {
						return ((Comparable) o2).compareTo(o1) * -1;
					} else {
						return o1.toString().compareTo(o2.toString());
					}
				}
			});
			keys.addAll(adjacencyTable.keys());
			cachedNodes = keys.toArray(new Object[keys.size()]);
		}
		return cachedNodes;
	}

	/**
	 * Answer the nodes in breadth first order.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesBreadthFirst() {
		return this.nodesBreadthFirst_(null);
	}

	/**
	 * Answer the nodes in breadth first order.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesBreadthFirst_(Object startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.nodesBreadthFirstDo_startNode_(new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aList.add(node);
				return null;
			}
		}, startNode);
		return aList.toArray();
	}

	/**
	 * Answer the nodes in depth first order.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesDepthFirst() {
		return this.nodesDepthFirst_(null);
	}

	/**
	 * Answer the nodes in depth first order.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesDepthFirst_(Object startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.nodesDepthFirstDo_startNode_(new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aList.add(node);
				return null;
			}
		}, startNode);
		return aList.toArray();
	}

	/**
	 * Answer the nodes in partial order.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesPartialOrder() {
		return this.nodesPartialOrder_(null);
	}

	/**
	 * Answer the nodes in partial order.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesPartialOrder_(Object startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.nodesPartialOrderDo_startNode_(new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aList.add(node);
				return null;
			}
		}, startNode);
		return aList.toArray();
	}

	/**
	 * Answer the nodes in topological sort.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesTopologicalSort() {
		return this.nodesPartialOrder();
	}

	/**
	 * Answer the nodes in topological sort.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] nodesTopologicalSort_(Object startNode) {
		return this.nodesPartialOrder_(startNode);
	}

	/**
	 * Answer my current roots.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] roots() {
		if (cachedRoots == null) {
			if (this.isEmpty()) {
				cachedRoots = new Object[0];
			} else {
				HashSet aSet = new HashSet();
				Object[] arcs = this.arcs();
				for (int i = 0; i < arcs.length; i++) {
					aSet.add(((Object[]) arcs[i])[2]);
				}

				ArrayList aList = new ArrayList();
				Object[] nodes = this.nodes();
				for (int i = 0; i < nodes.length; i++) {
					if (aSet.contains(nodes[i]) == false) {
						aList.add(nodes[i]);
					}
				}

				cachedRoots = aList.toArray();
			}
		}
		return cachedRoots;
	}

	/**
	 * Answer the shortest paths of all nodes.
	 * 
	 * @return java.util.Map
	 * @category accessing nodes
	 */
	public Map shortestPaths() {
		HashMap aMap = new HashMap(this.numberOfNodes());
		Object[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			aMap.put(nodes[i], this.shortestPathFrom_(nodes[i]));
		}
		return aMap;
	}

	/**
	 * Answer the shortest path from the specified node.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.util.Map
	 * @category  accessing nodes
	 */
	public Map shortestPathFrom_(Object startNode) {
		return (Map) this.visitDijkstra_(startNode).at_($("distanceTable"));
	}

	/**
	 * Answer the reachable nodes from the specified node.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public Object[] reachableNodesFrom_(final Object startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				if (node.equals(startNode) == false) {
					aList.add(node);
				}
				return null;
			}
		}, new StBlockClosure());
		return aList.toArray();
	}

	/**
	 * Answer my current arcs.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing arcs
	 */
	public Object[] arcs() {
		if (cachedArcs == null) {
			TreeSet arcs = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					Object[] arc1 = (Object[]) o1;
					Object[] arc2 = (Object[]) o2;
					Object first1 = arc1[0];
					Object first2 = arc2[0];
					Object last1 = arc1[2];
					Object last2 = arc2[2];

					Object key1 = first1;
					Object key2 = first2;
					if (first1.equals(first2)) {
						key1 = last1;
						key2 = last2;
					}

					if (key1 instanceof Comparable) {
						return ((Comparable) key1).compareTo(key2);
					} else if (key2 instanceof Comparable) {
						return ((Comparable) key2).compareTo(key1) * -1;
					} else {
						return key1.toString().compareTo(key2.toString());
					}
				}
			});

			Iterator i = adjacencyTable.entries().iterator();
			while (i.hasNext()) {
				Map.Entry firstEntry = (Map.Entry) i.next();
				Object first = firstEntry.getKey();

				Iterator j = ((JunReferenceTable) firstEntry.getValue()).entries().iterator();
				while (j.hasNext()) {
					Map.Entry lastEntry = (Map.Entry) j.next();
					Object last = lastEntry.getKey();

					JunAttributeTable attributes = (JunAttributeTable) adjacencyTable.at_with_(first, last);
					if (attributes != null) {
						arcs.add(new Object[] { first, attributes, last });
					}
				}
			}

			cachedArcs = arcs.toArray(new Object[arcs.size()]);
		}
		return cachedArcs;
	}

	/**
	 * Answer the arcs of the specified node.
	 * 
	 * @param aNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing arcs
	 */
	public Object[] arcsOf_(Object aNode) {
		if (cachedNodeToArcs == null) {
			cachedNodeToArcs = new HashMap();
		}
		Object[] arcs = (Object[]) cachedNodeToArcs.get(aNode);
		if (arcs == null) {
			TreeSet aTreeSet = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					Object last1 = ((Object[]) o1)[2];
					Object last2 = ((Object[]) o2)[2];

					if (last1 instanceof Comparable) {
						return ((Comparable) last1).compareTo(last2);
					} else if (last2 instanceof Comparable) {
						return ((Comparable) last2).compareTo(last2) * -1;
					} else {
						return last1.toString().compareTo(last2.toString());
					}
				}
			});
			aTreeSet.addAll(Arrays.asList(this.downArcsOfNode_(aNode)));
			arcs = aTreeSet.toArray();
			cachedNodeToArcs.put(aNode, arcs);
		}
		return arcs;
	}

	/**
	 * Answer the down arcs of the node.
	 * 
	 * @param aNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing arcs
	 */
	public Object[] downArcsOfNode_(Object aNode) {
		ArrayList aList = new ArrayList();
		Object[] arcs = this.arcs();
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			if (arc[0].equals(aNode)) {
				aList.add(arc);
			}
		}
		return aList.toArray();
	}

	/**
	 * Answer the up arcs of the node.
	 * 
	 * @param aNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing arcs
	 */
	public Object[] upArcsOfNode_(Object aNode) {
		ArrayList aList = new ArrayList();
		Object[] arcs = this.arcs();
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			if (arc[2].equals(aNode)) {
				aList.add(arc);
			}
		}
		return aList.toArray();
	}

	/**
	 * Answer the attributes of the arc between the nodes.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category accessing arcs
	 */
	public JunAttributeTable between_and_(Object first, Object last) {
		return adjacencyTable.at_with_(first, last);
	}

	/**
	 * Add the node.
	 * 
	 * @param node java.lang.Object
	 * @return jp.co.sra.jun.goodies.tables.JunReferenceTable
	 * @category adding
	 */
	public JunReferenceTable add_(Object node) {
		JunReferenceTable table = (JunReferenceTable) adjacencyTable.add_(node);
		if (table != null) {
			this.flushCachedNodes();
		}
		return table;
	}

	/**
	 * Remove the node.
	 * 
	 * @param node java.lang.Object
	 * @return jp.co.sra.jun.goodies.tables.JunReferenceTable
	 * @category removing
	 */
	public Object remove_(Object node) {
		Object result = adjacencyTable.remove_(node);
		if (result != null) {
			this.flushCachedAll();
			result = node;
		}
		return result;
	}

	/**
	 * Connect the first object with the last object.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @return java.lang.Boolean
	 * @category connecting 
	 */
	public Boolean connect_with_(Object first, Object last) {
		Boolean triplet = adjacencyTable.connect_with_undirected_(first, last, this.isUndirectedGraph());
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedAll();
		}
		return triplet;
	}

	/**
	 * Connect the first object with the last object, and specifies its attribute.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @param attribute java.util.Entry
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_attribute_(Object first, Object last, Entry attribute) {
		return this.connect_with_key_value_(first, last, attribute.getKey(), attribute.getValue());
	}

	/**
	 * Connect the first object with the last object, and specifies its attribute.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_key_value_(Object first, Object last, Object key, Object value) {
		Boolean triplet = this.connect_with_(first, last);
		if (triplet != null) {
			JunAttributeTable attributes = adjacencyTable.at_with_(first, last);
			attributes.at_put_(key, value);
			if (this.isUndirectedGraph()) {
				attributes = adjacencyTable.at_with_(last, first);
				attributes.at_put_(key, value);
			}
		}
		return triplet;
	}

	/**
	 * Connect the first object with the last object, and specifies its cost.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @param value int
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_cost_(Object first, Object last, int value) {
		return this.connect_with_key_value_(first, last, $("cost"), new Integer(Math.max(value, 0)));
	}

	/**
	 * Disconnect the first object with the last object.
	 * 
	 * @param first java.lang.Object
	 * @param last java.lang.Object
	 * @return java.lang.Boolean
	 * @category connecting 
	 */
	public Boolean disconnect_with_(Object first, Object last) {
		Boolean triplet = adjacencyTable.disconnect_with_undirected_(first, last, this.isUndirectedGraph());
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedAll();
		}
		return triplet;
	}

	/**
	 * Visit the nodes in the breadth first order.
	 * 
	 * @param aNode java.lang.Object
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitBreadthFirst_nodeDo_arcDo_(Object aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("nodesBreadthFirst"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("arcsBreadthFirst"), new ArrayList(this.numberOfArcs()));
		visitingAttributes.at_put_($("closedPath"), new ArrayList());
		HashMap visitingTable = new HashMap();
		for (int i = 0; i < this.nodes().length; i++) {
			visitingTable.put(this.nodes()[i], $("unvisited"));
		}
		visitingAttributes.at_put_($("visitingTable"), visitingTable);

		LinkedList graphQueue = new LinkedList();

		if (aNode == null) {
			for (int i = 0; i < this.roots().length; i++) {
				Object fromNode = this.roots()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, fromNode, graphQueue, nodeBlock, arcBlock);
				}
			}
			for (int i = 0; i < this.nodes().length; i++) {
				Object fromNode = this.nodes()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, fromNode, graphQueue, nodeBlock, arcBlock);
				}
			}
		} else {
			if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(aNode) == $("unvisited")) {
				this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, aNode, graphQueue, nodeBlock, arcBlock);
			}
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in the breadth first order.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode java.lang.Object
	 * @param graphQueue java.util.LinkedList
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	protected void visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(JunAttributeTable visitingAttributes, Object fromNode, LinkedList graphQueue, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		ArrayList nodesBreadthFirst = (ArrayList) visitingAttributes.at_($("nodesBreadthFirst"));
		ArrayList arcsBreadthFirst = (ArrayList) visitingAttributes.at_($("arcsBreadthFirst"));
		ArrayList closedPath = (ArrayList) visitingAttributes.at_($("closedPath"));
		HashMap visitingTable = (HashMap) visitingAttributes.at_($("visitingTable"));
		visitingTable.put(fromNode, $("entered"));
		graphQueue.add(new Object[] { fromNode, new Integer(0) });
		while (graphQueue.isEmpty() == false) {
			Object[] anArray = (Object[]) graphQueue.removeFirst();
			Object theNode = anArray[0];
			int indentLevel = ((Number) anArray[1]).intValue();
			visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("nodeSequenceNumber"))).intValue() + 1));
			Object[] argumentArray = new Object[] { theNode, new Integer(indentLevel), visitingAttributes.at_($("nodeSequenceNumber")) };
			nodesBreadthFirst.add(argumentArray);
			nodeBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
			visitingTable.put(theNode, $("visited"));
			Object[] arcs = this.arcsOf_(theNode);
			for (int i = 0; i < arcs.length; i++) {
				Object[] arc = (Object[]) arcs[i];
				Object toNode = arc[2];
				visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("arcSequenceNumber"))).intValue() + 1));
				argumentArray = new Object[] { arc, new Integer(indentLevel), visitingAttributes.at_($("arcSequenceNumber")) };
				arcsBreadthFirst.add(argumentArray);
				arcBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
				if (visitingTable.get(toNode) == $("unvisited")) {
					graphQueue.add(new Object[] { toNode, new Integer(indentLevel + 1) });
					visitingTable.put(toNode, $("entered"));
				} else {
					closedPath.add(toNode);
				}
			}
		}
	}

	/**
	 * Visit the nodes in the depth first order.
	 * 
	 * @param aNode java.lang.Object
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDepthFirst_nodeDo_arcDo_(Object aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("nodesDepthFirst"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("arcsDepthFirst"), new ArrayList(this.numberOfArcs()));
		visitingAttributes.at_put_($("topologicalSort"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("closedPath"), new ArrayList());
		HashMap visitingTable = new HashMap();
		for (int i = 0; i < this.nodes().length; i++) {
			visitingTable.put(this.nodes()[i], $("unvisited"));
		}
		visitingAttributes.at_put_($("visitingTable"), visitingTable);

		if (aNode == null) {
			for (int i = 0; i < this.roots().length; i++) {
				Object fromNode = this.roots()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, fromNode, 0, nodeBlock, arcBlock);
				}
			}
			for (int i = 0; i < this.nodes().length; i++) {
				Object fromNode = this.nodes()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, fromNode, 0, nodeBlock, arcBlock);
				}
			}
		} else {
			if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(aNode) == $("unvisited")) {
				this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, aNode, 0, nodeBlock, arcBlock);
			}
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in the depth first order.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode java.lang.Object
	 * @param indentLevel int
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	protected void visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(JunAttributeTable visitingAttributes, Object fromNode, int indentLevel, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		ArrayList nodesDepthFirst = (ArrayList) visitingAttributes.at_($("nodesDepthFirst"));
		ArrayList arcsDepthFirst = (ArrayList) visitingAttributes.at_($("arcsDepthFirst"));
		ArrayList topologicalSort = (ArrayList) visitingAttributes.at_($("topologicalSort"));
		ArrayList closedPath = (ArrayList) visitingAttributes.at_($("closedPath"));
		HashMap visitingTable = (HashMap) visitingAttributes.at_($("visitingTable"));
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("nodeSequenceNumber"))).intValue() + 1));
		Object[] argumentArray = new Object[] { fromNode, new Integer(indentLevel), visitingAttributes.at_($("nodeSequenceNumber")) };
		nodesDepthFirst.add(argumentArray);
		nodeBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		visitingTable.put(fromNode, $("visited"));
		Object[] arcs = this.arcsOf_(fromNode);
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			Object toNode = arc[2];
			visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("arcSequenceNumber"))).intValue() + 1));
			argumentArray = new Object[] { arc, new Integer(indentLevel), visitingAttributes.at_($("arcSequenceNumber")) };
			arcsDepthFirst.add(argumentArray);
			arcBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
			if (visitingTable.get(toNode) == $("unvisited")) {
				this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, toNode, indentLevel + 1, nodeBlock, arcBlock);
			} else {
				closedPath.add(toNode);
			}
		}
		topologicalSort.add(fromNode);
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param aNode java.lang.Object
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDijkstra_(Object aNode) {
		return this.visitDijkstra_attributeSymbol_(aNode, $("cost"));
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param aNode java.lang.Object
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDijkstra_attributeSymbol_(Object aNode, StSymbol aSymbol) {
		HashMap distanceTable = new HashMap();
		Object[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			distanceTable.put(nodes[i], null);
		}

		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("distanceTable"), distanceTable);
		visitingAttributes.at_put_($("visitedNodes"), new HashSet());
		visitingAttributes.at_put_($("unvisitedNodes"), new HashSet());

		if (aNode != null) {
			this.visitDijkstra_fromNode_attributeSymbol_(visitingAttributes, aNode, aSymbol);
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode java.lang.Object
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category visiting
	 */
	protected void visitDijkstra_fromNode_attributeSymbol_(JunAttributeTable visitingAttributes, Object fromNode, StSymbol aSymbol) {
		Collection visitedNodes = (Collection) visitingAttributes.at_($("visitedNodes"));
		Collection unvisitedNodes = (Collection) visitingAttributes.at_($("unvisitedNodes"));
		Map distanceTable = (Map) visitingAttributes.at_($("distanceTable"));
		distanceTable.put(fromNode, new Integer(0));
		visitedNodes.add(fromNode);
		Object[] reachableNodes = this.reachableNodesFrom_(fromNode);
		for (int i = 0; i < reachableNodes.length; i++) {
			distanceTable.put(reachableNodes[i], null);
			unvisitedNodes.add(reachableNodes[i]);
		}
		Object[] arcs = this.arcsOf_(fromNode);
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			Object toNode = arc[2];
			JunAttributeTable attributes = (JunAttributeTable) arc[1];
			int length = 1;
			if (attributes.at_(aSymbol) != null) {
				length = ((Number) attributes.at_(aSymbol)).intValue();
				length = Math.max(length, 0);
			}
			distanceTable.put(toNode, new Integer(length));
		}
		while (unvisitedNodes.isEmpty() == false) {
			Object aNode = null;
			Number minDistance = null;
			Iterator iterator = unvisitedNodes.iterator();
			while (iterator.hasNext()) {
				Object node = iterator.next();
				Number theDistance = (Number) distanceTable.get(node);
				if (theDistance != null) {
					if (minDistance == null) {
						minDistance = theDistance;
						aNode = node;
					} else {
						if (theDistance.intValue() < minDistance.intValue()) {
							minDistance = theDistance;
							aNode = node;
						}
					}
				}
			}
			if (aNode == null) {
				aNode = unvisitedNodes.iterator().next();
			}
			unvisitedNodes.remove(aNode);
			visitedNodes.add(aNode);

			arcs = this.arcsOf_(aNode);
			for (int i = 0; i < arcs.length; i++) {
				Object[] arc = (Object[]) arcs[i];
				Object toNode = arc[2];
				if (unvisitedNodes.contains(toNode)) {
					JunAttributeTable attributes = (JunAttributeTable) arc[1];
					int length = 1;
					if (attributes.at_(aSymbol) != null) {
						length = ((Number) attributes.at_(aSymbol)).intValue();
						length = Math.max(length, 0);
					}
					Number toDistance = (Number) distanceTable.get(toNode);
					if (toDistance == null) {
						Number aDistance = (Number) distanceTable.get(aNode);
						if (aDistance == null) {
							//
						} else {
							length += aDistance.intValue();
						}
					} else {
						Number aDistance = (Number) distanceTable.get(aNode);
						if (aDistance == null) {
							length = Math.min(toDistance.intValue(), length);
						} else {
							length = Math.min(toDistance.intValue(), length + aDistance.intValue());
						}
					}
					distanceTable.put(toNode, new Integer(length));
				}
			}
		}
	}

	/**
	 * Enumerate nodes in breadth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesBreadthFirstDo_(StBlockClosure aBlock) {
		this.nodesBreadthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in breadth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating nodes
	 */
	public void nodesBreadthFirstDo_startNode_(StBlockClosure aBlock, Object startNode) {
		JunAttributeTable attributes = this.visitBreadthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("nodesBreadthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate nodes in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesDepthFirstDo_(StBlockClosure aBlock) {
		this.nodesDepthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating nodes
	 */
	public void nodesDepthFirstDo_startNode_(StBlockClosure aBlock, Object startNode) {
		JunAttributeTable attributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("nodesDepthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate nodes in partial order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesPartialOrderDo_(StBlockClosure aBlock) {
		this.nodesPartialOrderDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in partial order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating nodes
	 */
	public void nodesPartialOrderDo_startNode_(StBlockClosure aBlock, Object startNode) {
		final HashMap visitingDictionary = new HashMap(this.size());
		JunAttributeTable visitingAttributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				visitingDictionary.put(node, new Object[] { indent, sequence });
				return null;
			}
		}, new StBlockClosure());

		ArrayList aList = (ArrayList) visitingAttributes.at_($("topologicalSort"));
		for (int i = 0; i < aList.size(); i++) {
			Object node = aList.get(i);
			Object[] anArray = (Object[]) visitingDictionary.get(node);
			Object indent = anArray[0];
			Object sequence = anArray[1];
			aBlock.value_value_value_(node, indent, sequence);
		}
	}

	/**
	 * Enumerate nodes in topological sort order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesTopologicalSortDo_(StBlockClosure aBlock) {
		this.nodesPartialOrderDo_(aBlock);
	}

	/**
	 * Enumerate nodes in topological sort order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating nodes
	 */
	public void nodesTopologicalSortDo_startNode_(StBlockClosure aBlock, Object startNode) {
		this.nodesPartialOrderDo_startNode_(aBlock, startNode);
	}

	/**
	 * Enumerate arcs in the breadth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating arcs
	 */
	public void arcsBreadthFirstDo_(StBlockClosure aBlock) {
		this.arcsBreadthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate arcs in the breadth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating arcs
	 */
	public void arcsBreadthFirstDo_startNode_(StBlockClosure aBlock, Object startNode) {
		JunAttributeTable attributes = this.visitBreadthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("arcsBreadthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate arcs in the depth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating arcs
	 */
	public void arcsDepthFirstDo_(StBlockClosure aBlock) {
		this.arcsDepthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate arcs in the depth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode java.lang.Object
	 * @category enumerating arcs
	 */
	public void arcsDepthFirstDo_startNode_(StBlockClosure aBlock, Object startNode) {
		JunAttributeTable attributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("arcsDepthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Flush the cached information.
	 * 
	 * @category flushing
	 */
	protected void flushCachedAll() {
		this.flushCachedNodes();
		this.flushCachedArcs();
	}

	/**
	 * Flush the cached nodes.
	 * 
	 * @category flushing
	 */
	protected void flushCachedNodes() {
		cachedNodes = null;
		cachedRoots = null;
	}

	/**
	 * Flush the cached arcs.
	 * 
	 * @category flushing
	 */
	protected void flushCachedArcs() {
		cachedArcs = null;
		cachedRoots = null;
		cachedNodeToArcs = null;
	}

	/**
	 * Answer true if the receiver is set to be a directed graph, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isDirectedGraph() {
		return !this.isUndirectedGraph();
	}

	/**
	 * Answer true if the receiver is set to be an undirected graph, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isUndirectedGraph() {
		return isUndirectedGraph;
	}

	/**
	 * Answer true if the receiver has no entities, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return adjacencyTable.isEmpty();
	}

	/**
	 * Answer true if the receiver is isomorphic to the specified matrix, otherwise false.
	 * 
	 * @param anotherMatrix jp.co.sra.jun.goodies.tables.JunAdjacencyMatrix
	 * @return boolean
	 * @category testing
	 */
	public boolean isIsomorphicTo_(JunAdjacencyMatrix anotherMatrix) {
		if (anotherMatrix == null) {
			return false;
		}
		if (this.isEmpty() && anotherMatrix.isEmpty()) {
			return true;
		}
		if (this.numberOfNodes() != anotherMatrix.numberOfNodes()) {
			return false;
		}
		if (this.numberOfArcs() != anotherMatrix.numberOfArcs()) {
			return false;
		}
		if (this.numberOfRoots() != anotherMatrix.numberOfRoots()) {
			return false;
		}
		if (this.numberOfArcs() == 0 && anotherMatrix.numberOfArcs() == 0) {
			return true;
		}

		Object[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			Object[] anotherNodes = anotherMatrix.nodes();
			for (int j = 0; j < anotherNodes.length; j++) {
				if (this.downArcsOfNode_(nodes[i]).length == anotherMatrix.downArcsOfNode_(anotherNodes[j]).length && this.upArcsOfNode_(nodes[i]).length == anotherMatrix.upArcsOfNode_(anotherNodes[j]).length) {
					if (this.privateIsomorphicTo_with_and_(anotherMatrix, nodes[i], anotherNodes[j])) {
						return true;
					}
				}
			}
		}

		return false;
	}

	/**
	 * Update the unique numbers.
	 * 
	 * @param uniqueNumberTable java.util.Map
	 * @category updating
	 */
	public void _updateUniqueNumbers(Map uniqueNumberTable) {
		Map.Entry[] adjacencies = adjacencyTable.adjacencies();
		for (int i = 0; i < adjacencies.length; i++) {
			StSymbol oldUniqueNumber = (StSymbol) adjacencies[i].getKey();
			JunReferenceTable referenceTable = (JunReferenceTable) adjacencies[i].getValue();
			StSymbol newUniqueNumber = (StSymbol) uniqueNumberTable.get(oldUniqueNumber);
			adjacencyTable.removeKey_(oldUniqueNumber);
			adjacencyTable.at_put_(newUniqueNumber, referenceTable);

			Map.Entry[] references = referenceTable.references();
			for (int j = 0; j < references.length; j++) {
				oldUniqueNumber = (StSymbol) references[j].getKey();
				JunAttributeTable attributeTable = (JunAttributeTable) references[j].getValue();
				newUniqueNumber = (StSymbol) uniqueNumberTable.get(oldUniqueNumber);
				referenceTable.removeKey_(oldUniqueNumber);
				referenceTable.at_put_(newUniqueNumber, attributeTable);
			}
		}
		this.flushCachedAll();
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());

		aWriter.write(" (nodes");
		Object[] nodes = this.nodes();
		int size = nodes.length;
		if (size <= this.printMax()) {
			for (int i = 0; i < size; i++) {
				this.printNodeOn_(nodes[i], aWriter);
			}
		} else {
			for (int i = 0; i < this.printMax() / 2; i++) {
				this.printNodeOn_(nodes[i], aWriter);
			}
			aWriter.write(" ...");
			for (int i = size - this.printMax() / 2; i < size; i++) {
				this.printNodeOn_(nodes[i], aWriter);
			}
		}
		aWriter.write(')');

		aWriter.write(" (arcs");
		Object[] arcs = this.arcs();
		size = arcs.length;
		if (size <= this.printMax()) {
			for (int i = 0; i < size; i++) {
				this.printArcOn_((Object[]) arcs[i], aWriter);
			}
		} else {
			for (int i = 0; i < this.printMax() / 2; i++) {
				this.printArcOn_((Object[]) arcs[i], aWriter);
			}
			aWriter.write(" ...");
			for (int i = size - this.printMax() / 2; i < size; i++) {
				this.printArcOn_((Object[]) arcs[i], aWriter);
			}
		}
		aWriter.write(')');

		aWriter.write(')');
	}

	/**
	 * Print the string representation of the node on the writer.
	 * 
	 * @param node java.lang.Object
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category printing
	 */
	protected void printNodeOn_(Object node, Writer aWriter) throws IOException {
		aWriter.write(' ');
		aWriter.write(node.toString());
	}

	/**
	 * Print the string representation of the arc on the writer.
	 * 
	 * @param arc java.lang.Object[]
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category printing
	 */
	protected void printArcOn_(Object[] arc, Writer aWriter) throws IOException {
		aWriter.write(" (");
		aWriter.write(arc[0].toString());
		aWriter.write("=>");
		aWriter.write(arc[2].toString());
		aWriter.write(')');
	}

	/**
	 * Answer the maximum number for printing elements.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.goodies.tables.JunAttributeTable#printMax()
	 * @category printing
	 */
	protected int printMax() {
		return 100;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return this._className();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.undirectedGraphFromLispList(aList);
		this.adjacencyTableFromLispList(aList);
	}

	/**
	 * Convert my attributes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons aList = new JunLispCons(this.kindName());
		aList.add_(this.undirectedGraphToLispList());
		aList.add_(this.adjacencyTableToLispList());
		return aList;
	}

	/**
	 * Get my undirectedGraph from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void undirectedGraphFromLispList(JunLispList aList) {
		JunLispList undirectedGraphList = aList._findSublistWithHead($("undirectedGraph"));
		if (undirectedGraphList == null) {
			return;
		}

		Boolean aBoolean = (Boolean) undirectedGraphList.tail();
		if (aBoolean.booleanValue()) {
			this.beUndirectedGraph();
		} else {
			this.beDirectedGraph();
		}
	}

	/**
	 * Convert my undirectedGraph to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons undirectedGraphToLispList() {
		return new JunLispCons($("undirectedGraph"), Boolean.valueOf(isUndirectedGraph));
	}

	/**
	 * Get my adjacency table from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void adjacencyTableFromLispList(JunLispList aList) {
		JunLispList adjacencyTableList = aList._findSublistWithHead($("adjacencyTable"));
		if (adjacencyTableList == null) {
			return;
		}

		adjacencyTable = new JunAdjacencyTable((JunLispList) adjacencyTableList.tail());
	}

	/**
	 * Convert my adjacency table to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons adjacencyTableToLispList() {
		return new JunLispCons($("adjacencyTable"), adjacencyTable.toLispList());
	}

	/**
	 * Private method to check the two matrixes are isomorphic.
	 * 
	 * @param anotherMatrix jp.co.sra.jun.goodies.tables.JunAdjacencyMatrix
	 * @param aNode java.lang.Object
	 * @param anotherNode java.lang.Object
	 * @return boolean
	 * @category private
	 */
	protected boolean privateIsomorphicTo_with_and_(JunAdjacencyMatrix anotherMatrix, Object aNode, Object anotherNode) {
		final Collection aCollection = new ArrayList(this.numberOfNodes());
		JunAttributeTable aTable = this.visitDepthFirst_nodeDo_arcDo_(aNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aCollection.add(indent);
				return null;
			}
		}, new StBlockClosure());
		final Collection anotherCollection = new ArrayList(anotherMatrix.numberOfNodes());
		JunAttributeTable anotherTable = anotherMatrix.visitDepthFirst_nodeDo_arcDo_(anotherNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				anotherCollection.add(indent);
				return null;
			}
		}, new StBlockClosure());
		if (aCollection.equals(anotherCollection) == false) {
			return false;
		}

		aCollection.clear();
		Iterator iterator = ((Map) aTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("unvisited")) {
				aCollection.add(entry);
			}
		}
		anotherCollection.clear();
		iterator = ((Map) anotherTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("unvisited")) {
				anotherCollection.add(entry);
			}
		}
		if (aCollection.size() != anotherCollection.size()) {
			return false;
		}

		aCollection.clear();
		iterator = ((Map) aTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("visited")) {
				aCollection.add(entry);
			}
		}
		anotherCollection.clear();
		iterator = ((Map) anotherTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("visited")) {
				anotherCollection.add(entry);
			}
		}

		JunAdjacencyMatrix aCopiedMatrix = (JunAdjacencyMatrix) this.copy();
		JunAdjacencyMatrix anotherCopiedMatrix = (JunAdjacencyMatrix) anotherMatrix.copy();
		iterator = aCollection.iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			aCopiedMatrix.remove_(entry.getKey());
		}
		iterator = anotherCollection.iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			anotherCopiedMatrix.remove_(entry.getKey());
		}

		return aCopiedMatrix.isIsomorphicTo_(anotherCopiedMatrix);
	}

}
